package com.usercentrics.tcf.core

import com.usercentrics.sdk.errors.UsercentricsException
import com.usercentrics.sdk.v2.tcf.facade.TCFFacade
import com.usercentrics.tcf.core.errors.GVLError
import com.usercentrics.tcf.core.model.gvl.*

/**
 * class with utilities for managing the global vendor list.  Will use JSON to
 * fetch the vendor list from specified url and will serialize it into this
 * object and provide accessors.  Provides ways to group vendors on the list by
 * purpose and feature.
 */
@Suppress("SpellCheckingInspection")
internal open class GVL(
    private val tcfFacade: TCFFacade,
    lastUpdated: String? = null,
    gvlSpecificationVersion: Int? = null,
    _vendorListVersion: Int? = null,
    _tcfPolicyVersion: Int? = null,
    _vendors: Map<String, Vendor>? = null,
    _features: Map<String, Feature>? = null,
    _purposes: Map<String, Purpose>? = null,
    _dataCategories: Map<String, DataCategory>? = null,
    _specialFeatures: Map<String, Feature>? = null,
    _specialPurposes: Map<String, Purpose>? = null,
    _stacks: Map<String, Stack>? = null,
) {

    companion object {
        private const val DEFAULT_LANGUAGE: String = "EN"
    }

    var lastUpdated: String? = lastUpdated
        private set
    var gvlSpecificationVersion: Int? = gvlSpecificationVersion
        private set
    var vendors: Map<String, Vendor>? = _vendors
        private set
    var vendorIds: List<Int>? = null
        private set
    var vendorListVersion: Int? = _vendorListVersion
        private set
    var tcfPolicyVersion: Int? = _tcfPolicyVersion
        private set
    var features: Map<String, Feature>? = _features
        private set
    var purposes: Map<String, Purpose>? = _purposes
        private set

    var dataCategories: Map<String, DataCategory>? = _dataCategories
        private set

    var specialFeatures: Map<String, Feature>? = _specialFeatures
        private set
    var specialPurposes: Map<String, Purpose>? = _specialPurposes
        private set
    var stacks: Map<String, Stack>? = _stacks
        private set

    /**
     * @param {boolean} internal reference of when the GVL is ready to be used
     */
    private var isReady = false

    /**
     * @param {IntMap<Vendor>} a collection of [[Vendor]]. Used as a backup if a whitelist is sets
     */
    private var fullVendorList = mapOf<String, Vendor>()
    private var initialLanguage = DEFAULT_LANGUAGE

    @Suppress("FoldInitializerAndIfToElvis")
    suspend fun initialize(): Result<Unit> {
        val vendorListResult = tcfFacade.getVendorList()

        val vendorList = vendorListResult.getOrNull()
        if (vendorList == null) {
            return Result.failure(vendorListResult.exceptionOrNull() ?: UsercentricsException(message = "Error when initializing TCF #111"))
        }

        this.populate(vendorList)
        return Result.success(Unit)
    }

    /**
     * changeLanguage - retrieves the purpose language translation and sets the
     * internal language variable
     *
     * @param {string} language - ISO 639-1 langauge code to change language to
     * @return {Promise<void | GVLError>} - returns the `readyPromise` and
     * resolves when this GVL is populated with the data from the language file.
     */
    suspend fun changeLanguage(language: String): Result<Unit> {
        val langUpper = language.uppercase()
        if (langUpper == initialLanguage) {
            return Result.success(Unit)
        }

        return try {
            val declarations = tcfFacade.getDeclarations(language)

            val exceptionOnDeclarations = declarations.exceptionOrNull()
            if (exceptionOnDeclarations != null) {
                throw exceptionOnDeclarations
            }

            initialLanguage = language
            populate(declarations.getOrNull()!!)

            Result.success(Unit)
        } catch (ex: Exception) {
            Result.failure(GVLError("Unable to fetch language ($language) declarations: ${ex.message}", ex))
        }
    }

    fun getLanguage(): String = initialLanguage

    private fun populate(declarations: Declarations) {
        purposes = declarations.purposes
        specialPurposes = declarations.specialPurposes
        features = declarations.features
        specialFeatures = declarations.specialFeatures
        stacks = declarations.stacks
        dataCategories = declarations.dataCategories
    }

    private fun populate(vendorList: VendorList) {
        purposes = vendorList.purposes
        specialPurposes = vendorList.specialPurposes
        features = vendorList.features
        specialFeatures = vendorList.specialFeatures
        stacks = vendorList.stacks
        dataCategories = vendorList.dataCategories

        gvlSpecificationVersion = vendorList.gvlSpecificationVersion
        tcfPolicyVersion = vendorList.tcfPolicyVersion
        vendorListVersion = vendorList.vendorListVersion
        lastUpdated = vendorList.lastUpdated
        vendors = vendorList.vendors

        this.fullVendorList = vendorList.vendors!!
        mapVendors(null)
        this.isReady = true
    }

    private fun mapVendors(vendorIds: List<Int>?) {
        var internalVendorIds = vendorIds

        if (vendorIds == null) {
            internalVendorIds = fullVendorList.keys.map { vId ->
                +vId.toInt()
            }
        }

        val tempVendors = mutableMapOf<String, Vendor>()
        val vendors = vendors
        internalVendorIds?.forEach { vendorId ->
            val vendor: Vendor? = vendors?.get(vendorId.toString())

            if (vendor != null && vendor.deletedDate == null) {
                tempVendors[vendorId.toString()] = vendor
            }
        }

        this.vendors = tempVendors
        this.vendorIds = internalVendorIds?.sorted()
    }

    /**
     * narrowVendorsTo - narrows vendors represented in this GVL to the list of ids passed in
     *
     * @param {number[]} vendorIds - list of ids to narrow this GVL to
     * @return {void}
     */
    fun narrowVendorsTo(vendorIds: List<Int>) {
        this.mapVendors(vendorIds = vendorIds)
    }

    /**
     * isReady - Whether or not this instance is ready to be used.  This will be
     * immediately and synchronously true if a vnedorlist object is passed into
     * the constructor or once the JSON vendorllist is retrieved.
     *
     * @return {boolean} whether or not the instance is ready to be interacted
     * with and all the data is populated
     */
    fun getIsReady(): Boolean {
        return this.isReady
    }

}
