package com.usercentrics.tcf.core

import com.usercentrics.sdk.core.time.DateTime
import com.usercentrics.sdk.services.tcf.Constants
import com.usercentrics.tcf.core.errors.TCModelError
import com.usercentrics.tcf.core.model.PurposeRestrictionVector
import com.usercentrics.tcf.core.model.Vector
import com.usercentrics.tcf.core.model.gvl.Purpose
import kotlinx.serialization.Contextual

//type StringOrNumber = number | string;
//export type TCModelPropType = number | Date | string | boolean | Vector | PurposeRestrictionVector;

internal sealed class TCModelPropType {
    class Int(val value: kotlin.Int) : TCModelPropType()
    class Date(val value: Long?) : TCModelPropType()
    class String(val value: kotlin.String) : TCModelPropType()
    class Vector(val value: com.usercentrics.tcf.core.model.Vector) : TCModelPropType()
    class PurposeRestrictionVector(val value: com.usercentrics.tcf.core.model.PurposeRestrictionVector) : TCModelPropType()

    class StringOrNumber(val value: com.usercentrics.tcf.core.StringOrNumber) : TCModelPropType()
    class Boolean(val value: kotlin.Boolean) : TCModelPropType()
}

internal sealed class StringOrNumber {
    class Int(val value: kotlin.Int) : StringOrNumber()
    class String(val value: kotlin.String) : StringOrNumber()
}

const val publisherCountryCodeDefault = "AA"

@Suppress("PrivatePropertyName")
internal data class TCModel(
    private val _gvl_: GVL
) {
    /**
     * Set of available consent languages published by the IAB
     */

    var gvl_: GVL? = _gvl_
        private set

    private var isServiceSpecific_ = false
    private var supportOOB_ = true
    private var useNonStandardStacks_ = false
    private var purposeOneTreatment_ = false
    private var publisherCountryCode_ = publisherCountryCodeDefault
    private var version_ = 2
    private var consentScreen_ = StringOrNumber.Int(0)
    private var policyVersion_ = StringOrNumber.Int(Constants.TCF_POLICY_VERSION)
    private var consentLanguage_ = "EN"
    private var cmpId_ = StringOrNumber.Int(0)
    private var cmpVersion_ = StringOrNumber.Int(0)
    private var vendorListVersion_ = StringOrNumber.Int(0)
    private var numCustomPurposes_ = 0

    @Contextual
    var created: Long? = null

    @Contextual
    var lastUpdated: Long? = null

    /**
     * The TCF designates certain Features as special, that is, a CMP must afford
     * the user a means to opt in to their use. These Special Features are
     * published and numbered in the GVL separately from normal Features.
     * Provides for up to 12 special features.
     */
    var specialFeatureOptins: Vector = Vector()

    /**
     * Renamed from `PurposesAllowed` in TCF v1.1
     * The user’s consent value for each Purpose established on the legal basis
     * of consent. Purposes are published in the Global Vendor List (see. [[GVL]]).
     */
    var purposeConsents = Vector()

    /**
     * The user’s permission for each Purpose established on the legal basis of
     * legitimate interest. If the user has exercised right-to-object for a
     * purpose.
     */
    var purposeLegitimateInterests = Vector()

    /**
     * The user’s consent value for each Purpose established on the legal basis
     * of consent, for the publisher.  Purposes are published in the Global
     * Vendor List.
     */
    var publisherConsents = Vector()

    /**
     * The user’s permission for each Purpose established on the legal basis of
     * legitimate interest.  If the user has exercised right-to-object for a
     * purpose.
     */
    var publisherLegitimateInterests = Vector()

    /**
     * The user’s consent value for each Purpose established on the legal basis
     * of consent, for the publisher.  Purposes are published in the Global
     * Vendor List.
     */
    var publisherCustomConsents = Vector()

    /**
     * The user’s permission for each Purpose established on the legal basis of
     * legitimate interest.  If the user has exercised right-to-object for a
     * purpose that is established in the publisher's custom purposes.
     */
    var publisherCustomLegitimateInterests = Vector()

    /**
     * set by a publisher if they wish to collect consent and LI Transparency for
     * purposes outside of the TCF
     */
    val customPurposes = mutableMapOf<String, Purpose>()

    /**
     * Each [[Vendor]] is keyed by id. Their consent value is true if it is in
     * the Vector
     */
    var vendorConsents = Vector()

    /**
     * Each [[Vendor]] is keyed by id. Whether their Legitimate Interests
     * Disclosures have been established is stored as boolean.
     * see: [[Vector]]
     */
    var vendorLegitimateInterests = Vector()

    /**
     * The value included for disclosed vendors signals which vendors have been
     * disclosed to the user in the interface surfaced by the CMP. This section
     * content is required when writing a TC string to the global (consensu)
     * scope. When a CMP has read from and is updating a TC string from the
     * global consensu.org storage, the CMP MUST retain the existing disclosure
     * information and only add information for vendors that it has disclosed
     * that had not been disclosed by other CMPs in prior interactions with this
     * device/user agent.
     */
    var vendorsDisclosed = Vector()

    /**
     * Signals which vendors the publisher permits to use OOB legal bases.
     */
    var vendorsAllowed = Vector()

    // TODO ATOMIC
    var publisherRestrictions = PurposeRestrictionVector()

    /**
     * Constructs the TCModel. Passing a [[GVL]] is optional when constructing
     * as this TCModel may be constructed from decoding an existing encoded
     * TCString.
     *
     * @param {GVL} [gvl]
     */
    init {
        this.setCreatedAndUpdatedFields()
    }

    /**
     * @return {GVL} the gvl instance set on this TCModel instance
     */
    fun getGvl(): GVL? = this.gvl_

    /**
     * @param {number} integer - A unique ID will be assigned to each Consent
     * Manager Provider (CMP) from the iab.
     *
     * @throws {TCModelError} if the value is not an integer greater than 1 as those are not valid.
     */
    fun setCmpId(integer: StringOrNumber) {
        if (integer is StringOrNumber.Int && integer.value > 1) {
            this.cmpId_ = integer
        } else {
            throw TCModelError("cmpId", integer)
        }
    }

    /**
     * Each change to an operating CMP should receive a
     * new version number, for logging proof of consent. CmpVersion defined by
     * each CMP.
     *
     * @param {number} integer
     *
     * @throws {TCModelError} if the value is not an integer greater than 1 as those are not valid.
     */
    fun setCmpVersion(integer: StringOrNumber) {
        if (integer is StringOrNumber.Int && integer.value > -1) {
            this.cmpVersion_ = integer
        } else {
            throw TCModelError("cmpVersion", integer)
        }
    }

    /**
     * The screen number is CMP and CmpVersion
     * specific, and is for logging proof of consent.(For example, a CMP could
     * keep records so that a publisher can request information about the context
     * in which consent was gathered.)
     *
     * @param {number} integer
     *
     * @throws {TCModelError} if the value is not an integer greater than 0 as those are not valid.
     */
    fun setConsentScreen(integer: StringOrNumber) {
        if (integer is StringOrNumber.Int && integer.value > -1) {
            this.consentScreen_ = integer
        } else {
            throw TCModelError("consentScreen", integer)
        }
    }

    /**
     * @param {string} lang - [two-letter ISO 639-1 language
   * code](http://www.loc.gov/standards/iso639-2/php/code_list.php) in which
     * the CMP UI was presented
     *
     * @throws {TCModelError} if the value is not a length-2 string of alpha characters
     */
    fun setConsentLanguage(lang: String) {
        this.consentLanguage_ = lang
    }

    /**
     * @param {string} countryCode - [two-letter ISO 3166-1 alpha-2 country
   * code](https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2) of the publisher,
     * determined by the CMP-settings of the publisher.
     *
     * @throws {TCModelError} if the value is not a length-2 string of alpha characters
     */
    fun setPublisherCountryCode(countryCode: String) {
        val regexText = "^([A-z]){2}\$".toRegex()
        if (regexText.matches(countryCode)) {
            this.publisherCountryCode_ = countryCode.uppercase()
        } else {
            throw TCModelError("publisherCountryCode", countryCode)
        }
    }

    /**
     * Version of the GVL used to create this TCModel. Global
     * Vendor List versions will be released periodically.
     *
     * @param {number} integer
     *
     * @throws {TCModelError} if the value is not an integer greater than 0 as those are not valid.
     */
    fun setVendorListVersion(integer: StringOrNumber) {

        /**
         * first coerce to a number via leading '+' then take the integer value by
         * bitshifting to the right.  This works on all types in JavaScript and if
         * it's not valid then value will be 0.
         */
        // integer = +integer>>0
        // On Kotlin we don't need because we used a Sealed Class

        if (integer !is StringOrNumber.Int) {
            throw TCModelError("vendorListVersion", integer)
        }

        if (integer.value < 0) {
            throw TCModelError("vendorListVersion", integer)
        }

        if (integer.value >= 0) {
            this.vendorListVersion_ = integer
        }
    }

    /**
     * From the corresponding field in the GVL that was
     * used for obtaining consent. A new policy version invalidates existing
     * strings and requires CMPs to re-establish transparency and consent from
     * users.
     *
     * If a TCF policy version number is different from the one from the latest
     * GVL, the CMP must re-establish transparency and consent.
     *
     * @param {number} num - You do not need to set this.  This comes
     * directly from the [[GVL]].
     *
     */
    fun setPolicyVersion(num: StringOrNumber) {

        var internalPolicyVersion: Int = -1

        if (num is StringOrNumber.String) {

            try {
                internalPolicyVersion = num.value.toInt()

            } catch (nfe: NumberFormatException) {
                throw TCModelError("policyVersion", num)
            }
        }

        if (num is StringOrNumber.Int) {
            internalPolicyVersion = num.value
        }

        if (internalPolicyVersion < 0) {
            throw TCModelError("policyVersion", num)

        } else {
            this.policyVersion_ = StringOrNumber.Int(internalPolicyVersion)
        }
    }

    fun setVersion(num: StringOrNumber) {
        if (num is StringOrNumber.String) {
            try {
                this.version_ = num.value.toInt()
            } catch (nfe: NumberFormatException) {
                throw TCModelError("version", num)
            }
        }

        if (num is StringOrNumber.Int) {
            this.version_ = num.value
        }
    }

    fun getVersion(): Int {
        return this.version_
    }

    fun getPolicyVersion(): Int {
        return this.policyVersion_.value
    }

    /**
     * Whether the signals encoded in this TC String were from site-specific
     * storage `true` versus ‘global’ consensu.org shared storage `false`. A
     * string intended to be stored in global/shared scope but the CMP is unable
     * to store due to a user agent not accepting third-party cookies would be
     * considered site-specific `true`.
     *
     * @param {boolean} bool - value to set. Some changes to other fields in this
     * model will automatically change this value like adding publisher
     * restrictions.
     */
    fun setIsServiceSpecific(bool: Boolean) {
        this.isServiceSpecific_ = bool
    }

    fun getIsServiceSpecific(): Boolean {
        return this.isServiceSpecific_
    }

    /**
     * Non-standard stacks means that a CMP is using publisher-customized stack
     * descriptions. Stacks (in terms of purposes in a stack) are pre-set by the
     * IAB. As are titles. Descriptions are pre-set, but publishers can customize
     * them. If they do, they need to set this bit to indicate that they've
     * customized descriptions.
     *
     * @param {boolean} bool - value to set
     */
    fun setUseNonStandardStacks(bool: Boolean) {
        this.useNonStandardStacks_ = bool
    }

    fun getSupportOOB(): Boolean {
        return this.supportOOB_
    }

    fun setSupportOOB(supportOOB: Boolean) {
        this.supportOOB_ = supportOOB
    }

    /**
     * `false` There is no special Purpose 1 status.
     * Purpose 1 was disclosed normally (consent) as expected by Policy.  `true`
     * Purpose 1 not disclosed at all. CMPs use PublisherCC to indicate the
     * publisher’s country of establishment to help Vendors determine whether the
     * vendor requires Purpose 1 consent. In global scope TC strings, this field
     * must always have a value of `false`. When a CMP encounters a global scope
     * string with `purposeOneTreatment=true` then that string should be
     * considered invalid and the CMP must re-establish transparency and consent.
     *
     * @param {boolean} bool
     */
    fun setPurposeOneTreatment(bool: Boolean) {
        this.purposeOneTreatment_ = bool
    }

    /**
     * unsetAllVendorConsents - unsets all vendors on the GVL Consent (false)
     *
     * @return {void}
     */
    fun unsetAllVendorConsents() {
        this.vendorConsents.clear()
    }

    /**
     * unsetAllVendorLegitimateInterests - unsets all vendors on the GVL LegitimateInterests (false)
     *
     * @return {void}
     */
    fun unsetAllVendorLegitimateInterests() {
        this.vendorLegitimateInterests.clear()
    }

    /**
     * unsetAllPurposeLegitimateInterests - unsets all purposes on the GVL LI Transparency (false)
     *
     * @return {void}
     */
    fun unsetAllPurposeLegitimateInterests() {
        this.purposeLegitimateInterests.clear()
    }

    fun getNumCustomPurposes(): StringOrNumber {
        var len = this.numCustomPurposes_
        if (this.customPurposes.isNotEmpty()) {
            /**
             * Keys are not guaranteed to be in order and likewise there is no
             * requirement that the customPurposes be non-sparse.  So we have to sort
             * and take the highest value.  Even if the set only contains 3 purposes
             * but goes to ID 6 we need to set the number to 6 for the encoding to
             * work properly since it's positional.
             */
            val purposeIds: List<String> = this.customPurposes.keys.sortedBy { it.toInt() }.toMutableList()
            len = purposeIds.last().toInt()
        }
        return StringOrNumber.Int(len)
    }

    fun setNumCustomPurposes(num: StringOrNumber) {
        var internalNum: Int = -1
        if (num is StringOrNumber.String) {
            try {
                internalNum = num.value.toInt()
            } catch (nfe: NumberFormatException) {
                throw TCModelError("numCustomPurposes", num)
            }
        }

        if (num is StringOrNumber.Int) {
            internalNum = num.value
        }

        if (internalNum < 0) {
            throw TCModelError("numCustomPurposes", num)
        } else {
            this.numCustomPurposes_ = internalNum
        }
    }

    fun setCreatedAndUpdatedFields() {
        val now = DateTime().atMidnight().timestamp()
        this.lastUpdated = now
        this.created = now
    }

    fun getFieldByName(name: String): TCModelPropType {
        return when (name) {
            "version" -> TCModelPropType.Int(this.version_)
            "created" -> TCModelPropType.Date(this.created)
            "lastUpdated" -> TCModelPropType.Date(this.lastUpdated)
            "cmpId" -> TCModelPropType.StringOrNumber(this.cmpId_)
            "cmpVersion" -> TCModelPropType.StringOrNumber(this.cmpVersion_)
            "consentScreen" -> TCModelPropType.StringOrNumber(this.consentScreen_)
            "consentLanguage" -> TCModelPropType.String(this.consentLanguage_)
            "vendorListVersion" -> TCModelPropType.StringOrNumber(this.vendorListVersion_)
            "purposeConsents" -> TCModelPropType.Vector(this.purposeConsents)
            "vendorConsents" -> TCModelPropType.Vector(this.vendorConsents)

            "policyVersion" -> TCModelPropType.StringOrNumber(this.policyVersion_)
            "isServiceSpecific" -> TCModelPropType.Boolean(this.isServiceSpecific_)
            "useNonStandardStacks" -> TCModelPropType.Boolean(this.useNonStandardStacks_)
            "specialFeatureOptins" -> TCModelPropType.Vector(this.specialFeatureOptins)

            "purposeLegitimateInterests" -> TCModelPropType.Vector(this.purposeLegitimateInterests)
            "purposeOneTreatment" -> TCModelPropType.Boolean(this.purposeOneTreatment_)
            "publisherCountryCode" -> TCModelPropType.String(this.publisherCountryCode_)

            "vendorLegitimateInterests" -> TCModelPropType.Vector(this.vendorLegitimateInterests)
            "publisherRestrictions" -> TCModelPropType.PurposeRestrictionVector(this.publisherRestrictions)

            "publisherConsents" -> TCModelPropType.Vector(this.publisherConsents)
            "publisherLegitimateInterests" -> TCModelPropType.Vector(this.publisherLegitimateInterests)
            "numCustomPurposes" -> TCModelPropType.Int(this.numCustomPurposes_)
            "publisherCustomConsents" -> TCModelPropType.Vector(this.publisherCustomConsents)
            "publisherCustomLegitimateInterests" -> TCModelPropType.Vector(this.publisherCustomLegitimateInterests)

            "vendorsAllowed" -> TCModelPropType.Vector(this.vendorsAllowed)
            "vendorsDisclosed" -> TCModelPropType.Vector(this.vendorsDisclosed)
            else -> {
                throw TCModelError("Unable to get field from TCModel", name)
            }
        }
    }

    fun clearConsents() {
        this.purposeConsents.clear()
        this.purposeLegitimateInterests.clear()

        this.vendorConsents.clear()
        this.vendorLegitimateInterests.clear()

        this.specialFeatureOptins.clear()
    }
}
