package com.usercentrics.tcf.core.encoder

import com.usercentrics.tcf.core.StringOrNumber
import com.usercentrics.tcf.core.TCModel
import com.usercentrics.tcf.core.TCModelPropType
import com.usercentrics.tcf.core.encoder.field.*
import com.usercentrics.tcf.core.encoder.sequence.FieldSequence
import com.usercentrics.tcf.core.encoder.sequence.SequenceVersionMapType
import com.usercentrics.tcf.core.errors.DecodingError
import com.usercentrics.tcf.core.errors.EncodingError
import com.usercentrics.tcf.core.model.Segment
import com.usercentrics.tcf.core.model.SegmentIDs

internal class SegmentEncoder {

    companion object {

        val fieldSequence: FieldSequence = FieldSequence()

        internal fun encode(tcModel: TCModel, segment: Segment): String {
            val version = tcModel.getVersion()
            if (version != 2) {
                throw EncodingError("Unsupported TCF version: $version")
            }

            val sequence: List<String>?
            try {
                sequence = (fieldSequence.two as SequenceVersionMapType.SVMItemMap).map[segment]
            } catch (_: Throwable) {
                throw EncodingError("Unable to encode version $version, segment: $segment")
            }

            var bitField = ""

            /**
             * If this is anything other than the core segment we have a "segment id"
             * to append to the front of the string
             */
            if (segment != Segment.CORE) {
                val segmentKeyId = SegmentIDs.KEY_TO_ID[segment]
                if (segmentKeyId != null) {
                    bitField = IntEncoder.encode(
                        StringOrNumber.Int(segmentKeyId), BitLength.getByName(
                            "segmentType"
                        )?.integer!!
                    )
                } else {
                    throw EncodingError("Unable to find segment key for $segment")
                }
            }

            sequence?.forEach { key ->
                val value: TCModelPropType = tcModel.getFieldByName(key)

                val bitLength = BitLength.getByName(key)
                var numBits = 0

                if (bitLength == null) {
                    if (isPublisherCustom(key)) {
                        /**
                         * publisherCustom[Consents | LegitimateInterests] are an edge case
                         * because they are of variable length. The length is defined in a
                         * separate field named numCustomPurposes.
                         */
                        val numCustomPurposes = tcModel.getNumCustomPurposes()
                        if (numCustomPurposes is StringOrNumber.Int) {
                            numBits = numCustomPurposes.value
                        }
                        if (numCustomPurposes is StringOrNumber.String) {
                            numBits = numCustomPurposes.value.toInt()
                        }
                    }
                } else {
                    numBits = bitLength.integer
                }

                try {
                    when (key) {
                        "version", "numCustomPurposes" -> {
                            value as TCModelPropType.Int
                            bitField += IntEncoder.encode(StringOrNumber.Int(value.value), numBits)
                        }
                        "cmpId",
                        "cmpVersion",
                        "consentScreen",
                        "vendorListVersion",
                        "policyVersion" -> {
                            value as TCModelPropType.StringOrNumber
                            val internalValue = value.value
                            internalValue as StringOrNumber.Int
                            bitField += IntEncoder.encode(
                                StringOrNumber.Int(internalValue.value),
                                numBits
                            )
                        }
                        "segmentType" -> {
                            value as TCModelPropType.StringOrNumber
                            bitField += IntEncoder.encode(value.value, numBits)
                        }
                        "created", "lastUpdated" -> {
                            value as TCModelPropType.Date
                            bitField += DateEncoder.encode(value.value!!, numBits)
                        }
                        "consentLanguage", "publisherCountryCode" -> {
                            value as TCModelPropType.String
                            bitField += LangEncoder.encode(value.value, numBits)
                        }
                        "isServiceSpecific",
                        "useNonStandardStacks",
                        "purposeOneTreatment" -> {
                            value as TCModelPropType.Boolean
                            bitField += BooleanEncoder.encode(value.value)
                        }
                        "specialFeatureOptins",
                        "purposeConsents",
                        "purposeLegitimateInterests",
                        "publisherConsents",
                        "publisherLegitimateInterests",
                        "publisherCustomConsents",
                        "publisherCustomLegitimateInterests" -> {
                            value as TCModelPropType.Vector
                            bitField += FixedVectorEncoder.encode(value.value, numBits)
                        }
                        "vendorConsents",
                        "vendorLegitimateInterests",
                        "vendorsDisclosed",
                        "vendorsAllowed" -> {
                            value as TCModelPropType.Vector
                            bitField += VendorVectorEncoder.encode(value.value)
                        }
                        "publisherRestrictions" -> {
                            value as TCModelPropType.PurposeRestrictionVector
                            bitField += PurposeRestrictionVectorEncoder.encode(value.value)
                        }
                        else -> {
                            throw EncodingError("Error encoding $segment->$key, value: $value")
                        }
                    }
                } catch (error: Throwable) {
                    throw EncodingError("Error encoding $segment->$key: ${error.message}")
                }
            }
            return Base64Url.encode(bitField)
        }

        fun decode(encodedString: String, tcModel: TCModel, segment: String): TCModel {
            val tempTCModel: TCModel = tcModel

            val bitField: String = Base64Url.decode(encodedString)
            var bStringIdx = 0

            if (segment == Segment.CORE.type) {
                val encodedVersion: Int = IntEncoder.decode(
                    value = bitField.substring(bStringIdx, bStringIdx + BitLength.version.integer),
                    numBits = BitLength.version.integer
                ).toInt()

                tempTCModel.setVersion(StringOrNumber.Int(encodedVersion))
            }

            if (segment != Segment.CORE.type) {
                bStringIdx += BitLength.segmentType.integer
            }

            val fieldSequence = fieldSequence.two as SequenceVersionMapType.SVMItemMap
            val sequence = fieldSequence.map[Segment.getSegmentByType(segment)] ?: error("Unable to find fieldSequence")

            sequence.forEach { key ->
                var numBits = BitLength.getByName(key)?.integer
                if (numBits == null) {
                    if (isPublisherCustom(key)) {
                        /**
                         * publisherCustom[Consents | LegitimateInterests] are an edge case
                         * because they are of variable length. The length is defined in a
                         * separate field named numCustomPurposes.
                         */
                        numBits = (tempTCModel.getNumCustomPurposes() as StringOrNumber.Int).value
                    }
                }

                if (numBits != 0) {
                    /**
                     * numBits could be 0 if this is a publisher custom purposes field and
                     * no custom purposes are defined. If that is the case, we don't need
                     * to gather no bits and we don't need to increment our bStringIdx
                     * pointer because those would all be 0 increments and would mess up
                     * the next logical if statement.
                     */
                    val endPosition = if (numBits == null) {
                        bitField.length
                    } else {
                        numBits + bStringIdx
                    }

                    val bits = bitField.substring(bStringIdx, endPosition)
                    when (key) {
                        "version" -> {
                            val decodedVersion = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setVersion(StringOrNumber.Int(decodedVersion.toInt()))
                        }
                        "created" -> {
                            val decodedCreated = DateEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.created = decodedCreated
                        }
                        "lastUpdated" -> {
                            val decodedCreated = DateEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.lastUpdated = decodedCreated
                        }
                        "cmpId" -> {
                            val decodedCmpId = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setCmpId(StringOrNumber.Int(decodedCmpId.toInt()))
                        }
                        "cmpVersion" -> {
                            val decodedCmpIdVersion = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setCmpVersion(StringOrNumber.Int(decodedCmpIdVersion.toInt()))
                        }
                        "consentScreen" -> {
                            val decodedConsentScreen = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setConsentScreen(StringOrNumber.Int(decodedConsentScreen.toInt()))
                        }
                        "consentLanguage" -> {
                            val decodedConsentLanguage = LangEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setConsentLanguage(decodedConsentLanguage)
                        }
                        "vendorListVersion" -> {
                            val decodedVendorListVersion = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setVendorListVersion(
                                StringOrNumber.Int(
                                    decodedVendorListVersion.toInt()
                                )
                            )
                        }
                        "purposeConsents" -> {
                            val decodedPurposeConsents =
                                FixedVectorEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.purposeConsents = decodedPurposeConsents
                        }
                        "vendorConsents" -> {
                            val decodedVendorConsents =
                                VendorVectorEncoder.decode(bits)
                            tempTCModel.vendorConsents = decodedVendorConsents
                        }
                        "policyVersion" -> {
                            val decodedPolicyVersion = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setPolicyVersion(StringOrNumber.Int(decodedPolicyVersion.toInt()))
                        }
                        "isServiceSpecific" -> {
                            val decodedIsServiceSpecific = BooleanEncoder.decode(bits)
                            tempTCModel.setIsServiceSpecific(decodedIsServiceSpecific)
                        }
                        "useNonStandardStacks" -> {
                            val decodedUseNonStandardStacks = BooleanEncoder.decode(bits)
                            tempTCModel.setUseNonStandardStacks(decodedUseNonStandardStacks)
                        }
                        "specialFeatureOptins" -> {
                            val decodedSpecialFeatureOptins =
                                FixedVectorEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.specialFeatureOptins = decodedSpecialFeatureOptins
                        }
                        "purposeLegitimateInterests" -> {
                            val decodedPurposeLegitimateInterests =
                                FixedVectorEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.purposeLegitimateInterests = decodedPurposeLegitimateInterests
                        }
                        "purposeOneTreatment" -> {
                            val decodedPurposeOneTreatment = BooleanEncoder.decode(bits)
                            tempTCModel.setPurposeOneTreatment(decodedPurposeOneTreatment)
                        }
                        "publisherCountryCode" -> {
                            val decodedPublisherCountryCode =
                                LangEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setPublisherCountryCode(decodedPublisherCountryCode)
                        }
                        "vendorLegitimateInterests" -> {
                            val decodedVendorLegitimateInterests =
                                VendorVectorEncoder.decode(bits)
                            tempTCModel.vendorLegitimateInterests = decodedVendorLegitimateInterests
                        }
                        "publisherRestrictions" -> {
                            val decodedPublisherRestrictions =
                                PurposeRestrictionVectorEncoder.decode(bits)
                            tempTCModel.publisherRestrictions = decodedPublisherRestrictions
                        }
                        "publisherConsents" -> {
                            val decodedPublisherConsents =
                                FixedVectorEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.publisherConsents = decodedPublisherConsents
                        }
                        "publisherLegitimateInterests" -> {
                            val decodedPublisherLegitimateInterests =
                                FixedVectorEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.publisherLegitimateInterests = decodedPublisherLegitimateInterests
                        }
                        "numCustomPurposes" -> {
                            val decodedNumCustomPurposes = IntEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.setNumCustomPurposes(
                                StringOrNumber.Int(
                                    decodedNumCustomPurposes.toInt()
                                )
                            )
                        }
                        "publisherCustomConsents" -> {
                            val decodedPublisherCustomConsents =
                                FixedVectorEncoder.decode(bits, (numBits ?: 0))
                            tempTCModel.publisherCustomConsents = decodedPublisherCustomConsents
                        }
                        "publisherCustomLegitimateInterests" -> {
                            val decodedPublisherCustomLegitimateInterests = FixedVectorEncoder
                                .decode(bits, (numBits ?: 0))
                            tempTCModel.publisherCustomLegitimateInterests = decodedPublisherCustomLegitimateInterests
                        }
                        "vendorsAllowed" -> {
                            val decodedVendorsAllowed =
                                VendorVectorEncoder.decode(bits)
                            tempTCModel.vendorsAllowed = decodedVendorsAllowed
                        }
                        "vendorsDisclosed" -> {
                            val decodedVendorsDisclosed =
                                VendorVectorEncoder.decode(bits)
                            tempTCModel.vendorsDisclosed = decodedVendorsDisclosed
                        }
                        else -> {
                            throw DecodingError("Unable to set decoded version of the key: $key")
                        }
                    }

                    if (numBits is Int) {
                        bStringIdx += numBits

                    } else {
                        val tcModelKeyProperty = tcModel.getFieldByName(key)
                        bStringIdx += when (tcModelKeyProperty) {
                            is TCModelPropType.Vector -> {
                                tcModelKeyProperty.value.bitLength
                            }
                            is TCModelPropType.PurposeRestrictionVector -> {
                                tcModelKeyProperty.value.bitLength
                            }
                            else -> {
                                throw DecodingError("Unable to increase bitLength for key: $key")
                            }
                        }
                    }
                }
            }
            return tempTCModel
        }

        fun isPublisherCustom(key: String): Boolean {
            return key.indexOf("publisherCustom") == 0
        }
    }
}
