package com.usercentrics.tcf.core.encoder

import com.usercentrics.tcf.core.GVL
import com.usercentrics.tcf.core.StringOrNumber
import com.usercentrics.tcf.core.TCModel
import com.usercentrics.tcf.core.errors.EncodingError
import com.usercentrics.tcf.core.model.RestrictionType
import com.usercentrics.tcf.core.model.Vector
import com.usercentrics.tcf.core.model.gvl.Vendor

internal typealias ProcessorFunction = (tcModel: TCModel, gvl: GVL) -> TCModel

internal class SemanticPreEncoder {

    companion object {
        private fun firstProcessorFunction(tcModel: TCModel): TCModel {
            return tcModel
        }

        private fun firstProcessorFunctionWrapper(
            tcModel: TCModel,
            @Suppress("UNUSED_PARAMETER") gvl: GVL
        ): TCModel {
            return firstProcessorFunction(tcModel)
        }

        private fun secondProcessorFunction(tcModel: TCModel, gvl: GVL): TCModel {
            /**
             * in case this wasn't set previously.  This should filter out invalid
             * purpose restrictions.
             */

            val purposeRestrictionVector = tcModel.publisherRestrictions.setGvl(gvl)

            /**
             * Purpose 1 is never allowed to be true for legitimate interest
             * As of TCF v2.2 purposes 3,4,5 & 6 are not allowed to be true for LI
             */
            tcModel.purposeLegitimateInterests.unset(listOf(1, 3, 4, 5, 6))

            /**
             * If a Vendor does not declare a purpose for consent or legitimate
             * interest they should not have a positive signal for it. This code
             * removes positive signals created mistakingly.
             */
            val vectorToIntMap: MutableMap<String, Vector> = mutableMapOf()

            vectorToIntMap["legIntPurposes"] = tcModel.vendorLegitimateInterests
            vectorToIntMap["purposes"] = tcModel.vendorConsents

            vectorToIntMap.forEach {

                val gvlVendorKey: String = it.key
                val vector: Vector = it.value

                vector.forEach { value: Boolean, id: Int ->
                    if (!value) {
                        return@forEach
                    }

                    val vendor: Vendor? = gvl.vendors!![id.toString()]
                    if (vendor == null || !vendor.deletedDate.isNullOrBlank()) {
                        /**
                         * If the vendor doesn't exist, then they should not receive a
                         * positive signal
                         */

                        vector.unset(id)
                        return@forEach
                    }

                    if (gvlVendorKey == "legIntPurposes" && vendor.purposes.isEmpty() && vendor.legIntPurposes.isEmpty() && vendor.specialPurposes.isNotEmpty()) {
                        /**
                         * Per June 2021 Policy change, Vendors declaring only Special Purposes must
                         * have their legitimate interest Vendor bit set if they have been disclosed.
                         * This block ensures their LI bit remains set
                         */
                        vector.set(id)
                        return@forEach
                    } else if (gvlVendorKey == "legIntPurposes" && vendor.purposes.isNotEmpty() && vendor.legIntPurposes.isEmpty() && vendor.specialPurposes.isNotEmpty()
                    ) {
                        /**
                         * Per June 2021 Policy change, Vendors declaring only Special Purposes must
                         * have their legitimate interest Vendor bit set if they have been disclosed.
                         * This block ensures their LI bit remains set
                         */
                        vector.set(id)
                        return@forEach
                    }

                    val restrictions = purposeRestrictionVector.getRestrictions(id)

                    if (gvlVendorKey == "legIntPurposes") {
                        val vendorPurposes = vendor.purposes
                        val vendorLegIntPurposes = vendor.legIntPurposes

                        var restrictedLegitimateInterestPurposes = vendorLegIntPurposes.toMutableList()
                        var restrictedPurposes = vendorPurposes.toMutableList()

                        restrictions.forEach { restriction ->
                            when (restriction.restrictionType) {
                                RestrictionType.REQUIRE_LI -> {
                                    restrictedPurposes = restrictedPurposes.filter { purposeId ->
                                        if (purposeId == restriction.getPurposeId()) {
                                            if (vendor.flexiblePurposes.contains(purposeId)) {
                                                restrictedLegitimateInterestPurposes.add(purposeId)
                                            }
                                            return@filter false
                                        }
                                        return@filter true
                                    }.toMutableList()
                                }
                                RestrictionType.REQUIRE_CONSENT -> {
                                    restrictedLegitimateInterestPurposes = restrictedLegitimateInterestPurposes.filter { purposeId ->
                                        if (purposeId == restriction.getPurposeId()) {
                                            if (vendor.flexiblePurposes.contains(purposeId)) {
                                                restrictedPurposes.add(purposeId)
                                            }
                                            return@filter false
                                        }
                                        return@filter true
                                    }.toMutableList()
                                }
                                RestrictionType.NOT_ALLOWED -> {
                                    restrictedPurposes = vendorPurposes.filter { purposeId ->
                                        purposeId != restriction.getPurposeId()
                                    }.toMutableList()

                                    restrictedLegitimateInterestPurposes = vendorLegIntPurposes.filter { purposeId ->
                                        purposeId != restriction.getPurposeId()
                                    }.toMutableList()
                                }
                            }
                        }

                        if (restrictedPurposes.isEmpty() && restrictedLegitimateInterestPurposes.isEmpty() && vendor.specialPurposes.isNotEmpty()) {
                            return@forEach
                        }

                        if (restrictedLegitimateInterestPurposes.isEmpty()) {
                            vector.unset(id)
                            return@forEach
                        }
                    }

                    val length: Int = when (gvlVendorKey) {
                        "purposes" -> vendor.purposes.size
                        "legIntPurposes" -> vendor.legIntPurposes.size
                        else -> -1
                    }
                    if (length != 0) {
                        return@forEach
                    }

                    val isServiceSpecific = tcModel.getIsServiceSpecific()
                    if (isServiceSpecific && vendor.flexiblePurposes.isEmpty()) {
                        vector.unset(id)
                        return@forEach
                    }

                    if (!isServiceSpecific) {
                        vector.unset(id)
                        return@forEach
                    }

                    /**
                     * They have some flexible purposes, we should check for a
                     * publisher restriction value that would enable this vendor to
                     * have the override-preferred basis.
                     */
                    var isValid = false
                    var i = 0
                    while (i < restrictions.size && !isValid) {
                        val restrictionType = restrictions[i].restrictionType
                        isValid = ((restrictionType == RestrictionType.REQUIRE_CONSENT && gvlVendorKey == "purposes") ||
                                (restrictionType == RestrictionType.REQUIRE_LI && gvlVendorKey == "legIntPurposes"))
                        i++
                    }

                    if (!isValid) {
                        vector.unset(id)
                    }
                }
            }

            tcModel.vendorsDisclosed.set(gvl.vendors?.map {
                it.value.id
            }?.toList()!!)

            return tcModel
        }

        private val processor: List<ProcessorFunction> = listOf(
            Companion::firstProcessorFunctionWrapper,
            Companion::secondProcessorFunction
        )

        fun process(tcModel: TCModel): TCModel {
            val gvl = tcModel.getGvl() ?: throw EncodingError("Unable to encode TCModel without a GVL")
            if (!gvl.getIsReady()) {
                throw EncodingError("Unable to encode TCModel tcModel.gvl.readyPromise is not resolved")
            }

            val internalTCModel: TCModel = tcModel//.clone()
            internalTCModel.setConsentLanguage(gvl.getLanguage().uppercase())
            gvl.vendorListVersion?.let { internalTCModel.setVendorListVersion(StringOrNumber.Int(it)) }

            val processorFunctionIndex = internalTCModel.getVersion() - 1
            val processorFunction: ProcessorFunction?

            try {
                processorFunction = processor[processorFunctionIndex]
            } catch (_: Throwable) {
                throw EncodingError("Invalid version ${internalTCModel.getVersion()}")
            }

            return processorFunction(internalTCModel, gvl)
        }
    }
}
