package com.usercentrics.tcf.core.encoder.field

import com.usercentrics.tcf.core.StringOrNumber
import com.usercentrics.tcf.core.encoder.BitLength
import com.usercentrics.tcf.core.errors.DecodingError
import com.usercentrics.tcf.core.model.Vector

internal class VendorVectorEncoder {

    companion object {

        fun encode(value: Vector): String {
            // collectors for range encoding
            val ranges: MutableList<MutableList<Int>> = mutableListOf()
            var range: MutableList<Int> = mutableListOf()

            // since both encodings need the maxId, start with that
            var retrString = IntEncoder.encode(StringOrNumber.Int(value.getMaxId()), BitLength.maxId.integer)

            // bit field will be just the vendors as we walk through the vector
            var bitField = ""
            var rangeIsSmaller = false

            // some math
            val headerLength = BitLength.maxId.integer + BitLength.encodingType.integer
            val bitFieldLength = headerLength + value.getMaxId()
            val minRangeLength = ((BitLength.vendorId.integer * 2) + BitLength.singleOrRange.integer + BitLength.numEntries.integer)

            // gets larger as we walk through the vector
            var rangeLength = headerLength + BitLength.numEntries.integer

            // walk through every value in the vector
            value.forEach { curValue: Boolean, i: Int ->

                // build our bitfield no matter what
                bitField += BooleanEncoder.encode(curValue)

                /**
                 * A range is a minimum of 45 bits, if the number of vendors in this
                 * vector is less than 45 then we know that a bitfield encoding will be
                 * shorter than any range encoding.
                 *
                 * The second check checks while we walk through the vector and abandons
                 * building the ranges once it becomes larger
                 */
                rangeIsSmaller = (value.getMaxId() > minRangeLength && rangeLength < bitFieldLength)

                /**
                 * if the curValue is true and our rangeLength is less than the bitField
                 * length, we'll continue to push these ranges into the array.  Once the
                 * ranges become a larger encoding there is no reason to continue
                 * building the structure because we will be choosing the bitfield
                 * encoding
                 */
                if (rangeIsSmaller && curValue) {

                    /**
                     * Look ahead to see if this is the last value in our range
                     */
                    val nextValue = value.has(i + 1)

                    // if there isn't a next value, then we'll wrap up this range
                    if (!nextValue) {
                        /**
                         * this is the last value of the range, so we'll push it on to the
                         * end into position 1
                         */
                        range.add(i)

                        // add to the range length the additional vendorId
                        rangeLength += BitLength.vendorId.integer

                        // store the array in our bigger array
                        ranges.add(range.toMutableList())

                        // clear the array for the next range
                        range.clear()
                        range = mutableListOf()

                    } else if (range.size == 0) {

                        // this is the first value for this range
                        range.add(i)

                        // update our count with the new range overhead
                        rangeLength += BitLength.singleOrRange.integer
                        rangeLength += BitLength.vendorId.integer

                    }
                }
            }

            if (rangeIsSmaller) {
                retrString += VectorEncodingType.RANGE.value
                retrString += buildRangeEncoding(ranges)

            } else {
                retrString += VectorEncodingType.FIELD.value
                retrString += bitField
            }
            return retrString
        }

        fun decode(value: String): Vector {
            val vector: Vector
            var index = 0

            val maxId: Int = IntEncoder.decode(value.substring(index, index + BitLength.maxId.integer), BitLength.maxId.integer).toInt()
            index += BitLength.maxId.integer

            val encodingType = VectorEncodingType.getVectorEncodingTypeByValue(IntEncoder.decode(value[index].toString(), BitLength.encodingType.integer).toInt())
            index += BitLength.encodingType.integer

            /**
             * Range is handled in batches so we'll need a different decoding scheme
             */
            if (encodingType == VectorEncodingType.RANGE) {

                vector = Vector()

                val numEntries = IntEncoder.decode(value.substring(index, index + BitLength.numEntries.integer), BitLength.numEntries.integer).toInt()

                index += BitLength.numEntries.integer

                // loop through each group of entries
                for (i in 0 until numEntries) {

                    // Ranges can represent a single id or a range of ids
                    val isIdRange: Boolean = BooleanEncoder.decode(value[index].toString())

                    index += BitLength.singleOrRange.integer

                    /**
                     * regardless of whether or not it's a single entry or range, the next
                     * set of bits is a vendor ID
                     */
                    val firstId = IntEncoder.decode(value.substring(index, index + BitLength.vendorId.integer), BitLength.vendorId.integer).toInt()

                    index += BitLength.vendorId.integer

                    // if it's a range. the next set of bits is the second id
                    if (isIdRange) {
                        val secondId = IntEncoder.decode(value.substring(index, index + BitLength.vendorId.integer), BitLength.vendorId.integer).toInt()

                        index += BitLength.vendorId.integer

                        // we'll need to set or unset all the vendor ids between the first and second
                        for (j in firstId..secondId) {
                            vector.set(j)
                        }

                    } else {
                        vector.set(firstId)
                    }
                }

            } else {
                val bitField = value.substring(index, index + maxId)

                index += maxId
                vector = FixedVectorEncoder.decode(bitField, maxId)
            }

            vector.bitLength = index

            return vector
        }

        private fun buildRangeEncoding(ranges: List<List<Int>>): String {

            // describe the number of entries to follow
            val numEntries: Int = ranges.size
            var rangeString = IntEncoder.encode(StringOrNumber.Int(numEntries), BitLength.numEntries.integer)

            // each range
            ranges.forEach { range ->

                // is this range a single?
                val single: Boolean = range.size == 1

                // first is the indicator of whether this is a single id or range (two)
                // 0 is single and range is 1
                rangeString += BooleanEncoder.encode(!single)

                // second is the first (or only) vendorId
                rangeString += IntEncoder.encode(StringOrNumber.Int(range[0]), BitLength.vendorId.integer)

                if (!single) {
                    // add the second id if it exists
                    rangeString += IntEncoder.encode(StringOrNumber.Int(range[1]), BitLength.vendorId.integer)
                }
            }
            return rangeString
        }
    }
}
