package com.usercentrics.tcf.core.model

//type SingleIDOrCollection = number | number[] | IntMap<unknown> | Set<number | string>;
internal sealed class SingleIDOrCollection {
    class Int(val value: kotlin.Int) : SingleIDOrCollection()
    class List<Int>(val value: kotlin.collections.List<Int>) : SingleIDOrCollection()
    class Map<String, Any>(val value: kotlin.collections.Map<String, Any>) : SingleIDOrCollection()
    class Set<Int>(val value: kotlin.collections.Set<Int>) : SingleIDOrCollection()
    class String(val value: kotlin.String) : SingleIDOrCollection()
}

/**
 * Vector class is like a Set except it keeps track of a max id
 */
@Suppress("PrivatePropertyName")
//export type IdBoolTuple = [number, boolean];// => Pair<Int,Boolean>
internal class Vector : Iterable<Pair<Int, Boolean>> {
    /**
     * if this originated from an encoded string we'll need a place to store the
     * bit length; it can be set and got from here
     */
    var bitLength = 0
    private var maxId_ = 0

    /**
     * keep a set for faster lookup
     */
    private val set_ = mutableSetOf<Int>()

    override fun iterator(): Iterator<Pair<Int, Boolean>> = object : Iterator<Pair<Int, Boolean>> {
        var i = 1

        override fun hasNext(): Boolean = i <= maxId_

        override fun next(): Pair<Int, Boolean> {
            val index = i
            i += 1
            return index to has(index)
        }
    }

    /**
     * maxId
     *
     * @return {number} - the highest id in this Vector
     */
    fun getMaxId(): Int = this.maxId_

    /**
     * get
     *
     * @param {number} id - key for value to check
     * @return {boolean} - value of that key, if never set it will be false
     */
    fun has(id: Int): Boolean {
        return this.set_.contains(id)
    }

    fun unset(item: Int) {
        unset(listOf(item))
    }

    fun unset(items: List<Int>) {
        this.set_.removeAll(items.toSet())
        this.bitLength = 0
        this.maxId_ = set_.maxOrNull() ?: 0
    }

    fun set(item: Map<String, *>) {
        set(item.keys.toList().map { it.toInt() })
    }

    fun set(item: Int) {
        set(listOf(item))
    }

    fun set(items: List<Int>) {
        this.set_.addAll(items)
        this.bitLength = 0
        this.maxId_ = set_.maxOrNull() ?: 0
        this.bitLength = 0
    }

    fun clear() {
        this.set_.clear()
    }

    /**
     * forEach - to traverse from id=1 to id=maxId in a sequential non-sparse manner
     *
     *
     * @param {forEachCallback} callback - callback to execute
     * @return {void}
     *
     * @callback forEachCallback
     * @param {boolean} value - whether or not this id exists in the vector
     * @param {number} id - the id number of the current iteration
     */
    fun forEach(callback: (value: Boolean, id: Int) -> Unit) {
        var i = 1
        while (i <= this.maxId_) {
            callback(this.has(i), i)
            i++
        }
    }

    fun getSize(): Int {
        return this.set_.size
    }

    override fun equals(other: Any?): Boolean {
        if(this === other) return true

        other as Vector
        if (!set_.containsAll(other.set_)) return false
        if (maxId_!= other.maxId_) return false
        if (bitLength != other.bitLength) return false
        return true
    }
}
