/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.collaborationengine;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.vaadin.collaborationengine.CollaborationEngine;
import com.vaadin.collaborationengine.CollaborationEngineConfiguration;
import com.vaadin.collaborationengine.FileLicenseStorage;
import com.vaadin.collaborationengine.LicenseEvent;
import com.vaadin.collaborationengine.LicenseStorage;
import com.vaadin.flow.internal.MessageDigestUtil;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.YearMonth;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.List;

class LicenseHandler {
    static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_DATE;
    static final ObjectMapper MAPPER = LicenseHandler.createObjectMapper();
    private final CollaborationEngine ce;
    final LicenseStorage licenseStorage;
    final LicenseInfo license;

    LicenseHandler(CollaborationEngine collaborationEngine) {
        this.ce = collaborationEngine;
        CollaborationEngineConfiguration configuration = collaborationEngine.getConfiguration();
        if (configuration.isLicenseCheckingEnabled()) {
            LicenseStorage configuredStorage = configuration.getLicenseStorage();
            this.licenseStorage = configuredStorage != null ? configuredStorage : new FileLicenseStorage(configuration);
            String licenseProperty = configuration.getLicenseProperty();
            if (licenseProperty != null) {
                this.license = this.parseLicense(this.getLicenseFromProperty(licenseProperty));
            } else {
                Path dataDirPath = configuration.getDataDirPath();
                if (dataDirPath == null) {
                    throw FileLicenseStorage.createDataDirNotConfiguredException();
                }
                Path licenseFilePath = LicenseHandler.createLicenseFilePath(dataDirPath);
                this.license = this.parseLicense(this.getLicenseFromFile(licenseFilePath));
            }
            if (this.license.endDate.isBefore(this.getCurrentDate())) {
                CollaborationEngine.LOGGER.warn("Your Collaboration Engine license has expired. Your application will still continue to work, but the collaborative features will be disabled. Please contact Vaadin about obtaining a new, up-to-date license for your application. https://vaadin.com/collaboration");
            }
        } else {
            this.licenseStorage = null;
            this.license = null;
        }
    }

    private Reader getLicenseFromProperty(String licenseProperty) {
        byte[] license = Base64.getDecoder().decode(licenseProperty);
        return new InputStreamReader(new ByteArrayInputStream(license));
    }

    private Reader getLicenseFromFile(Path licenseFilePath) {
        try {
            return Files.newBufferedReader(licenseFilePath);
        }
        catch (NoSuchFileException e) {
            throw this.createLicenseNotFoundException(licenseFilePath, e);
        }
        catch (IOException e) {
            throw new IllegalStateException("Collaboration Engine wasn't able to read the license file at '" + licenseFilePath + "'. Check that the file is readable by the app, and not locked.", e);
        }
    }

    private LicenseInfo parseLicense(Reader licenseReader) {
        try {
            JsonNode licenseJson = MAPPER.readTree(licenseReader);
            LicenseInfoWrapper licenseInfoWrapper = (LicenseInfoWrapper)MAPPER.treeToValue((TreeNode)licenseJson, LicenseInfoWrapper.class);
            String calculatedChecksum = LicenseHandler.calculateChecksum(licenseJson.get("content"));
            if (licenseInfoWrapper.checksum == null || !licenseInfoWrapper.checksum.equals(calculatedChecksum)) {
                throw this.createLicenseInvalidException(null);
            }
            return licenseInfoWrapper.content;
        }
        catch (IOException e) {
            throw this.createLicenseInvalidException(e);
        }
    }

    synchronized boolean registerUser(String userId) {
        boolean hasActiveSeat;
        LocalDate currentDate = this.getCurrentDate();
        if (this.isGracePeriodEnded(currentDate)) {
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.GRACE_PERIOD_ENDED);
        }
        if (this.license.endDate.isBefore(currentDate)) {
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.LICENSE_EXPIRED);
            return false;
        }
        if (this.license.endDate.minusDays(31L).isBefore(currentDate)) {
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.LICENSE_EXPIRES_SOON);
        }
        YearMonth month = YearMonth.from(currentDate);
        List<String> users = this.licenseStorage.getUserEntries(this.license.key, month);
        int effectiveQuota = this.isGracePeriodOngoing(currentDate) ? this.license.quota * 10 : this.license.quota;
        boolean bl = hasActiveSeat = users.size() <= effectiveQuota ? users.contains(userId) : users.stream().limit(effectiveQuota).anyMatch(user -> user.equals(userId));
        if (hasActiveSeat) {
            return true;
        }
        if (users.size() >= effectiveQuota) {
            if (this.getGracePeriodStarted() != null) {
                return false;
            }
            this.fireLicenseEvent(LicenseEvent.LicenseEventType.GRACE_PERIOD_STARTED);
        }
        this.licenseStorage.addUserEntry(this.license.key, month, userId);
        return true;
    }

    LocalDate getGracePeriodStarted() {
        return this.licenseStorage.getLatestLicenseEvents(this.license.key).get(LicenseEvent.LicenseEventType.GRACE_PERIOD_STARTED.name());
    }

    private boolean isGracePeriodOngoing(LocalDate currentDate) {
        return this.getGracePeriodStarted() != null && !this.isGracePeriodEnded(currentDate);
    }

    private boolean isGracePeriodEnded(LocalDate currentDate) {
        return this.getGracePeriodStarted() != null && currentDate.isAfter(this.getLastGracePeriodDate());
    }

    private LocalDate getLastGracePeriodDate() {
        return this.getGracePeriodStarted().plusDays(30L);
    }

    private void fireLicenseEvent(LicenseEvent.LicenseEventType type) {
        String message;
        String eventName = type.name();
        if (this.licenseStorage.getLatestLicenseEvents(this.license.key).get(eventName) != null) {
            return;
        }
        switch (type) {
            case GRACE_PERIOD_STARTED: {
                LocalDate gracePeriodEnd = this.getCurrentDate().plusDays(31L);
                message = type.createMessage(gracePeriodEnd.format(DATE_FORMATTER));
                break;
            }
            case LICENSE_EXPIRES_SOON: {
                message = type.createMessage(this.license.endDate.format(DATE_FORMATTER));
                break;
            }
            default: {
                message = type.createMessage(new Object[0]);
            }
        }
        LicenseEvent event = new LicenseEvent(this.ce, type, message);
        this.licenseStorage.setLicenseEvent(this.license.key, eventName, this.getCurrentDate());
        this.ce.getConfiguration().getLicenseEventHandler().handleLicenseEvent(event);
    }

    private LocalDate getCurrentDate() {
        return LocalDate.now(this.ce.getClock());
    }

    private RuntimeException createLicenseInvalidException(Throwable cause) {
        return new IllegalStateException("The content of the license property or file is not valid. If you have made any changes to the file, please revert those changes. If that's not possible, contact Vaadin to get a new copy of the license file.", cause);
    }

    private RuntimeException createLicenseNotFoundException(Path licenseFilePath, Throwable cause) {
        return new IllegalStateException("Collaboration Engine failed to find the license file at '" + licenseFilePath + ". Using Collaboration Engine in production requires a valid license property or file. Instructions for obtaining a license can be found in the Vaadin documentation. If you already have a license, make sure that the 'vaadin.ce.license' property is set or, if you have a license file, the 'vaadin.ce.dataDir' property is pointing to the correct directory and that the directory contains the license file.", cause);
    }

    static String calculateChecksum(JsonNode node) throws JsonProcessingException {
        return Base64.getEncoder().encodeToString(MessageDigestUtil.sha256((String)MAPPER.writeValueAsString((Object)node)));
    }

    static ObjectMapper createObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule((Module)new JavaTimeModule());
        objectMapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.NON_PRIVATE);
        objectMapper.setDateFormat((DateFormat)new SimpleDateFormat("yyyy-MM-dd"));
        return objectMapper;
    }

    static Path createLicenseFilePath(Path dirPath) {
        return Paths.get(dirPath.toString(), "ce-license.json");
    }

    static class LicenseInfoWrapper {
        final LicenseInfo content;
        final String checksum;

        @JsonCreator
        LicenseInfoWrapper(@JsonProperty(value="content", required=true) LicenseInfo content, @JsonProperty(value="checksum", required=true) String checksum) {
            this.content = content;
            this.checksum = checksum;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown=true)
    static class LicenseInfo {
        final String key;
        final int quota;
        final LocalDate endDate;

        @JsonCreator
        LicenseInfo(@JsonProperty(value="key", required=true) String key, @JsonProperty(value="quota", required=true) int quota, @JsonProperty(value="endDate", required=true) LocalDate endDate) {
            this.key = key;
            this.quota = quota;
            this.endDate = endDate;
        }
    }
}

