package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.DevToolsMessageHandler;
import com.vaadin.copilot.ai.AICommandHandler;
import com.vaadin.copilot.analytics.AnalyticsClient;
import com.vaadin.copilot.feedback.FeedbackHandler;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.ide.IdeHandler;
import com.vaadin.copilot.ide.IdePluginCommandHandler;
import com.vaadin.copilot.ide.OpenComponentInIDE;
import com.vaadin.copilot.plugins.accessibilitychecker.AccessibilityCheckerMessageHandler;
import com.vaadin.copilot.plugins.i18n.I18nHandler;
import com.vaadin.copilot.plugins.info.InfoHandler;
import com.vaadin.copilot.plugins.themeeditor.ThemeEditorMessageHandler;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.startup.ApplicationConfiguration;
import elemental.json.Json;
import elemental.json.JsonObject;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class Copilot implements DevToolsMessageHandler {

    public static final String PREFIX = "copilot-";

    static final String VALUE = "value";

    static final String CONTEXT = "context";

    private Boolean enable = null;

    ProjectManager projectManager;

    private List<CopilotCommand> commands;

    private AnalyticsClient analyticsClient;

    private CopilotIDEPlugin idePlugin;

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        if (!isEnabled()) {
            return;
        }

        try {
            VaadinServletContext context = getContext();
            ApplicationConfiguration applicationConfiguration = ApplicationConfiguration
                    .get(context);
            Path projectRoot = applicationConfiguration.getProjectFolder()
                    .toPath();
            CopilotIDEPlugin.setProjectRoot(projectRoot);
            projectManager = new ProjectManager(applicationConfiguration);
            idePlugin = CopilotIDEPlugin.getInstance();
            if (!isDevelopmentMode() && isUsageStatisticsEnabled()) {
                analyticsClient = new AnalyticsClient();
            }
            setupCommands(applicationConfiguration, context);
            devToolsInterface.send(PREFIX + "init", null);

            for (CopilotCommand copilotCommand : commands) {
                copilotCommand.handleConnect(devToolsInterface);
            }

        } catch (IOException e) {
            throw new RuntimeException("Unable to create project manager", e);
        }
    }

    private void setupCommands(
            ApplicationConfiguration applicationConfiguration,
            VaadinServletContext context) {
        commands = List.of(
                // This must be first as it is more of an interceptor than a
                // handler
                new IdeHandler(idePlugin),
                new OpenComponentInIDE(applicationConfiguration,
                        projectManager),
                new ProjectFileHandler(projectManager),
                new AICommandHandler(projectManager),
                new UserInfoHandler(context),
                new I18nHandler(applicationConfiguration, projectManager),
                new IdePluginCommandHandler(projectManager),
                new ApplicationInitializer(projectManager, context),
                new MachineConfigurationHandler(),
                new ThemeEditorMessageHandler(projectManager),
                new RouteHandler(projectManager, context),
                new AccessibilityCheckerMessageHandler(projectManager),
                new InfoHandler(applicationConfiguration),
                new FeedbackHandler(), new JavaRewriteHandler(projectManager),
                new PreviewHandler(), new JavaParserHandler());

    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (!isEnabled()) {
            return false;
        }

        if (!command.startsWith(PREFIX)) {
            return false;
        }

        Map<String, String> context = getContextMap(data);
        // remove context from data not to cause unmarshalling issues
        data.remove(CONTEXT);
        if (command.equals("copilot-track-event")) {
            String event = PREFIX + data.getString("event");
            if (data.hasKey(VALUE)) {
                track(event, Map.of(VALUE, data.getString(VALUE)), context);
            } else {
                track(event, null, context);
            }
            return true;
        }
        try {
            command = command.substring(PREFIX.length());
            for (CopilotCommand copilotCommand : commands) {
                if (copilotCommand.handleMessage(command, data,
                        devToolsInterface)) {
                    return true;
                }
            }
        } finally {
            track("copilot-command-" + command, null, context);
        }

        JsonObject respData = Json.createObject();
        if (data.hasKey(CopilotCommand.KEY_REQ_ID)) {
            respData.put(CopilotCommand.KEY_REQ_ID,
                    data.getString(CopilotCommand.KEY_REQ_ID));
        }
        data.put("error", "Unknown command " + command);
        devToolsInterface.send("unknown-command", data);
        return true;
    }

    private void track(String event, Map<String, String> properties,
            Map<String, String> context) {
        if (analyticsClient != null) {
            analyticsClient.track(event, properties, context);
        }
    }

    private boolean isUsageStatisticsEnabled() {
        return ApplicationConfiguration
                .get(VaadinService.getCurrent().getContext())
                .isUsageStatisticsEnabled();
    }

    public static boolean isDevelopmentMode() {
        return Boolean.getBoolean("copilot.development");
    }

    private Map<String, String> getContextMap(JsonObject data) {
        JsonObject context = data.getObject(CONTEXT);
        Map<String, String> contextMap = new HashMap<>();
        for (String key : context.keys()) {
            contextMap.put(key, context.getString(key));
        }
        return contextMap;
    }

    private boolean isEnabled() {
        if (enable == null) {
            enable = CopilotStatus.isEnabled(getContext());
        }
        return enable;
    }

    private VaadinServletContext getContext() {
        return (VaadinServletContext) VaadinService.getCurrent().getContext();

    }
}
