package com.vaadin.copilot.javarewriter;

import com.vaadin.flow.internal.JsonUtils;
import elemental.json.JsonArray;
import elemental.json.JsonBoolean;
import elemental.json.JsonNull;
import elemental.json.JsonNumber;
import elemental.json.JsonObject;
import elemental.json.JsonString;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Represents a Java component to be added to the source code.
 *
 * @param tag
 *            the tag of the component, used to determine the Java class
 * @param className
 *            the class name if known, or null to use the tag to look up the
 *            class
 * @param props
 *            the properties of the component
 * @param children
 *            the child components
 */
public record JavaComponent(String tag,String className,Map<String,Object>props,List<JavaComponent>children){

// @formatter:off
    private static Object propertyValueFromJson(Object jsonValue) {
        if (jsonValue instanceof JsonBoolean jsonboolean) {
            return jsonboolean.getBoolean();
        } else if (jsonValue instanceof JsonString jsonString) {
            return jsonString.getString();
        } else if (jsonValue instanceof JsonNull) {
            return null;
        } else if (jsonValue instanceof JsonNumber number) {
            if (number.asString().equals("" + (int) number.asNumber())) {
                // Integer
                return (int) ((JsonNumber) jsonValue).getNumber();
            }
            return number.asNumber();
        } else if (jsonValue instanceof JsonObject jsonObject) {
            Map<String, Object> values = new HashMap<>();
            for (String key : jsonObject.keys()) {
                values.put(key, propertyValueFromJson(jsonObject.get(key)));
            }
            return values;
        } else if (jsonValue instanceof JsonArray jsonArray) {
            List<Object> values = new ArrayList<>();
            for (int i = 0; i < jsonArray.length(); i++) {
                values.add(propertyValueFromJson(jsonArray.get(i)));
            }
            return values;
        } else {
            throw new IllegalArgumentException("Unsupported JSON value: " + jsonValue);
        }
    }

    /**
     * Creates a new JavaComponent instance from a JSON object.
     *
     * @param json the JSON object
     * @return the JavaComponent instance
     */
    public static JavaComponent componentFromJson(JsonObject json) {
        String tag = json.hasKey("tag") ? json.getString("tag") : null;
        String className = json.hasKey("className") ? json.getString("className") : null;
        Map<String, Object> props = new HashMap<>();
        JsonObject jsonProps = json.getObject("props");
        if (jsonProps != null) {
            for (String key : jsonProps.keys()) {
                props.put(key, propertyValueFromJson(jsonProps.get(key)));
            }
        }
        List<JavaComponent> children = new ArrayList<>();
        JsonArray jsonChildren = json.getArray("children");
        if (jsonChildren != null) {
            for (int i = 0; i < jsonChildren.length(); i++) {
                JavaComponent child = JavaComponent.componentFromJson(jsonChildren.getObject(i));
                if ("prefix".equals(child.props.get("slot"))) {
                    child.props.remove("slot");
                    props.put("prefixComponent", child);
                } else if ("add-button".equals(child.props.get("slot"))) {
                    child.props.remove("slot");
                    props.put("uploadButton", child);
                } else {
                    children.add(child);
                }
            }
        }
        return new JavaComponent(tag, className, props, children);
    }


    /**
     * Creates a new JavaComponent instance from a JSON array.
     *
     * @param template the JSON array
     * @return the JavaComponent instances
     */
    public static List<JavaComponent> componentsFromJson(JsonArray template) {
        return JsonUtils.stream(template).map(JsonObject.class::cast).map(JavaComponent::componentFromJson).toList();
    }
    // @formatter:on
}
