package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.vaadin.flow.dom.Style;

import java.util.List;

public class JavaStyleRewriter {

    public record StyleInfo(String property, String value,
            boolean isDashSeparatedProperty) {
    }

    /**
     * Gets the (active) styles of a component.
     *
     * @param componentInfo
     *            the component to get the styles of
     * @return the styles, as a list of style names and values
     */
    public static List<StyleInfo> getStyles(ComponentInfo componentInfo) {
        return JavaRewriterUtil.findCalls(Style.class, componentInfo).stream()
                .map(JavaStyleRewriter::extractStyle).toList();
    }

    /**
     * Extracts the style information from a method call to a method inside
     * Style.
     * <p>
     * Note that this only processes the method call and not any chained calls.
     *
     * @param methodCallExpr
     *            a method call that refers to a method inside Style.
     * @return The style info for the method call.
     */
    private static StyleInfo extractStyle(MethodCallExpr methodCallExpr) {
        String setter = methodCallExpr.getNameAsString();
        if (setter.equals("set")) {
            String dashProperty = String.valueOf(JavaRewriterUtil
                    .fromExpression(methodCallExpr.getArgument(0), null));
            String value = String.valueOf(JavaRewriterUtil
                    .fromExpression(methodCallExpr.getArgument(1), null));
            return new StyleInfo(dashProperty, value, true);
        } else {
            int argCount = methodCallExpr.getArguments().size();
            if (argCount != 1) {
                throw new IllegalArgumentException(
                        "Expected styles method call expression to have one argument but was "
                                + argCount + " for " + methodCallExpr);
            }
            Expression argument = methodCallExpr.getArgument(0);
            String value;
            if (argument.isFieldAccessExpr()) {
                String fieldName = argument.asFieldAccessExpr()
                        .getNameAsString();
                // This is copied from Styles.java
                value = fieldName.replace("_", "-").toLowerCase();
            } else if (argument.isStringLiteralExpr()) {
                value = argument.asStringLiteralExpr().getValue();
            } else {
                throw new IllegalArgumentException(
                        "Unexpected argument type in style call: "
                                + methodCallExpr);
            }
            String property = JavaRewriterUtil.getPropertyName(setter);
            return new StyleInfo(property, value, false);
        }
    }

}
