package com.vaadin.copilot.ide;

import com.vaadin.open.OSUtils;
import com.vaadin.open.Open;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Utility class for currently used IDE
 * <p>
 * Supports detecting VS Code, Eclipse and IntelliJ.
 * </p>
 */
public final class IdeUtils {

    public enum IDE {
        IDEA(".idea", "idea"), VSCODE(".vscode", "vscode"), ECLIPSE(".eclipse",
                "eclipse");

        // dot directory for IDE metadata files
        private final String metaDir;

        // value used in plugins
        private final String pluginIde;

        IDE(String metaDir, String pluginIde) {
            this.metaDir = metaDir;
            this.pluginIde = pluginIde;
        }

        public String getMetaDir() {
            return metaDir;
        }

        public String getPluginIde() {
            return pluginIde;
        }
    }

    private IdeUtils() {
        // Utils only
    }

    /**
     * Finds IDE used to run application
     *
     * @return optional IDE
     */
    public static Optional<IDE> findIde() {
        List<ProcessHandle.Info> processes = getProcessTree();
        for (ProcessHandle.Info info : processes) {
            if (isIdea(info)) {
                return Optional.of(IDE.IDEA);
            }
            if (isVSCode(info)) {
                return Optional.of(IDE.VSCODE);
            }
            if (isEclipse(info)) {
                return Optional.of(IDE.ECLIPSE);
            }
        }
        return Optional.empty();
    }

    /**
     * Opens the given file at the given line number in the IDE used to launch
     * the current Java application.
     * <p>
     * If you are running the Java application from the command line or from an
     * unsupported IDE, then this method does nothing.
     *
     * @param file
     *            the file to open
     * @param lineNumber
     *            the line number to highlight
     */
    public static void openFile(File file, int lineNumber) {
        CopilotIDEPlugin idePlugin = CopilotIDEPlugin.getInstance();
        if (idePlugin.isActive()) {
            try {
                // use 0 as first line
                idePlugin.showInIde(file.getAbsolutePath(), lineNumber - 1, 0);
                return;
            } catch (IOException ex) {
                getLogger().warn(
                        "Cannot show file in IDE using Copilot IDE Plugin", ex);
            }
        }

        if (!openFileUsingBinary(file, lineNumber)) {
            getLogger().warn("Cannot open {} in IDE", file);
        }
    }

    private static boolean openFileUsingBinary(File file, int lineNumber) {
        String absolutePath = file.getAbsolutePath();

        Optional<ProcessHandle.Info> maybeIdeCommand = findIdeCommandInfo();
        if (maybeIdeCommand.isEmpty()) {
            getLogger().debug("Unable to detect IDE from process tree");
            printProcessTree(msg -> getLogger().debug(msg));
            return false;
        }

        ProcessHandle.Info processInfo = maybeIdeCommand.get();

        if (isVSCode(processInfo)) {
            return Open.open("vscode://file" + absolutePath + ":" + lineNumber);
        } else if (isIdea(processInfo)) {
            try {
                run(getBinary(processInfo), "--line", lineNumber + "",
                        absolutePath);
                return true;
            } catch (IOException e) {
                getLogger().error("Unable to launch IntelliJ IDEA", e);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }

        } else if (isEclipse(processInfo)) {
            if (OSUtils.isMac()) {
                try {
                    run("open", "-a", getBinary(processInfo), absolutePath);
                    return true;
                } catch (IOException e) {
                    getLogger().error("Unable to launch Eclipse", e);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            } else {
                try {
                    run(getBinary(processInfo),
                            absolutePath + ":" + lineNumber);
                    return true;
                } catch (IOException e) {
                    getLogger().error("Unable to launch Eclipse", e);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }

        return false;

    }

    static String getBinary(ProcessHandle.Info info) {
        String cmd = info.command().orElseThrow();
        if (isIdea(info)) {
            return getIdeaBinary(info);
        } else if (isEclipse(info)) {
            return cmd.replaceFirst("/Contents/MacOS/eclipse$", "");
        }
        return cmd;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(IdeUtils.class);
    }

    public static void main(String[] args) {
        // This is so it will be easier to debug problems in the future
        printProcessTree(System.out::println);
    }

    private static void printProcessTree(Consumer<String> printer) {
        for (ProcessHandle.Info info : getProcessTree()) {
            printer.accept("Process tree:");
            info.command()
                    .ifPresent(value -> printer.accept("Command: " + value));
            info.commandLine().ifPresent(
                    value -> printer.accept("Command line: " + value));
            info.arguments().ifPresent(values -> {
                for (int i = 0; i < values.length; i++) {
                    printer.accept("Arguments[" + i + "]: " + values[i]);
                }
            });
            printer.accept("");
        }

    }

    static void run(String command, String... arguments)
            throws IOException, InterruptedException {
        List<String> cmd = new ArrayList<>();
        cmd.add(command);
        cmd.addAll(Arrays.asList(arguments));
        ProcessBuilder pb = new ProcessBuilder().command(cmd);
        pb.redirectErrorStream(true);
        Process process = pb.start();
        int exitCode = process.waitFor();
        if (exitCode != 0) {
            String output = IOUtils.toString(process.getInputStream(),
                    StandardCharsets.UTF_8);
            throw new IOException(
                    "Command " + cmd + " terminated with exit code " + exitCode
                            + ".\nOutput:\n" + output);
        }
    }

    private static List<ProcessHandle.Info> getProcessTree() {
        return getParentProcesses().stream().map(ProcessHandle::info)
                .collect(Collectors.toList());
    }

    private static Optional<ProcessHandle.Info> findIdeCommandInfo() {
        return findIdeCommand(getProcessTree());
    }

    static Optional<ProcessHandle.Info> findIdeCommand(
            List<ProcessHandle.Info> processes) {
        for (ProcessHandle.Info info : processes) {
            if (isIdea(info) || isVSCode(info) || isEclipse(info)) {
                return Optional.of(info);
            }
        }
        return Optional.empty();
    }

    private static String getCommandAndArguments(ProcessHandle.Info info) {
        return info.commandLine().orElse(null);
    }

    private static List<ProcessHandle> getParentProcesses() {
        List<ProcessHandle> proceses = new ArrayList<>();
        ProcessHandle p = ProcessHandle.current();
        while (p != null) {
            proceses.add(p);
            p = p.parent().orElse(null);
        }
        return proceses;
    }

    static boolean isEclipse(ProcessHandle.Info info) {
        Optional<String> cmd = info.command();
        if (cmd.isPresent()) {
            String lowerCmd = cmd.get().toLowerCase(Locale.ENGLISH);
            // Eclipse has a lot of other products like Temurin and Adoptium so
            // we cannot check with "contains"
            return lowerCmd.endsWith("eclipse")
                    || lowerCmd.endsWith("eclipse.exe");
        }

        return false;
    }

    static boolean isIdea(ProcessHandle.Info info) {
        return getIdeaBinary(info) != null;
    }

    private static String getIdeaBinary(ProcessHandle.Info info) {
        String commandAndArguments = getCommandAndArguments(info);
        if (commandAndArguments != null
                && commandAndArguments.contains("idea_rt.jar")) {
            String replaced = commandAndArguments
                    .replaceFirst(".*[:;]([^:;]*)(idea_rt.jar).*", "$1$2");
            if (!replaced.equals(commandAndArguments)) {
                File binFolder = new File(
                        new File(replaced).getParentFile().getParentFile(),
                        "bin");
                Optional<File> bin = Stream.of("idea", "idea.sh", "idea.bat")
                        .map(binName -> new File(binFolder, binName))
                        .filter(File::exists).findFirst();
                if (bin.isPresent()) {
                    return bin.get().getAbsolutePath();
                }
            }
        }
        return info.command().filter(cmd -> cmd.contains("idea")).orElse(null);
    }

    static boolean isVSCode(ProcessHandle.Info info) {
        String termProgram = System.getenv("TERM_PROGRAM");
        if ("vscode".equalsIgnoreCase(termProgram)) {
            return true;
        }

        String cmd = getCommandAndArguments(info);
        if (cmd != null) {
            String cmdLower = cmd.toLowerCase(Locale.ENGLISH);
            if (cmdLower.contains("vscode") || cmdLower.contains("vs code")
                    || cmdLower.contains("code helper")
                    || cmdLower.contains("visual studio code")) {
                return true;
            }
        }

        return false;
    }

}
