package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.nodeTypes.NodeWithOptionalScope;
import com.github.javaparser.ast.nodeTypes.NodeWithRange;
import com.github.javaparser.ast.nodeTypes.NodeWithSimpleName;

/**
 * Represents the expression where a component has been attached to its
 * parent. This expression can be a method call expression i.e.:
 * add(component) or an object creation expression i.e.: new
 * VerticalLayout(component). To keep compatibility and usability this
 * wrapper exposes the expression as the following proxies related
 * interfaces:
 * <ul>
 * <li>{@link NodeWithArguments} if the expression is a method call
 * expression</li>
 * <li>{@link NodeWithRange} if the expression is an object creation
 * expression</li>
 * <li>{@link NodeWithSimpleName} if the expression is a method call
 * expression</li>
 * <li>{@link NodeWithOptionalScope} if the expression is a method call
 * expression</li>
 * <li>{@link Node} for the expression itself</li>
 * <li>{@link Expression} for the expression itself</li>
 * </ul>
 */
public record AttachExpression(Expression expression) {

    public NodeWithArguments<Expression> getNodeWithArguments() {
        if (expression.isMethodCallExpr()
                || expression.isObjectCreationExpr()) {
            return (NodeWithArguments) expression;
        }
        return null;
    }

    public NodeWithRange<Node> getNodeWithRange() {
        if (expression.isMethodCallExpr()
                || expression.isObjectCreationExpr()) {
            return expression;
        }
        return null;
    }

    public NodeWithSimpleName<Expression> getNodeWithSimpleName() {
        if (expression.isMethodCallExpr()) {
            return (NodeWithSimpleName) expression;
        } else if (expression.isObjectCreationExpr()) {
            return null;
        }
        return null;
    }

    public NodeWithOptionalScope<Expression> getNodeWithOptionalScope() {
        if (expression.isMethodCallExpr()
                || expression.isObjectCreationExpr()) {
            return (NodeWithOptionalScope) expression;
        }
        return null;
    }

    public Node getNode() {
        return expression;
    }

    public MethodCallExpr getMethodCallExpression() {
        if (expression().isMethodCallExpr()) {
            return (MethodCallExpr) expression;
        }
        return null;
    }

    public ObjectCreationExpr getObjectCreationExpression() {
        if (expression().isObjectCreationExpr()) {
            return (ObjectCreationExpr) expression;
        }
        return null;
    }
}
