package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.printer.lexicalpreservation.LexicalPreservingPrinter;
import com.github.javaparser.symbolsolver.JavaSymbolSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.CombinedTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ReflectionTypeSolver;
import com.vaadin.flow.shared.util.SharedUtil;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;

import static com.vaadin.copilot.javarewriter.DataEntityRecordRewriter.DEFAULT_ENTITY_RECORD_NAME;

/**
 * Rewrites Java source code to add or replace constructor parameters, method
 * invocations and more.
 */
public class JavaRewriter {

    private final String source;

    /**
     * A code snippet to be inserted into the source code.
     *
     * @param code
     *            the code snippet
     */
    public record Code(String code) {

    }

    /**
     * Holder for a setter name and associated value
     */
    public record SetterAndValue(String setter, Object value) {
    }

    /**
     * Information about extracting an inline variable to local variable
     */
    public record ExtractInlineVariableResult(BlockStmt blockStmt,
            String newVariableName, int index) {
    }

    /**
     * The result of a duplicate operation
     *
     * @param nameMapping
     *            a map from old component name to new component name
     * @param childAddCalls
     *            a list of add calls from the parent to ddd children
     * @param variableDeclaration
     *            the variable declaration of the new component when it is
     *            declared as local variable to manage children attachment in
     *            constructor
     * @param assignExpr
     *            the assign expression of the new component when it is declared
     *            as field to manage children attachment in constructor
     */
    public record DuplicateInfo(Map<String, String> nameMapping,
            List<MethodCallExpr> childAddCalls,
            VariableDeclarationExpr variableDeclaration,
            AssignExpr assignExpr) {

    }

    /**
     * Where to add a component
     */
    public enum Where {
        BEFORE, APPEND
    }

    public enum AlignmentMode {
        ALIGN_ITEMS, SELF_HORIZONTALLY, SELF_VERTICALLY
    }

    private final ParserConfiguration parserConfiguration = new ParserConfiguration();
    protected CompilationUnit compilationUnit;

    /**
     * Creates a new JavaRewriter instance.
     *
     * @param source
     *            the Java source code to rewrite
     */
    public JavaRewriter(String source) {
        this.source = source;
        parseSource(source);
    }

    private void parseSource(String source) {
        parserConfiguration
                .setLanguageLevel(ParserConfiguration.LanguageLevel.JAVA_17);
        CombinedTypeSolver combinedTypeSolver = new CombinedTypeSolver();
        combinedTypeSolver.add(new ReflectionTypeSolver(false));
        JavaSymbolSolver symbolSolver = new JavaSymbolSolver(
                combinedTypeSolver);
        parserConfiguration.setSymbolResolver(symbolSolver);

        StaticJavaParser.setConfiguration(parserConfiguration);
        this.compilationUnit = LexicalPreservingPrinter
                .setup(StaticJavaParser.parse(source));
    }

    public CompilationUnit getCompilationUnit() {
        return compilationUnit;
    }

    public String getResult() {
        // Added hack to remove DELETE_THIS from the result for empty
        // parametrized classOrInterfaceDeclaration
        return LexicalPreservingPrinter.print(compilationUnit)
                .replace("DELETE_THIS", "");
    }

    public int getFirstModifiedRow() {
        int row = 1;
        for (int i = 0; i < source.length(); i++) {
            if (source.charAt(i) != getResult().charAt(i)) {
                return row;
            }
            if (source.charAt(i) == '\n') {
                row++;
            }
        }
        return -1;
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) or a function call in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to replace or add, if the constructor
     *            parameter is not found
     * @param value
     *            the new value for the constructor parameter or function call
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    public boolean replaceFunctionCall(ComponentInfo componentInfo,
            String function, Object value) {
        if (replaceConstructorParam(componentInfo, function, value)) {
            return true;
        }
        return replaceOrAddCall(componentInfo, function, value);
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the setter function
     * @param value
     *            the new value for the constructor parameter
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    private boolean replaceConstructorParam(ComponentInfo componentInfo,
            String function, Object value) {
        ObjectCreationExpr objectCreationExpr = componentInfo
                .objectCreationExpr();

        // Find constructor based on number of arguments
        Optional<Constructor<?>> constructor = JavaRewriterUtil
                .findConstructor(componentInfo.type(), objectCreationExpr);
        Optional<Map<Integer, String>> mappings = constructor
                .map(c -> ConstructorAnalyzer.get().getMappings(c));
        if (mappings.isPresent()) {
            Optional<Map.Entry<Integer, String>> mapping = mappings.get()
                    .entrySet().stream()
                    .filter(entry -> entry.getValue().equals(function))
                    .findFirst();
            if (mapping.isPresent()) {
                objectCreationExpr.setArgument(mapping.get().getKey(),
                        JavaRewriterUtil.toExpression(value));
                return true;
            }
        }
        return false;
    }

    /**
     * Adds a function call to the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to add
     * @param parameters
     *            parameters for the function
     * @return {@code true} if the addition was successful, {@code false}
     *         otherwise
     */
    public boolean addCall(ComponentInfo componentInfo, String function,
            Object... parameters) {
        return doReplaceOrAddCall(componentInfo, function, false, parameters);
    }

    /**
     * Replaces a function call in the source code, if found, otherwise adds the
     * function call.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function call to add or replace
     * @param parameters
     *            new parameters for the function
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    public boolean replaceOrAddCall(ComponentInfo componentInfo,
            String function, Object... parameters) {
        return doReplaceOrAddCall(componentInfo, function, true, parameters);
    }

    private boolean doReplaceOrAddCall(ComponentInfo componentInfo,
            String function, boolean replace, Object... parameters) {
        List<Expression> parameterExpressions = new ArrayList<>();
        for (Object parameter : parameters) {
            parameterExpressions.add(JavaRewriterUtil.toExpression(parameter));
        }

        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(componentInfo);

        MethodCallExpr functionCall = functionCalls.stream()
                .filter(m -> m.getNameAsString().equals(function)).findFirst()
                .orElse(null);
        if (replace && functionCall != null) {
            if (parameterExpressions.size() == 1) {
                functionCall.setArgument(0, parameterExpressions.get(0));
            }

            return true;
        }

        JavaRewriterUtil.addFunctionCall(componentInfo, function,
                parameterExpressions);
        return true;
    }

    /**
     * Gets the (active) value of a property of a component.
     * <p>
     * The property value is determined by looking for a setter method call in
     * the source code. If the property is not set using a setter, the
     * constructor is checked.
     * <p>
     * If the property is not set using a setter or in the constructor,
     * {@code null} is returned.
     * <p>
     * If the property is set using a method call, the method call expression is
     * returned.
     *
     * @param componentInfo
     *            the component to get the property value from
     * @param property
     *            the property name
     * @return the property value, or null if the property is not set
     */
    public Object getPropertyValue(ComponentInfo componentInfo,
            String property) {
        String setterName = JavaRewriterUtil.getSetterName(property,
                componentInfo.type(), false);
        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(componentInfo);
        List<MethodCallExpr> candidates = functionCalls.stream()
                .filter(m -> m.getNameAsString().equals(setterName)).toList();
        if (!candidates.isEmpty()) {
            // If there would be multiple calls, the last one is the effective
            // one
            MethodCallExpr setterCall = candidates.get(candidates.size() - 1);
            Expression arg = setterCall.getArguments().get(0);
            return JavaRewriterUtil.fromExpression(arg,
                    setterCall.resolve().getParam(0).getType());
        }

        // Try constructor
        ObjectCreationExpr createExpression = componentInfo
                .objectCreationExpr();
        if (createExpression != null) {
            Optional<Constructor<?>> maybeConstructor = JavaRewriterUtil
                    .findConstructor(componentInfo.type(), createExpression);
            if (maybeConstructor.isPresent()) {
                Constructor<?> constructor = maybeConstructor.get();
                for (int i = 0; i < constructor.getParameterCount(); i++) {
                    String mappedProperty = JavaRewriterUtil
                            .getMappedProperty(constructor, i);
                    if (setterName.equals(mappedProperty)) {
                        return JavaRewriterUtil.fromExpression(
                                createExpression.getArgument(i),
                                createExpression.resolve().getParam(i)
                                        .getType());
                    }
                }
            }
        }
        return null;
    }

    /**
     * Gets the (active) styles of a component.
     *
     * @param componentInfo
     *            the component to get the styles of
     * @return the styles, as a list of style names and values
     */
    public List<JavaStyleRewriter.StyleInfo> getStyles(
            ComponentInfo componentInfo) {
        return JavaStyleRewriter.getStyles(componentInfo);
    }

    /**
     * Sets the given inline style on the given component, replacing an existing
     * style property if present.
     *
     * @param component
     *            the component to set the style on
     * @param property
     *            the style property to set
     * @param value
     *            the style value to set or null to remove the style
     */
    public void setStyle(ComponentInfo component, String property,
            String value) {
        JavaStyleRewriter.setStyle(component, property, value);
    }

    /**
     * Finds a component in the source code.
     * <p>
     * Note that this will create a constructor if the component is a route
     * class and there are no constructors.
     *
     * @param typeAndSourceLocation
     *            the type and source location of the component
     * @return the component info
     */
    public ComponentInfo findComponentInfo(
            ComponentTypeAndSourceLocation typeAndSourceLocation) {
        return ComponentInfo.find(typeAndSourceLocation, this);
    }

    /**
     * Deletes a component from the source code.
     *
     * @param componentInfo
     *            the component to delete
     * @return {@code true} if the deletion was successful, {@code false}
     *         otherwise
     */
    public boolean delete(ComponentInfo componentInfo) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(componentInfo);
        List<MethodCallExpr> otherMethodCalls = JavaRewriterUtil
                .findMethodCallNonStatements(componentInfo);

        List<Expression> parameterUsages = JavaRewriterUtil
                .findParameterUsage(componentInfo);

        Optional<Range> attachRange = componentInfo.attachCall().getNode()
                .getRange();
        if (attachRange.isPresent()) {
            parameterUsages = parameterUsages.stream()
                    .filter(parameter -> parameter.getRange().isPresent()
                            && !parameter.getRange().get()
                                    .overlapsWith(attachRange.get()))
                    .toList();
        }

        parameterUsages.forEach(expr -> {
            if (expr.getParentNode().isPresent()
                    && ((Expression) expr.getParentNode().get())
                            .isMethodCallExpr()) {
                MethodCallExpr methodCallExpr = (MethodCallExpr) expr
                        .getParentNode().get();
                Optional<String> classNameOpt = JavaRewriterUtil
                        .findAncestor(methodCallExpr,
                                ClassOrInterfaceDeclaration.class)
                        .getFullyQualifiedName();
                if (classNameOpt.isEmpty()) {
                    throw new IllegalArgumentException(
                            "Could not find source class");
                }
                int argumentPosition = methodCallExpr.getArgumentPosition(expr);
                boolean arrayArgument = JavaRewriterUtil.isArrayArgument(
                        classNameOpt.get(), methodCallExpr.getNameAsString(),
                        argumentPosition);
                if (arrayArgument) {
                    JavaRewriterUtil.removeArgumentCalls(methodCallExpr,
                            List.of(expr), true);
                } else if (methodCallExpr.getArguments().size() == 1) {
                    JavaRewriterUtil.removeStatement(methodCallExpr);
                } else {
                    throw new IllegalArgumentException("Cannot handle "
                            + methodCallExpr.getNameAsString() + " method");
                }
            }

        });

        functionCalls.forEach(expr -> JavaRewriterUtil
                .findAncestorOrThrow(expr, Statement.class).remove());
        if (componentInfo.fieldDeclaration() != null) {
            componentInfo.fieldDeclaration().remove();
            if (componentInfo.fieldDeclarationAndAssignment() == null) {
                // Instance created elsewhere, i.e.
                // TextField foo;
                // ...
                // foo = new TextField();
                JavaRewriterUtil
                        .removeStatement(componentInfo.objectCreationExpr());
            }
        } else if (componentInfo.localVariableDeclarator() != null) {
            JavaRewriterUtil
                    .removeStatement(componentInfo.localVariableDeclarator());

        }

        otherMethodCalls.forEach(methodCall -> {
            if (!JavaRewriterUtil.removeFromStringConcatenation(methodCall)) {
                JavaRewriterUtil.removeStatement(methodCall);
            }
        });

        removeAttachCall(componentInfo);

        return true;
    }

    private Optional<Expression> removeAttachCall(ComponentInfo componentInfo) {
        Optional<Expression> addArgument = JavaRewriterUtil
                .getAttachArgument(componentInfo);
        addArgument.ifPresent(Expression::remove);
        if (componentInfo.attachCall() != null && componentInfo.attachCall()
                .getNodeWithArguments().getArguments().isEmpty()) {
            JavaRewriterUtil
                    .removeStatement(componentInfo.attachCall().getNode());
        }
        return addArgument;
    }

    /**
     * Moves a component in the source code.
     *
     * @param component
     *            the component to move
     * @param container
     *            the new container for the component, if where is Where.APPEND.
     * @param reference
     *            the reference component to move the component before, if where
     *            is Where.BEFORE.
     * @param where
     *            where to move the component
     */
    public void moveComponent(ComponentInfo component, ComponentInfo container,
            ComponentInfo reference, Where where) {
        if (container == null) {
            throw new IllegalArgumentException(
                    "Container component must be non-null");
        }
        if (component.equals(container) || component.equals(reference)
                || container.equals(reference)) {
            throw new IllegalArgumentException(
                    "Component, container and reference must be different");
        }
        removeAttachCall(component);

        List<MethodCallExpr> containerFunctionCalls = JavaRewriterUtil
                .findMethodCallStatements(container);
        String componentFieldOrVariableName = JavaRewriterUtil
                .getFieldOrVariableName(component);
        // For inline we add the full expression, otherwise the name reference
        var toAdd = componentFieldOrVariableName == null
                ? component.objectCreationExpr()
                : new NameExpr(componentFieldOrVariableName);

        if (where == Where.APPEND) {
            if (reference != null) {
                throw new IllegalArgumentException(
                        "Reference component must be null when appending");
            }

            // Find the last add statement for the container and then add the
            // component after that
            MethodCallExpr added = JavaRewriterUtil.addAfterLastFunctionCall(
                    containerFunctionCalls, "add", toAdd);
            if (added == null) {
                throw new IllegalArgumentException(
                        "No add call found for the container component");
            }
        } else if (where == Where.BEFORE) {
            if (reference == null) {
                throw new IllegalArgumentException(
                        "Reference component must be non-null when moving before");
            }

            Optional<Expression> referenceAddArgument = JavaRewriterUtil
                    .findReference(reference.attachCall().getNodeWithArguments()
                            .getArguments(), reference);

            if (referenceAddArgument.isPresent()) {
                int refAddIndex = reference.attachCall().getNodeWithArguments()
                        .getArguments().indexOf(referenceAddArgument.get());
                reference.attachCall().getNodeWithArguments().getArguments()
                        .add(refAddIndex, toAdd);

                // We can now end up with having "add" before the component
                // constructor and setters, e.g.

                // Input input1 = new Input();
                // add(input3, input1);
                // Input input2 = new Input();
                // add(input2);
                // Input input3 = new Input();

                // We move the add call down after all functions
                // BUT there can also be add calls between the current
                // add location and where it should end up.
                // If so, they need to be moved also
                // We then get

                // Input input1 = new Input();
                // Input input2 = new Input();
                // Input input3 = new Input();
                // add(input3, input1);
                // add(input2);

                List<MethodCallExpr> componentFunctionCalls = JavaRewriterUtil
                        .findMethodCallStatements(component);
                if (!componentFunctionCalls.isEmpty()) {
                    MethodCallExpr lastCall = componentFunctionCalls
                            .get(componentFunctionCalls.size() - 1);
                    BlockStmt insertBlock = JavaRewriterUtil
                            .findAncestorOrThrow(lastCall, BlockStmt.class);
                    int finalAddLocation = JavaRewriterUtil
                            .findBlockStatementIndex(lastCall) + 1;

                    int attachCallIndex = JavaRewriterUtil
                            .findBlockStatementIndex(
                                    reference.attachCall().getNode());

                    List<MethodCallExpr> allAddCalls = containerFunctionCalls
                            .stream()
                            .filter(m -> m.getNameAsString()
                                    .equals(reference.attachCall()
                                            .getNodeWithSimpleName().getName()
                                            .asString()))
                            .toList();
                    for (MethodCallExpr addCall : allAddCalls) {
                        int addCallIndex = JavaRewriterUtil
                                .findBlockStatementIndex(addCall);
                        if (addCallIndex >= attachCallIndex
                                && addCallIndex < finalAddLocation) {
                            Statement statement = JavaRewriterUtil
                                    .findAncestorOrThrow(addCall,
                                            Statement.class);
                            statement.remove();
                            insertBlock.addStatement(finalAddLocation - 1,
                                    statement);
                        }
                    }
                }

            } else {
                throw new IllegalArgumentException(
                        "Reference component not found in the add call");
            }
        } else {
            throw new IllegalArgumentException("Unknown where: " + where);
        }
    }

    public void duplicate(ComponentInfo component) {
        duplicate(component, true);
    }

    /**
     * Duplicates a component in the source code.
     *
     * @param component
     *            the component to duplicate
     * @param handleAdd
     *            true to automatically add the new component next to the old
     *            one, false to handle add calls like any other method call
     * @return a map from old component name to new component name
     */

    public DuplicateInfo duplicate(ComponentInfo component, boolean handleAdd) {
        if (component.routeConstructor() != null) {
            throw new IllegalArgumentException(
                    "Cannot duplicate a route class");
        }

        // Insert new component after the original component
        InsertionPoint insertionPoint = findInsertionPointForAppend(component);
        String oldName;
        List<MethodCallExpr> childAddCalls = new ArrayList<>();
        VariableDeclarationExpr variableDeclaration = null;
        AssignExpr assignExpr = null;

        String duplicatedName = null;
        if (component.localVariableName() != null) {
            oldName = component.localVariableName();
            duplicatedName = JavaRewriterUtil.findFreeVariableName(
                    component.localVariableName(), insertionPoint.getBlock());
            VariableDeclarator newLocalVariable = JavaRewriterUtil
                    .clone(component.localVariableDeclarator());
            newLocalVariable.setName(duplicatedName);
            variableDeclaration = new VariableDeclarationExpr(newLocalVariable);
            JavaRewriterUtil.appendExpressionAsNextSiblingInBlockAncestor(
                    component.localVariableDeclarator(),
                    new ExpressionStmt(variableDeclaration));
        } else if (component.fieldName() != null) {
            oldName = component.fieldName();
            duplicatedName = insertionPoint
                    .getFreeVariableName(component.fieldName());
            FieldDeclaration newField;
            if (component.fieldDeclarationAndAssignment() != null) {
                newField = JavaRewriterUtil
                        .clone(component.fieldDeclarationAndAssignment());
                if (newField.getVariables().isNonEmpty()) {
                    variableDeclaration = new VariableDeclarationExpr(
                            newField.getVariable(0));
                }
            } else {
                newField = JavaRewriterUtil.clone(component.fieldDeclaration());
                if (newField.getVariables().size() > 1) {
                    newField.getVariables()
                            .removeIf(variableDeclarator -> !variableDeclarator
                                    .getNameAsString().equals(oldName));
                }
                // Assignment, e.g. button = new Button();
                assignExpr = JavaRewriterUtil
                        .clone(component.assignmentExpression());
                assignExpr.setTarget(new NameExpr(duplicatedName));
                JavaRewriterUtil.appendExpressionAsNextSiblingInBlockAncestor(
                        component.assignmentExpression(),
                        new ExpressionStmt(assignExpr));
            }
            newField.getVariable(0).setName(duplicatedName);
            JavaRewriterUtil.addFieldAfter(newField,
                    component.fieldDeclaration());
        } else {
            // Inline
            oldName = null;
        }

        // Duplicate method calls
        if (duplicatedName != null) {
            List<MethodCallExpr> calls = JavaRewriterUtil
                    .findMethodCallStatements(component);
            for (MethodCallExpr call : calls) {
                MethodCallExpr newCall = JavaRewriterUtil.clone(call);
                JavaRewriterUtil.setNameExprScope(newCall,
                        new NameExpr(duplicatedName));
                insertionPoint.add(new ExpressionStmt(newCall));
                if (newCall.getNameAsString().equals("add")) {
                    childAddCalls.add(newCall);
                }
            }

            List<Expression> parameterUsages = JavaRewriterUtil
                    .findParameterUsage(component);
            // excluding attach call
            Optional<Range> attachCallRangeOptional = component.attachCall()
                    .getNodeWithRange().getRange();
            if (attachCallRangeOptional.isPresent()) {
                Range attachRange = attachCallRangeOptional.get();
                parameterUsages = parameterUsages.stream()
                        .filter(parameter -> parameter.getRange().isPresent()
                                && !parameter.getRange().get()
                                        .overlapsWith(attachRange))
                        .toList();
            }
            for (Expression parameterUsage : parameterUsages) {
                MethodCallExpr methodCallExpr = JavaRewriterUtil
                        .findAncestorOrThrow(parameterUsage,
                                MethodCallExpr.class);
                int argumentPosition = methodCallExpr
                        .getArgumentPosition(parameterUsage);
                boolean addAsArg = false;
                Optional<Expression> scope = JavaRewriterUtil
                        .getScopeIgnoreComposite(component, methodCallExpr);
                if (scope.isEmpty() || scope.get().isThisExpr()) {
                    Optional<String> classNameOpt = JavaRewriterUtil
                            .findAncestor(parameterUsage,
                                    ClassOrInterfaceDeclaration.class)
                            .getFullyQualifiedName();
                    if (classNameOpt.isEmpty()) {
                        throw new IllegalArgumentException(
                                "Could not find source class");
                    }
                    addAsArg = JavaRewriterUtil.isArrayArgument(
                            classNameOpt.get(),
                            methodCallExpr.getNameAsString(), argumentPosition);
                }
                if (addAsArg) {
                    methodCallExpr.getArguments().add(argumentPosition + 1,
                            new NameExpr(duplicatedName));
                } else {
                    MethodCallExpr clone = JavaRewriterUtil
                            .clone(methodCallExpr);
                    clone.setArgument(argumentPosition,
                            new NameExpr(duplicatedName));
                    insertionPoint.add(new ExpressionStmt(clone));
                }

            }

        }
        // Attach the new component
        if (handleAdd) {
            Expression componentAddArgument = JavaRewriterUtil
                    .getAttachArgumentOrThrow(component);

            int addIndex;
            NodeWithArguments<?> expressionToAdd;
            Optional<Expression> parentNode = componentAddArgument
                    .getParentNode().map(Expression.class::cast);
            if (parentNode.isPresent()
                    && parentNode.get().isObjectCreationExpr()) {
                ObjectCreationExpr objectCreationExpr = parentNode.get()
                        .asObjectCreationExpr();
                addIndex = objectCreationExpr.getArguments()
                        .indexOf(componentAddArgument) + 1;
                expressionToAdd = objectCreationExpr;
            } else {
                addIndex = component.attachCall().getNodeWithArguments()
                        .getArgumentPosition(componentAddArgument) + 1;
                expressionToAdd = component.attachCall().getNodeWithArguments();
            }
            if (duplicatedName != null) {
                expressionToAdd.getArguments().add(addIndex,
                        new NameExpr(duplicatedName));
            } else {
                expressionToAdd.getArguments().add(addIndex,
                        JavaRewriterUtil.clone(component.objectCreationExpr()));
            }
        }

        Map<String, String> nameMappings;
        if (oldName != null) {
            nameMappings = Collections.singletonMap(oldName, duplicatedName);
        } else {
            nameMappings = Collections.emptyMap();
        }

        return new DuplicateInfo(nameMappings, childAddCalls,
                variableDeclaration, assignExpr);
    }

    /**
     * Adds the given code snippet to the source code either before the
     * reference component (Where.BEFORE) or by appending to the layout
     * (Where.APPEND).
     *
     * @param referenceComponent
     *            the reference component (BEFORE) or container (APPEND) to add
     *            the code
     * @param where
     *            where to add the code
     * @param template
     *            the code to add, as JSON array of objects with "tag", "props"
     *            and "children"
     */
    public void addComponentUsingTemplate(ComponentInfo referenceComponent,
            Where where, List<JavaComponent> template) {

        if (where == Where.APPEND) {
            //
            // HorizontalLayout layout = new HorizontalLayout(button);
            // layout.setThis();
            // layout.addThat();
            // ...
            // add(layout);
            InsertionPoint insertionPoint = findInsertionPointForAppend(
                    referenceComponent);
            if (referenceComponent.routeConstructor() != null) {
                createComponentStatements(insertionPoint, null, template,
                        "this", null);
            } else {
                createComponentStatements(
                        insertionPoint, null, template, JavaRewriterUtil
                                .getFieldOrVariableName(referenceComponent),
                        null);
            }

        } else if (where == Where.BEFORE) {
            // HorizontalLayout hl = new HorizontalLayout();
            // button = new Button("Hello");
            // button.setThis();
            // button.addThat();
            // hl.add(button);
            //
            // Insert constructor + setters before the reference constructor and
            // add the component so that it is added right before the reference
            BlockStmt insertBlock = referenceComponent.componentCreateScope();
            int insertIndex = JavaRewriterUtil.findBlockStatementIndex(
                    referenceComponent.objectCreationExpr());
            InsertionPoint insertionPoint = new InsertionPoint(insertBlock,
                    insertIndex);
            createComponentStatements(insertionPoint, null, template,
                    JavaRewriterUtil.getFieldOrVariableName(referenceComponent),
                    referenceComponent);

        } else {
            throw new IllegalArgumentException("Unknown where: " + where);
        }

    }

    private InsertionPoint findInsertionPointForAppend(
            ComponentInfo component) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(component);
        if (!functionCalls.isEmpty()) {
            // There are some component.setThis() or component.somethingElse()
            // calls, add a new component after those
            MethodCallExpr lastCall = functionCalls
                    .get(functionCalls.size() - 1);
            return JavaRewriterUtil.findLocationAfter(lastCall);
        } else if (component.routeConstructor() != null) {
            return JavaRewriterUtil
                    .findLocationAtEnd(component.routeConstructor().getBody());
        }

        // There are no component.anything() calls, add before the
        // add(component) call
        return JavaRewriterUtil
                .findLocationBefore(component.attachCall().expression());
    }

    public void setAlignment(ComponentInfo component,
            AlignmentMode alignmentMode, boolean selected,
            List<String> lumoClasses) {
        String[] lumoUtilityInnerClassNames = alignmentMode == JavaRewriter.AlignmentMode.ALIGN_ITEMS
                ? new String[] { "AlignItems", "JustifyContent" }
                : new String[] { "AlignSelf" };
        LumoRewriterUtil.removeClassNameArgs(component,
                lumoUtilityInnerClassNames);
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);
        if (selected) {
            List<Expression> parameterToAddList = new ArrayList<>();
            for (String lumoUtilityInnerClassName : lumoUtilityInnerClassNames) {
                parameterToAddList
                        .addAll(LumoRewriterUtil.getLumoMethodArgExpressions(
                                lumoUtilityInnerClassName, lumoClasses));
            }
            boolean added = LumoRewriterUtil.addClassNameWithArgs(component,
                    parameterToAddList);
            if (!added) {
                replaceOrAddCall(component, "addClassNames",
                        parameterToAddList.toArray(new Object[0]));
            }
        }
    }

    /**
     * Sets gap to selected component
     *
     * @param component
     *            component to set gap
     * @param newValue
     *            lumo utility gap variable literal.
     */
    public void setGap(ComponentInfo component, String newValue) {
        LumoRewriterUtil.removeClassNameArgs(component, "Gap", "Gap.Column",
                "Gap.Row");
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil
                .findMethodCallStatements(component);
        methodCallStatements.stream()
                .filter(methodCallExpr -> StringUtils
                        .equals(methodCallExpr.getNameAsString(), "setSpacing"))
                .findAny().ifPresent(JavaRewriterUtil::removeStatement);
        LumoRewriterUtil.removeThemeArgStartsWith(methodCallStatements,
                "spacing");

        if (StringUtils.isNotEmpty(newValue)) {
            List<Expression> gapExpressions = LumoRewriterUtil
                    .getLumoMethodArgExpressions("Gap", List.of(newValue));
            boolean added = LumoRewriterUtil.addClassNameWithArgs(component,
                    gapExpressions);
            if (!added) {
                replaceOrAddCall(component, "addClassNames",
                        gapExpressions.toArray(new Object[0]));
            }
        } else {
            MethodCallExpr call = JavaRewriterUtil.addAfterLastFunctionCall(
                    methodCallStatements, "setSpacing",
                    new BooleanLiteralExpr(false));
            if (call == null) {
                replaceOrAddCall(component, "setSpacing", false);
            }

        }
    }

    /**
     * Merges all the components and wraps them using the given component and
     * places the result in place of the first component.
     *
     * @param components
     *            The components to merge. The first component will be replaced
     *            with the wrapper component
     * @param wrapperComponent
     *            The component to wrap the merged components in.
     */
    public void mergeAndReplace(List<ComponentInfo> components,
            JavaComponent wrapperComponent) {
        ComponentInfo firstComponent = components.get(0);
        if (firstComponent.componentAttachScope() == null) {
            throw new IllegalArgumentException("Routes cannot be wrapped");
        }
        for (int i = 1; i < components.size(); i++) {
            ComponentInfo component = components.get(i);
            if (component.componentAttachScope() != firstComponent
                    .componentAttachScope()) {
                throw new IllegalArgumentException(
                        "Only components attached in the same block are currently supported");
            }
        }

        // Add wrapper init code where the first component is attached
        int wrapperInsertIndex = JavaRewriterUtil
                .findBlockStatementIndex(firstComponent.attachCall().getNode());
        InsertionPoint firstComponentAttachLocation = new InsertionPoint(
                firstComponent.componentAttachScope(), wrapperInsertIndex);

        List<VariableDeclarator> statements = createComponentStatements(
                firstComponentAttachLocation, null, wrapperComponent, false,
                null, null);

        // Remove all attach calls for the components and attach them to the
        // wrapper instead. Replace the first component add with an add for the
        // wrapper
        NameExpr wrapperVariable = new NameExpr(
                statements.get(0).getNameAsString());
        MethodCallExpr addCall = new MethodCallExpr(wrapperVariable, "add");

        Node wrapperAdd = null;
        for (int i = 0; i < components.size(); i++) {
            ComponentInfo component = components.get(i);
            Optional<Expression> componentAddArgument;
            if (i == 0) {
                componentAddArgument = Optional.of(
                        JavaRewriterUtil.getAttachArgumentOrThrow(component));
                // Attach wrapper in place of the first component
                Optional<Node> maybeParent = componentAddArgument
                        .flatMap(Node::getParentNode);
                if (maybeParent.isEmpty()) {
                    throw new IllegalArgumentException(
                            "No add argument found for the first component");
                }

                wrapperAdd = maybeParent.get();
                wrapperAdd.replace(componentAddArgument.get(), wrapperVariable);
            } else {
                componentAddArgument = removeAttachCall(component);
            }
            componentAddArgument.ifPresent(addCall::addArgument);
        }
        // Add the wrapper.add(components) call
        if (wrapperAdd == null) {
            throw new IllegalArgumentException("Wrapper was never added");
        }
        firstComponentAttachLocation.getBlock().addStatement(
                JavaRewriterUtil.findBlockStatementIndex(wrapperAdd), addCall);

    }

    /**
     * Replaces a parameter name in a method call.
     *
     * @param call
     * @param oldVariableName
     * @param newVariableName
     */
    public void replaceCallParameter(NodeWithArguments<?> call,
            String oldVariableName, String newVariableName) {
        call.getArguments().stream().forEach(arg -> {
            if (arg.isNameExpr() && arg.asNameExpr().getNameAsString()
                    .equals(oldVariableName)) {
                arg.asNameExpr().setName(newVariableName);
            }
        });
    }

    private void createComponentStatements(InsertionPoint insertionPoint,
            JavaComponent parent, List<JavaComponent> template,
            String layoutVariableName, ComponentInfo referenceComponent) {
        for (JavaComponent javaComponent : template) {
            createComponentStatements(insertionPoint, parent, javaComponent,
                    true, layoutVariableName, referenceComponent);
        }
    }

    private List<VariableDeclarator> createComponentStatements(
            InsertionPoint insertionPoint, JavaComponent parent,
            JavaComponent javaComponent, boolean attach,
            String layoutVariableName, ComponentInfo referenceComponent) {

        List<VariableDeclarator> createdVariables = new ArrayList<>();

        String componentClassName = javaComponent.className() != null
                ? javaComponent.className()
                : FlowComponentQuirks.getClassForTag(javaComponent.tag());

        ClassOrInterfaceType fullType = StaticJavaParser
                .parseClassOrInterfaceType(componentClassName);

        Map<String, Object> setters = new HashMap<>();
        Class<?> componentType = JavaRewriterUtil
                .getClass(fullType.getNameWithScope());
        javaComponent.props().forEach((prop, value) -> {
            SetterAndValue setterAndValue = JavaRewriterUtil
                    .getSetterAndValue(componentType, prop, value);
            setters.put(setterAndValue.setter(), setterAndValue.value());
        });

        // Import
        JavaRewriterUtil.addImport(compilationUnit,
                fullType.getNameWithScope());
        ClassOrInterfaceType type = fullType.clone().removeScope();
        // we create a second type for instantiation to avoid explicit type and
        // allow to modify separately the ClassOrInterfaceType for declaration
        // and definition
        ClassOrInterfaceType typeForInstantiation = type.clone();

        // Find data entity available name in case of data provider
        String dataEntityRecordName = JavaRewriterUtil.findFreeRecordName(
                DEFAULT_ENTITY_RECORD_NAME,
                JavaRewriterUtil.findAncestor(insertionPoint.getBlock(),
                        ClassOrInterfaceDeclaration.class));

        // Constructor call
        ObjectCreationExpr constructor = JavaRewriterUtil
                .createComponentConstructor(javaComponent, typeForInstantiation,
                        dataEntityRecordName);

        // Removes setter property to be used in constructor
        JavaRewriterUtil
                .getSingleStringParamConstructor(fullType, setters.keySet())
                .ifPresent(constructorProp -> {
                    Object value = setters.remove(constructorProp);
                    getParameterList(insertionPoint, value)
                            .forEach(constructor::addArgument);
                });

        // Gets a valid available variable name for the new component
        String variableName = JavaRewriterUtil
                .getNewComponentName(javaComponent, type, insertionPoint);

        NameExpr variableNameExpr = new NameExpr(variableName);

        VariableDeclarator decl = new VariableDeclarator(type, variableName,
                constructor);
        VariableDeclarationExpr declarationExpr = new VariableDeclarationExpr(
                decl);
        createdVariables.add(decl);
        insertionPoint.add(new ExpressionStmt(declarationExpr));

        // We add data items included as props when the item is set
        // with in a data provider bean like
        // ListDataView::setItems(T... items)
        JavaDataProviderHandler.handleDataStatementsAndClearDataProps(
                compilationUnit, decl, javaComponent, insertionPoint,
                dataEntityRecordName);

        // Properties setters
        for (Map.Entry<String, Object> setter : setters.entrySet()) {
            if (FlowComponentQuirks.skipProps(javaComponent, setter.getKey())) {
                continue;
            }
            Object value = setter.getValue();
            if (value instanceof JavaComponent javaComponentValue) {
                value = createSubComponentStatements(insertionPoint,
                        javaComponent, List.of(javaComponentValue)).get(0);
            }
            insertSetter(insertionPoint, variableNameExpr, setter.getKey(),
                    value);
        }

        // Child components
        // First we remove children that are not meant to be added as components
        // but as method calls
        List<JavaComponent> methodChildren = FlowComponentQuirks
                .getMethodCallChildren(javaComponent);

        javaComponent.children().removeAll(methodChildren);

        // Then we add the children as method calls
        createMethodCallChildrenStatements(insertionPoint, javaComponent,
                methodChildren, variableNameExpr, dataEntityRecordName);

        // Then we add the children as components when items are not
        // added with a data provider but as dedicated types of items like
        // MessageList::setItems(com.vaadin.flow.component.messages.MessageListItem...
        // items )
        createComponentStatements(insertionPoint, javaComponent,
                javaComponent.children(), variableName, null);

        if (attach) {
            attachComponent(insertionPoint, parent, layoutVariableName,
                    referenceComponent, variableNameExpr, variableName);
        }

        return createdVariables;
    }

    private void attachComponent(InsertionPoint insertionPoint,
            JavaComponent parent, String layoutVariableName,
            ComponentInfo referenceComponent, NameExpr variableNameExpr,
            String variableName) {
        // Attach component
        if (referenceComponent != null) {
            AttachExpression referenceAttach = referenceComponent.attachCall();
            if (referenceAttach.getNodeWithArguments().getArguments()
                    .size() == 1) {
                // add(reference)
                ExpressionStmt addCall = createAddCall(parent, referenceAttach
                        .getNodeWithOptionalScope().getScope()
                        .orElse(JavaRewriterUtil.isNodeInCompositeClass(
                                referenceAttach.getNode())
                                        ? new MethodCallExpr("getContent")
                                        : null),
                        variableNameExpr);
                BlockStmt block = JavaRewriterUtil.findAncestorOrThrow(
                        referenceAttach.getNode(), BlockStmt.class);
                block.addStatement(JavaRewriterUtil.findBlockStatementIndex(
                        referenceAttach.getNode()), addCall);
            } else {
                Expression referenceArgument;
                // add(..., reference,...)
                if (layoutVariableName == null) {
                    // Reference is an inline call, like add(new Button()
                    referenceArgument = referenceComponent.objectCreationExpr();
                } else {
                    referenceArgument = referenceAttach.getNodeWithArguments()
                            .getArguments().stream()
                            .filter(arg -> arg instanceof NameExpr nameExpr
                                    && nameExpr.getNameAsString()
                                            .equals(layoutVariableName))
                            .findFirst().orElse(null);
                }
                if (referenceArgument == null) {
                    throw new IllegalArgumentException(
                            "Did not find reference argument ('" + variableName
                                    + "') in " + referenceAttach);
                }
                int index = referenceAttach.getNodeWithArguments()
                        .getArguments().indexOf(referenceArgument);
                referenceAttach.getNodeWithArguments().getArguments().add(index,
                        variableNameExpr);
            }
        } else if (layoutVariableName != null) {
            Expression scope = null;
            if (!layoutVariableName.equals("this")) {
                scope = new NameExpr(layoutVariableName);
            } else if (JavaRewriterUtil
                    .isNodeInCompositeClass(insertionPoint.getBlock())) {
                scope = new MethodCallExpr("getContent");
            }
            ExpressionStmt addCall = createAddCall(parent, scope,
                    variableNameExpr);
            // Just adding to the given layout
            insertionPoint.add(addCall);
        } else {
            throw new IllegalArgumentException(
                    "Either layoutVariableName or referenceAttach must be given to attach a component");
        }
    }

    private NodeList<Expression> getParameterList(InsertionPoint insertionPoint,
            Object value) {
        if (value instanceof List<?> list) {
            if (list.isEmpty()) {
                return new NodeList<>();
            }
            if (list.get(0) instanceof JavaComponent) {
                return JavaRewriterUtil.toExpressionList(
                        createSubComponentStatements(insertionPoint, null,
                                (List<JavaComponent>) list));
            }
        }
        return JavaRewriterUtil.toExpressionList(value);
    }

    private void insertSetter(InsertionPoint insertionPoint, Expression owner,
            String setterName, Object value) {
        if (setterName.equals("setStyle")) {
            insertStyles(insertionPoint, owner, (Map<String, String>) value);
        } else if (setterName.equals("setSlot")) {
            // getElement().setAttribute("slot", value)
            MethodCallExpr getElement = new MethodCallExpr(owner, "getElement");
            MethodCallExpr setAttributeCall = new MethodCallExpr(getElement,
                    "setAttribute",
                    new NodeList<>(JavaRewriterUtil.toExpression("slot"),
                            JavaRewriterUtil.toExpression(value)));
            insertionPoint.add(new ExpressionStmt(setAttributeCall));
        } else {
            NodeList<Expression> parameterExpression = getParameterList(
                    insertionPoint, value);
            MethodCallExpr setterCall = new MethodCallExpr(owner, setterName,
                    parameterExpression);
            insertionPoint.add(new ExpressionStmt(setterCall));
        }
    }

    private void insertStyles(InsertionPoint insertionPoint, Expression owner,
            Map<String, String> styles) {
        // button.getStyle().set("p1","value1").set("p2","value2");
        MethodCallExpr finalCall = new MethodCallExpr(owner, "getStyle");
        for (Map.Entry<String, String> entry : styles.entrySet()) {
            finalCall = new MethodCallExpr(finalCall, "set",
                    new NodeList<>(
                            JavaRewriterUtil.toExpression(entry.getKey()),
                            JavaRewriterUtil.toExpression(entry.getValue())));
        }
        insertionPoint.add(new ExpressionStmt(finalCall));

    }

    private NodeList<Expression> createSubComponentStatements(
            InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> components) {
        List<VariableDeclarator> variables = new ArrayList<>();
        for (JavaComponent javaComponent : components) {
            variables.addAll(createComponentStatements(insertionPoint, parent,
                    javaComponent, false, null, null));
        }
        return new NodeList<>(
                variables.stream().map(VariableDeclarator::getName)
                        .map(NameExpr::new).toArray(NameExpr[]::new));

    }

    private ExpressionStmt createAddCall(JavaComponent parent, Expression scope,
            Expression toAdd) {
        String methodName = "add";
        if (parent != null && parent.tag().equals("SideNav")) {
            methodName = "addItem";
        }
        return new ExpressionStmt(
                new MethodCallExpr(scope, methodName, new NodeList<>(toAdd)));
    }

    /**
     * Returns the source code.
     *
     * @return the source code
     */
    protected String getSource() {
        return source;
    }

    /**
     * This method is used to create method call statements for the children of
     * a component when the mapping from React template to Java code is <br>
     * React children -> Flow method calls.<br>
     * This mapping is quite adhoc per component and should be defined in the
     * {@link FlowComponentQuirks} utility class.
     *
     */
    private void createMethodCallChildrenStatements(
            InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> methodChildren, NameExpr owner,
            String dataEntityRecordName) {
        List<JavaComponent> gridColumns = new ArrayList<>();
        for (JavaComponent child : methodChildren) {
            if (FlowComponentQuirks.isGridColumnDefinition(child)) {
                gridColumns.add(child);
            } else {
                // Get the method call expression for the child component (if
                // any
                MethodCallExpr expression = FlowComponentQuirks
                        .getMethodCallExprFromComponent(child, owner,
                                dataEntityRecordName);
                String setterName = FlowComponentQuirks
                        .getSetterNameFromComponent(child);
                Object value = FlowComponentQuirks.getValueFromComponent(child);
                if (expression != null && setterName != null && value != null) {
                    insertSetter(insertionPoint, expression, setterName, value);
                }
            }
        }
        if (!gridColumns.isEmpty()) {
            MethodCallExpr expression = new MethodCallExpr(owner, "setColumns");
            for (JavaComponent gridColumn : gridColumns) {
                expression.addArgument(new StringLiteralExpr(
                        gridColumn.props().get("path").toString()));
            }
            insertionPoint.add(new ExpressionStmt(expression));
        }
    }
}
