package com.vaadin.copilot;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.Optional;

/**
 * Base class for Copilot Server API clients
 */
public class CopilotServerClient {

    private static final String SERVER_PORT_ENV = "COPILOT_SERVER_PORT";

    protected static final String SERVER_URL_ENV = "copilot.serverBaseUrl";

    private static String serverBaseUrlCache = null;

    private final HttpClient httpClient;

    private final ObjectMapper objectMapper;

    /**
     * Constructor initializing HttpClient and ObjectMapper
     */
    public CopilotServerClient() {
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_2)
                .followRedirects(HttpClient.Redirect.NORMAL).build();
        this.objectMapper = new ObjectMapper();
        this.objectMapper.registerModule(new JavaTimeModule());
        this.objectMapper.configure(
                DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    protected HttpClient getHttpClient() {
        return httpClient;
    }

    protected HttpRequest buildRequest(URI uri, String json) {
        return HttpRequest.newBuilder().uri(uri)
                .POST(HttpRequest.BodyPublishers.ofString(json))
                .header("Content-Type", "application/json; charset=utf-8")
                .timeout(Duration.ofSeconds(120)).build();
    }

    protected String writeAsJsonString(Object obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    protected <T> T readValue(String string, Class<T> clazz) {
        try {
            return objectMapper.readValue(string, clazz);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    protected URI getQueryURI(String path) {
        try {
            return new URI(getServerBaseUrl() + path);
        } catch (URISyntaxException e) {
            throw new IllegalStateException(
                    "Invalid server configuration, server uri is wrong", e);
        }
    }

    protected String getServerBaseUrl() {
        if (serverBaseUrlCache != null) {
            return serverBaseUrlCache;
        }
        serverBaseUrlCache = System.getenv(SERVER_URL_ENV);
        if (serverBaseUrlCache != null) {
            return serverBaseUrlCache;
        }

        if (Copilot.isDevelopmentMode()) {
            // Try a localhost server
            try {
                Integer serverPort = Optional
                        .ofNullable(System.getenv(SERVER_PORT_ENV))
                        .map(Integer::parseInt).orElse(8081);
                String localhostUrl = "http://localhost:%s/v1/"
                        .formatted(serverPort);
                String statusUrl = localhostUrl + "actuator/health";

                HttpRequest request = HttpRequest.newBuilder()
                        .uri(new URI(statusUrl)).timeout(Duration.ofSeconds(1))
                        .build();
                HttpResponse<String> response = httpClient.send(request,
                        HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() == 200) {
                    getLogger().info("Using the local server for development.");
                    return localhostUrl;
                }
            } catch (InterruptedException e) {
                getLogger().warn(
                        "Interrupted while trying to reach the local server.",
                        e);
                Thread.currentThread().interrupt();
            } catch (Exception e) {
                getLogger().warn(
                        "No local server found, as this exception occurred while trying to reach the local server: {}.",
                        e.getClass());
            }
            getLogger().info("Using the staging server for development.");
            serverBaseUrlCache = "https://copilot.stg.vaadin.com/v1/";
            return serverBaseUrlCache;
        } else {
            serverBaseUrlCache = "https://copilot.vaadin.com/v1/";
            return serverBaseUrlCache;
        }
    }

    protected Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

}
