package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

public class LumoRewriterUtil {
    private static final String addClassName = "addClassName";
    private static final String addClassNames = "addClassNames";
    private static final String setClassName = "setClassName";
    private static final List<String> classNameMethods = List.of(addClassName,
            addClassNames, setClassName);

    /**
     * Removes arguments from <code>getThemeList().add()</code> or
     * <code>getThemeList().addAll()</code>. If there is no arguments left after
     * removing, removes the method.
     *
     * @param methodCallStatements
     *            Method calls of the given node
     * @param startsWith
     *            string literal args to be removed
     */
    public static void removeThemeArgStartsWith(
            List<MethodCallExpr> methodCallStatements, String startsWith) {
        List<MethodCallExpr> themeAddCalls = getThemeAddCalls(
                methodCallStatements);
        for (MethodCallExpr addCall : themeAddCalls) {
            List<Expression> argsToRemove = addCall.getArguments().stream()
                    .filter(f -> f.isStringLiteralExpr()
                            && f.asStringLiteralExpr().asString()
                                    .startsWith(startsWith))
                    .collect(Collectors.toList());
            JavaRewriterUtil.removeArgumentCalls(addCall, argsToRemove, true);
        }
    }

    public static void removeSetThemeArgs(
            List<MethodCallExpr> methodCallExprList, String argName) {
        List<MethodCallExpr> list = methodCallExprList.stream()
                .filter(methodCallExpr -> methodCallExpr.getScope().isPresent())
                .filter(methodCallExpr -> methodCallExpr.getScope().get()
                        .toString().contains("getThemeList"))
                .filter(methodCallExpr -> methodCallExpr.getNameAsString()
                        .equals("set"))
                .toList();
        for (MethodCallExpr methodCallExpr : list) {
            if (!methodCallExpr.getArguments().isEmpty()) {
                Expression argument = methodCallExpr.getArgument(0);
                if (argument.isStringLiteralExpr() && argument
                        .asStringLiteralExpr().asString().equals(argName)) {
                    JavaRewriterUtil.removeStatement(methodCallExpr);
                }
            }
        }
    }

    /**
     * Removes arguments from <code>getThemeList().add()</code> or
     * <code>getThemeList().addAll()</code>. If there is no arguments left after
     * removing, removes the method.
     *
     * @param methodCallStatements
     *            Method calls of the given node
     * @param args
     *            exact string literals to be removed
     */
    public static void removeThemeArgs(
            List<MethodCallExpr> methodCallStatements, List<String> args) {
        List<MethodCallExpr> themeAddCalls = getThemeAddCalls(
                methodCallStatements);
        for (MethodCallExpr addCall : themeAddCalls) {
            List<Expression> argsToRemove = addCall.getArguments().stream()
                    .filter(f -> f.isStringLiteralExpr() && args
                            .contains(f.asStringLiteralExpr().asString()))
                    .collect(Collectors.toList());
            JavaRewriterUtil.removeArgumentCalls(addCall, argsToRemove, true);
        }
    }

    /**
     * Creates addClassNames(...) statement with given arguments if there is
     * none. Adds arguments to existing one.
     *
     * @param component
     *            the component to add class name
     * @param arguments
     *            class name arguments. Most likely StringLiteral or
     *            FieldAccessExpr generated for LumoUtility.
     * @return true if added successfully, false otherwise.
     */
    public static boolean addClassNameWithArgs(ComponentInfo component,
            List<Expression> arguments) {
        MethodCallExpr methodClassExpr;
        if (component.routeConstructor() != null) {
            methodClassExpr = new MethodCallExpr();
        } else {
            if (component.isAnonymousComponent()) {
                JavaRewriter.ExtractInlineVariableResult extractInlineVariableResult = JavaRewriterUtil
                        .extractInlineVariableToLocalVariable(component);
                if (extractInlineVariableResult == null) {
                    return false;
                }
                methodClassExpr = new MethodCallExpr(
                        new NameExpr(
                                extractInlineVariableResult.newVariableName()),
                        addClassName);
                arguments.forEach(methodClassExpr::addArgument);
                extractInlineVariableResult.blockStmt().addStatement(
                        extractInlineVariableResult.index() + 1,
                        methodClassExpr);
                return true;
            }
            String varName = component.localVariableName() != null
                    ? component.localVariableName()
                    : component.fieldName();
            methodClassExpr = new MethodCallExpr(new NameExpr(varName),
                    addClassName);
        }
        arguments.forEach(methodClassExpr::addArgument);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil
                .findMethodCallStatements(component);

        List<MethodCallExpr> classNameMethodCalls = methodCallStatements
                .stream()
                .filter(f -> classNameMethods.contains(f.getNameAsString()))
                .toList();
        // consider only classes after setClassName is set
        for (int i = classNameMethodCalls.size() - 1; i >= 0; i--) {
            if (classNameMethodCalls.get(i).getNameAsString()
                    .equals(setClassName)) {
                classNameMethodCalls = classNameMethodCalls.subList(i + 1,
                        classNameMethodCalls.size());
                break;
            }
        }

        Optional<MethodCallExpr> addClassNamesOptional = classNameMethodCalls
                .stream().filter(f -> f.getNameAsString().equals(addClassNames))
                .findAny();
        // add argument to existing one.
        if (addClassNamesOptional.isPresent()) {
            arguments.forEach(
                    arg -> addClassNamesOptional.get().addArgument(arg));
            return true;
        } else {
            // creates new addClassNames statement
            return JavaRewriterUtil.addAfterLastFunctionCall(
                    methodCallStatements, addClassNames,
                    arguments.toArray(Expression[]::new)) != null;
        }

    }

    /**
     * Searches addClassName, addClassNames, setClassName methods for given
     * component and then remove given lumo utility class names. e.g. if
     * self-start is given for removal, <b>LumoUtility.AlignSelf.START</b> and
     * <b>"self-start"</b> are removed from args.
     *
     * @param component
     *            the component to remove class names
     * @param lumoUtilityClassNames
     *            Utility class names such as "align-items", "gap-m"
     */
    public static void removeClassNameArgs(ComponentInfo component,
            String... lumoUtilityClassNames) {
        List<String> classNameDefinitionMethodNames = List.of(addClassName,
                addClassNames, setClassName);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil
                .findMethodCallStatements(component);
        List<MethodCallExpr> classNameDefinitionMethods = methodCallStatements
                .stream().filter(f -> classNameDefinitionMethodNames
                        .contains(f.getNameAsString()))
                .toList();
        List<Expression> argsWillBeRemoved = new ArrayList<>();
        for (String lumoUtilityClassName : lumoUtilityClassNames) {
            argsWillBeRemoved.addAll(getPossibleLumoUtilityMethodArgExpressions(
                    lumoUtilityClassName));
        }
        JavaRewriterUtil.removeArgumentCalls(classNameDefinitionMethods,
                argsWillBeRemoved, true);
    }

    public static void addLumoUtilityImport(CompilationUnit compilationUnit) {
        JavaRewriterUtil.addImport(compilationUnit,
                "com.vaadin.flow.theme.lumo.LumoUtility");
    }

    /**
     * @param lumoInnerClassName
     *            Inner class name of LumoUtility class. e.g. AlignSelf,
     *            AlignItems etc...
     * @return the list of expressions
     */
    private static List<Expression> getPossibleLumoUtilityMethodArgExpressions(
            String lumoInnerClassName) {
        BiMap<String, String> utilityClasses = getLumoFieldsNameValueMap(
                lumoInnerClassName);
        List<Expression> lumoRelatedUtilityClasses = new ArrayList<>();
        lumoRelatedUtilityClasses.addAll(utilityClasses.keySet().stream()
                .map(arg -> new FieldAccessExpr(
                        new NameExpr("LumoUtility." + lumoInnerClassName), arg))
                .toList());
        lumoRelatedUtilityClasses.addAll(utilityClasses.values().stream()
                .map(StringLiteralExpr::new).toList());
        return lumoRelatedUtilityClasses;
    }

    /**
     * Converts given class names to LumoUtility field access expressions. e.g.
     * items-center becomes AlignItems.CENTER
     *
     * @param lumoInnerClassName
     *            Inner class name of LumoUtility class. e.g. AlignSelf,
     *            AlignItems etc...
     * @param classNames
     *            html class names
     * @return list of expressions.
     */
    public static List<Expression> getLumoMethodArgExpressions(
            String lumoInnerClassName, List<String> classNames) {
        BiMap<String, String> classKeyValueMap = getLumoFieldsNameValueMap(
                lumoInnerClassName);
        return classNames.stream()
                .map(lumoClassValue -> classKeyValueMap.inverse()
                        .get(lumoClassValue))
                .filter(Objects::nonNull)
                .map(arg -> new FieldAccessExpr(
                        new NameExpr("LumoUtility." + lumoInnerClassName), arg))
                .map(k -> (Expression) k).toList();
    }

    /**
     * Converts given class names of given lumo inner class name lists and
     * return all.
     *
     * @param lumoInnerClassNames
     *            List of lumo inner class names e.g. Gap, Gap.Row, Gap.Column
     * @param classNames
     *            class names to look up in given lumo classes
     * @return List of expression to add to a method as arguments
     */
    public static List<Expression> getLumoMethodArgExpressions(
            List<String> lumoInnerClassNames, List<String> classNames) {
        List<Expression> expressions = new ArrayList<>();
        for (String lumoInnerClassName : lumoInnerClassNames) {
            List<Expression> lumoMethodArgExpressions = getLumoMethodArgExpressions(
                    lumoInnerClassName, classNames);
            expressions.addAll(lumoMethodArgExpressions);
        }
        return expressions;
    }

    /**
     * Using reflection to get variables from given Lumo class Key -> field
     * name, value -> field value.
     *
     * @return bidirectional map of class variables.
     */
    private static BiMap<String, String> getLumoFieldsNameValueMap(
            String innerClassName) {
        try {
            BiMap<String, String> biMap = HashBiMap.create();
            Class<?> lumoUtilityClazz = JavaRewriterUtil.getClass(
                    "com.vaadin.flow.theme.lumo.LumoUtility$" + innerClassName);
            for (Field declaredField : lumoUtilityClazz.getDeclaredFields()) {
                String name = declaredField.getName();
                String value = (String) declaredField.get(null);
                biMap.put(name, value);
            }
            return biMap;
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(
                    "There is no lumo utility field named " + innerClassName);
        }
    }

    private static List<MethodCallExpr> getThemeAddCalls(
            List<MethodCallExpr> methodCallStatements) {
        return methodCallStatements.stream()
                .filter(methodCallExpr -> methodCallExpr.getScope().isPresent())
                .filter(methodCallExpr -> methodCallExpr.getScope().get()
                        .toString().contains("getThemeList"))
                .filter(methodCallExpr -> methodCallExpr.getNameAsString()
                        .equals("add")
                        || methodCallExpr.getNameAsString().equals("addAll"))
                .collect(Collectors.toList());

    }
}
