/*
 * Copyright 2000-2024 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.copilot.plugins.info;

import com.vaadin.copilot.CopilotVersion;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;
import com.vaadin.flow.server.Mode;
import com.vaadin.flow.server.Platform;
import com.vaadin.flow.server.Version;
import com.vaadin.flow.server.frontend.FrontendVersion;
import com.vaadin.flow.server.startup.ApplicationConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.jar.JarFile;
import java.util.stream.Collectors;

public class CopilotInfo implements Serializable {

    public record NameAndVersion(String name, String version)
            implements Serializable {
    }

    public record JdkInfo(boolean jrebel, boolean extendedClassDefCapable,
            boolean runningWithExtendClassDef, boolean hotswapAgentFound,
            String hotswapAgentLocation, boolean runningWitHotswap,
            boolean runningInJavaDebugMode, String hotswapVersion,
            boolean hotswapVersionOk) {

    }

    private final List<NameAndVersion> versions = new ArrayList<>();
    private final JdkInfo jdkInfo;

    /**
     * Creates a new instance.
     */
    public CopilotInfo(ApplicationConfiguration applicationConfiguration) {
        // The order here is the order shown in dev tools
        if (EndpointRequestUtil.isHillaAvailable()) {
            versions.add(new NameAndVersion("Hilla", fetchHillaVersion()));
        }
        versions.add(new NameAndVersion("Flow", Version.getFullVersion()));
        if (isVaadinAvailable()) {
            versions.add(new NameAndVersion("Vaadin", fetchVaadinVersion()));
        }
        versions.add(
                new NameAndVersion("Copilot", CopilotVersion.getVersion()));
        versions.add(new NameAndVersion("Copilot IDE Plugin", ""));
        versions.add(new NameAndVersion("Java", fetchJavaVersion()));
        versions.add(new NameAndVersion("Java Hotswap", ""));

        String frontendHotswap = applicationConfiguration
                .getMode() == Mode.DEVELOPMENT_FRONTEND_LIVERELOAD
                        ? "Enabled, using Vite"
                        : "Disabled, using pre-built bundle";
        versions.add(new NameAndVersion("Frontend Hotswap", frontendHotswap));
        versions.add(new NameAndVersion("OS", fetchOperatingSystem()));

        jdkInfo = determineJdkInfo();
    }

    private JdkInfo determineJdkInfo() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        List<String> inputArguments = runtimeMXBean.getInputArguments();

        boolean jrebel = System.getProperty("rebel.base") != null;
        boolean extendedClassDefCapable = runtimeMXBean.getVmVendor()
                .contains("JetBrains");
        boolean runningWithExtendClassDef = inputArguments
                .contains("-XX:+AllowEnhancedClassRedefinition");
        boolean runningInJavaDebugMode = inputArguments.toString()
                .toLowerCase(Locale.ENGLISH).contains("jdwp");

        List<String> hotswapJavaAgent = inputArguments.stream()
                .filter(arg -> arg.startsWith("-javaagent:"))
                .filter(arg -> arg.endsWith("hotswap-agent.jar"))
                .collect(Collectors.toList());

        File javaHome = new File(System.getProperty("java.home"));
        File hotswapAgentLocation = null;
        if (inputArguments.contains("-XX:HotswapAgent=fatjar")) {
            hotswapAgentLocation = new File(
                    new File(new File(javaHome, "lib"), "hotswap"),
                    "hotswap-agent.jar");
        } else if (!hotswapJavaAgent.isEmpty()) {
            hotswapAgentLocation = new File(
                    hotswapJavaAgent.get(0).substring("-javaagent:".length()));
        }

        boolean runningWithHotswap = false;
        boolean hotswapAgentFound = false;
        String hotswapVersionString = null;
        String hotswapAgentJarPath = null;
        if (hotswapAgentLocation != null) {
            runningWithHotswap = true;
            hotswapAgentFound = hotswapAgentLocation.exists();
            hotswapVersionString = getHotswapAgentVersionString(
                    hotswapAgentLocation).orElse(null);
            hotswapAgentJarPath = hotswapAgentLocation.getAbsolutePath();
        }

        boolean hotswapVersionOk = checkHotswapAgentVersion(
                hotswapVersionString);
        return new JdkInfo(jrebel, extendedClassDefCapable,
                runningWithExtendClassDef, hotswapAgentFound,
                hotswapAgentJarPath, runningWithHotswap, runningInJavaDebugMode,
                hotswapVersionString, hotswapVersionOk);
    }

    private Optional<String> getHotswapAgentVersionString(
            File hotswapAgentLocation) {
        if (hotswapAgentLocation.exists()) {
            try (JarFile jarFile = new JarFile(hotswapAgentLocation)) {
                return Optional
                        .ofNullable(jarFile.getManifest().getMainAttributes()
                                .getValue("Implementation-Version"));
            } catch (IOException e) {
                getLogger().error("No META-INF/MANIFEST.MF found in {}",
                        hotswapAgentLocation, e);
            }
        }

        return Optional.empty();
    }

    private boolean checkHotswapAgentVersion(String hotswapVersionString) {
        return getHotswapAgentVersion(hotswapVersionString)
                .map(frontendVersion -> frontendVersion
                        .isEqualOrNewer(new FrontendVersion(1, 4, 2)))
                .orElse(false);
    }

    private Optional<FrontendVersion> getHotswapAgentVersion(
            String versionString) {
        if (versionString == null) {
            return Optional.empty();
        }
        String withoutSnapshot = versionString.replace("-SNAPSHOT", "");
        return Optional.of(new FrontendVersion(withoutSnapshot));
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    public static String fetchJavaVersion() {
        String vendor = System.getProperty("java.vendor");
        String version = System.getProperty("java.version");

        return vendor + " " + version;
    }

    public static String fetchOperatingSystem() {
        String arch = System.getProperty("os.arch");
        String name = System.getProperty("os.name");
        String version = System.getProperty("os.version");

        return arch + " " + name + " " + version;
    }

    public static String fetchVaadinVersion() {
        return isVaadinAvailable() ? Platform.getVaadinVersion().orElse("?")
                : "-";
    }

    public static String fetchHillaVersion() {
        return EndpointRequestUtil.isHillaAvailable()
                ? Platform.getHillaVersion().orElse("?")
                : "-";
    }

    public List<NameAndVersion> getVersions() {
        return versions;
    }

    public JdkInfo getJdkInfo() {
        return jdkInfo;
    }

    private static boolean isVaadinAvailable() {
        return Thread.currentThread().getContextClassLoader().getResource(
                "META-INF/maven/com.vaadin/vaadin-core/pom.properties") != null;
    }
}