package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.pro.licensechecker.MachineId;
import elemental.json.Json;
import elemental.json.JsonObject;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;

/**
 * Copilot 3 day preview handler. It has two basic functionality that starts a new preview and get the preview state.
 */
public class PreviewHandler extends CopilotServerClient implements CopilotCommand {
    private static final String REMAINING_TIME_IN_MILLIS = "remainingTimeInMillis";
    private static final String PREVIEW_ACTIVATED = "previewActivated";
    private static final String PREVIEW_ACTIVE = "active";

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (!data.hasKey(KEY_REQ_ID)) {
            return false;
        }
        JsonObject responseData = Json.createObject();
        responseData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
        if (!(command.equals("start-preview") || command.equals("get-preview"))) {
            return false;
        }
        if (command.equals("start-preview")) {
            try {
                startPreview();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            } catch (Exception e) {
                getLogger().debug("Could not start preview", e);
                ErrorHandler.sendErrorResponse(
                        devToolsInterface, command, responseData, "Could not connect server for starting the trial", e);
                return true;
            }
        }
        putPreviewData(responseData);
        devToolsInterface.send("copilot-get-preview-response", responseData);
        return true;
    }

    private void startPreview() throws IOException, InterruptedException {
        URI uri = getQueryURI("activation");
        JsonObject requestBody = Json.createObject();
        requestBody.put("machineId", MachineId.get());
        HttpRequest request = buildRequest(uri, requestBody.toJson());
        HttpResponse<String> response = getHttpClient().send(request, HttpResponse.BodyHandlers.ofString());
        if (response.statusCode() != 200) {
            throw new IllegalArgumentException("Starting preview failed");
        }
        ActivationResponse previewStartResponse = readValue(response.body(), ActivationResponse.class);
        if (previewStartResponse == null || !previewStartResponse.active) {
            throw new IllegalArgumentException("Starting preview failed");
        }
    }

    private void putPreviewData(JsonObject responseJson) {

        responseJson.put(PREVIEW_ACTIVATED, false);
        try {
            URI uri = getQueryURI("activation?machineId=" + MachineId.get());
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(uri)
                    .GET()
                    .timeout(Duration.ofSeconds(120))
                    .build();
            HttpResponse<String> response = getHttpClient().send(request, HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 404) {
                responseJson.put(PREVIEW_ACTIVATED, false);
                responseJson.put(PREVIEW_ACTIVE, false);
                return;
            }
            ActivationResponse getPreviewResponse = readValue(response.body(), ActivationResponse.class);
            responseJson.put(PREVIEW_ACTIVE, getPreviewResponse.active);
            responseJson.put(PREVIEW_ACTIVATED, true);
            responseJson.put(REMAINING_TIME_IN_MILLIS, getPreviewResponse.timeUntilExpireInMillis());
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        } catch (Exception ex) {
            // silent exception
            getLogger().debug("Could not read preview state", ex);
        }
    }

    public record ActivationResponse(boolean active, long activationTimeInMillis, long timeUntilExpireInMillis) {}
}
