package com.vaadin.copilot;

import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.routes.AccessRequirement;
import com.vaadin.flow.server.auth.AnonymousAllowed;
import com.vaadin.flow.shared.util.SharedUtil;
import jakarta.annotation.security.DenyAll;
import jakarta.annotation.security.PermitAll;
import jakarta.annotation.security.RolesAllowed;
import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Handles creation of views for new routes. */
public class RouteCreator {

    private final ProjectManager projectManager;

    public RouteCreator(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteCreator.class);
    }

    /**
     * Creates a Flow view for the given route using the given access requirement.
     *
     * @param route the path to create a view for
     * @param accessRequirement the access requirement for the view
     */
    public void createFlowView(String route, AccessRequirement accessRequirement) throws IOException {
        getLogger().debug("Creating Flow view for route {}", route);

        String[] parts = route.split("/");
        String filename = parts[parts.length - 1];
        if (filename.isEmpty() || filename.endsWith("/")) {
            filename = "Main";
        } else {
            filename = SharedUtil.capitalize(filename);
        }

        if (!projectManager.sanitizeFilename(filename).equals(filename)) {
            throw new IllegalArgumentException("Invalid filename " + filename);
        }

        String viewsPackage = projectManager.getFlowViewsPackage();

        File viewFile = projectManager.getJavaSourceFolder();
        for (String folder : viewsPackage.split("\\.")) {
            viewFile = new File(viewFile, folder);
        }

        String className = filename;
        viewFile = new File(viewFile, filename + ".java");
        if (viewFile.exists()) {
            throw new IllegalArgumentException("File already exists: " + viewFile.getAbsolutePath());
        }
        viewFile.getParentFile().mkdirs();
        projectManager.writeFile(
                viewFile,
                CopilotIDEPlugin.undoLabel("Add route"),
                getFlowRouteTemplate(viewsPackage, route, className, accessRequirement));
    }

    /**
     * Creates a Hilla view for the given route using the given access requirement.
     *
     * @param route the path to create a view for
     * @param accessRequirement the access requirement for the view
     */
    public void createHillaView(String route, AccessRequirement accessRequirement) throws IOException {
        getLogger().debug("Creating Hilla view for route {}", route);
        // Assumes FS router will set up routing
        String filename = route;
        if (filename.isEmpty() || filename.endsWith("/")) {
            filename += "@index";
        }

        if (!projectManager.sanitizeFilename(filename).equals(filename)) {
            throw new IllegalArgumentException("Invalid filename " + filename);
        }
        File viewFile = projectManager.getHillaViewsFolder();

        String[] parts = filename.split("/");
        for (String part : parts) {
            viewFile = new File(viewFile, part);
        }
        viewFile = addExtension(viewFile, ".tsx");
        if (viewFile.exists()) {
            throw new IllegalArgumentException("File already exists: " + viewFile.getAbsolutePath());
        }
        viewFile.getParentFile().mkdirs();
        projectManager.writeFile(
                viewFile,
                CopilotIDEPlugin.undoLabel("Add route"),
                getHillaRouteTemplate(parts[parts.length - 1], accessRequirement));
    }

    private String getHillaRouteTemplate(String routeName, AccessRequirement accessRequirement) {
        String functionName = SharedUtil.capitalize(routeName) + "View";
        if (functionName.matches("^[^A-Z].*")) {
            // React component names must start with a capital letter
            functionName = "A" + functionName;
        }
        String accessControl;
        switch (accessRequirement.getType()) {
            case PERMIT_ALL:
                accessControl = "loginRequired: true,";
                break;
            case ANONYMOUS_ALLOWED:
                accessControl = "";
                break;
            case DENY_ALL:
                accessControl = "rolesAllowed: [],";
                break;
            case ROLES_ALLOWED:
                accessControl = "rolesAllowed: ["
                        + Arrays.stream(accessRequirement.getRoles())
                                .map(role -> "\"" + role + "\"")
                                .collect(Collectors.joining(", "))
                        + "],";
                break;
            default:
                throw new IllegalArgumentException("Unknown access requirement: " + accessRequirement.getType());
        }
        return """
                import type { ViewConfig } from '@vaadin/hilla-file-router/types.js';

                export const config: ViewConfig = {
                  title: 'TITLE',
                  ACCESS_CONTROL
                };

                export default function FUNCTION_NAME() {
                    return (
                        <div>
                            <h1>ROUTE_NAME</h1>
                        </div>
                    );
                }
                """
                .replace("TITLE", routeName)
                .replace("FUNCTION_NAME", functionName)
                .replace("ROUTE_NAME", routeName)
                .replace("ACCESS_CONTROL", accessControl);
    }

    private String getFlowRouteTemplate(
            String packageName, String route, String className, AccessRequirement accessRequirement) {
        Class<? extends Annotation> accessClass;
        String accessParam = null;
        switch (accessRequirement.getType()) {
            case PERMIT_ALL:
                accessClass = PermitAll.class;
                break;
            case ANONYMOUS_ALLOWED:
                accessClass = AnonymousAllowed.class;
                break;
            case DENY_ALL:
                accessClass = DenyAll.class;
                break;
            case ROLES_ALLOWED:
                accessClass = RolesAllowed.class;
                accessParam = "{"
                        + Arrays.stream(accessRequirement.getRoles())
                                .map(role -> "\"" + role + "\"")
                                .collect(Collectors.joining(", "))
                        + "}";
                break;
            default:
                throw new IllegalArgumentException("Unknown access requirement: " + accessRequirement.getType());
        }

        String accessAnnotation = accessClass.getSimpleName();
        if (accessParam != null) {
            accessAnnotation += "(" + accessParam + ")";
        }

        return """
                package VIEW_PACKAGE;

                import com.vaadin.flow.component.html.H1;
                import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
                import com.vaadin.flow.router.Route;
                import com.vaadin.flow.server.auth.AnonymousAllowed;
                import ACCESS_CLASS_NAME;

                @Route(value = "ROUTE")
                @ACCESS_ANNOTATION
                public class CLASS_NAME extends HorizontalLayout {

                    public CLASS_NAME() {
                        add(new H1("This is CLASS_NAME"));
                    }

                }
                """
                .replace("VIEW_PACKAGE", packageName)
                .replace("ROUTE", route)
                .replace("ACCESS_CLASS_NAME", accessClass.getName())
                .replace("CLASS_NAME", className)
                .replace("ACCESS_ANNOTATION", accessAnnotation);
    }

    private static File addExtension(File viewFile, String ext) {
        return new File(viewFile.getParentFile(), viewFile.getName() + ext);
    }
}
