package com.vaadin.copilot.ai;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.ServerInfo;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.CopilotVersion;
import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.communication.CopilotServerRequest;
import com.vaadin.copilot.communication.CopilotServerResponse;
import com.vaadin.copilot.communication.CopilotServerResponseCode;
import com.vaadin.copilot.communication.StreamResponse;
import com.vaadin.copilot.communication.StreamResponseEnum;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;
import elemental.json.Json;
import elemental.json.JsonObject;
import io.netty.buffer.ByteBufAllocator;
import io.netty.handler.codec.http.HttpMethod;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import reactor.core.Disposable;
import reactor.core.publisher.Mono;
import reactor.netty.ByteBufMono;
import reactor.netty.http.client.HttpClient;

/** Command handler for AI related operations */
public class AICommandHandler extends CopilotServerClient implements CopilotCommand {
    private final Map<String, String> serverInfoVersions;
    private final ProjectManager projectManager;

    public AICommandHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
        ServerInfo serverInfo = new ServerInfo();
        serverInfoVersions = serverInfo.getVersions().stream()
                .collect(Collectors.toMap(ServerInfo.NameAndVersion::name, ServerInfo.NameAndVersion::version));
    }

    private final Map<String, Disposable> disposables = new HashMap<>();

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("prompt-text")) {
            Map<String, String> metadata = new HashMap<>(serverInfoVersions);
            ProKey proKey = getProKey();
            String machineId = getMachineId();
            if (proKey == null && machineId == null) {
                AICommunicationUtil.promptTextCannotCall(data, devToolsInterface);
                return false;
            }

            if (proKey != null) {
                metadata.put(AIConstants.PRO_KEY_KEY, proKey.toJson());
            }
            if (machineId != null) {
                metadata.put(AIConstants.MACHINE_ID_KEY, machineId);
            }
            metadata.put(AIConstants.VERSION_KEY, CopilotVersion.getVersion());

            String prompt = data.getString("text");
            // Hilla source files:
            Map<String, String> sources = AICommunicationUtil.getHillaSourceFiles(data);

            // Java source files:
            if (data.hasKey("uiid")) {
                try {
                    sources.putAll(getJavaSourceMap((int) data.getNumber("uiid")));
                } catch (IOException e) {
                    getLogger().error("Error reading requested project Flow Java files", e);
                    devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, Json.createObject());
                    return true;
                }
            }

            // Make all sources relative to the project, so we do not send
            // information about where the project is located
            Map<String, String> relativeSources = new HashMap<>();

            for (String filename : sources.keySet()) {
                try {
                    relativeSources.put(projectManager.makeRelative(filename), sources.get(filename));
                } catch (IOException e) {
                    JsonObject responseData = Json.createObject();
                    if (data.hasKey(KEY_REQ_ID)) {
                        responseData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
                    }
                    responseData.put("code", AIConstants.COPILOT_INTERNAL_ERROR);
                    responseData.put("message", "Error processing the files");
                    getLogger().error("Error making file relative to project", e);
                    devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
                    return true;
                }
            }

            try {
                CopilotServerRequest req = new CopilotServerRequest(prompt, relativeSources, metadata);
                JsonObject responseData = Json.createObject();
                responseData.put(KEY_CANCEL_REQ_ID, (data.hasKey(KEY_REQ_ID) ? data.getString(KEY_REQ_ID) : ""));
                getLogger().debug("Request Registered in client: " + responseData.getString(KEY_CANCEL_REQ_ID));
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_REQUEST_ID, responseData);
                queryCopilotServer(req, devToolsInterface, data);
            } catch (Exception e) {
                AICommunicationUtil.handlingExceptionsAndNotifyDevtoolsInterface(data, devToolsInterface, e);
                return true;
            }
            return true;
        } else if (command.equals("prompt-cancel")) {
            JsonObject responseData = Json.createObject();
            if (data.hasKey(KEY_CANCEL_REQ_ID) && data.hasKey(KEY_REQ_ID)) {
                responseData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
                responseData.put(KEY_CANCEL_REQ_ID, data.getString(KEY_CANCEL_REQ_ID));
            } else {
                getLogger().error("Impossible to cancel without request and cancel request Ids");
                return false;
            }
            getLogger()
                    .debug(
                            "Trying to cancel Disposable object with id: {} - object: {}",
                            data.getString(KEY_REQ_ID),
                            disposables.get(data.getString(KEY_REQ_ID)));

            if (disposables.get(data.getString(KEY_CANCEL_REQ_ID)) != null) {
                responseData.put("message", "Request cancelled successfully");
                Disposable disposable = disposables.get(data.getString(KEY_CANCEL_REQ_ID));
                disposable.dispose();
                disposables.remove(data.getString(KEY_CANCEL_REQ_ID));
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_CANCEL_OK, responseData);
            } else {
                responseData.put("message", "Error cancelling the request with id: " + data.getString(KEY_REQ_ID));
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_CANCEL_FAILED, responseData);
            }
            return true;
        }
        return false;
    }

    private Map<String, String> getJavaSourceMap(int uiId) throws IOException {
        Map<String, String> sources = new HashMap<>();
        for (Map.Entry<ComponentTracker.Location, File> entry :
                FlowUtil.findActiveJavaFiles(uiId).entrySet()) {
            ArrayList<String> javaFileNames = new ArrayList<>();
            File javaFile = entry.getValue();
            javaFileNames.add(javaFile.getName());
            sources.put(javaFile.getAbsolutePath(), projectManager.readFile(javaFile.getPath()));
            getLogger().debug("Java filenames: {}", javaFileNames);
        }
        return sources;
    }

    private void addComponents(Component component, List<Component> componentList) {
        componentList.add(component);
        component.getChildren().forEach(c -> addComponents(c, componentList));
    }

    private void queryCopilotServer(
            CopilotServerRequest req, DevToolsInterface devToolsInterface, JsonObject dataJson) {

        URI queryUriStream = getQueryURI("stream");
        String json = writeAsJsonString(req);
        if (Copilot.isDevelopmentMode()) {
            getLogger().debug("Querying copilot server at {} using {}", queryUriStream, json);
        }

        ObjectMapper objectMapper = new ObjectMapper(); // Jackson's JSON parser
        AtomicReference<StringBuilder> jsonBuffer = new AtomicReference<>(new StringBuilder());

        AtomicBoolean completedSuccessfully = new AtomicBoolean(false);

        Disposable disposable = HttpClient.create()
                .responseTimeout(Duration.of(AIConstants.CLIENT_MAX_TIMEOUT, ChronoUnit.SECONDS))
                .headers(headers -> headers.set("Content-Type", "application/json; charset=utf-8"))
                .request(HttpMethod.POST)
                .send(ByteBufMono.fromString(Mono.just(json), StandardCharsets.UTF_8, ByteBufAllocator.DEFAULT))
                .uri(queryUriStream)
                .responseContent()
                .asString(StandardCharsets.UTF_8)
                .doOnError(e -> AICommunicationUtil.handlingExceptionsAndNotifyDevtoolsInterface(
                        dataJson, devToolsInterface, e))
                .subscribe(
                        data -> {
                            // Append the new data to the buffer, removing any potential
                            // "data:" prefixes
                            String newData = data.trim();

                            if (newData.startsWith("data:")) {
                                newData = newData.substring(5);
                            }
                            jsonBuffer.get().append(newData);

                            // Try to parse the accumulated data as JSON
                            String accumulatedData = jsonBuffer.get().toString();
                            try {
                                // Attempt to parse the accumulated data as JSON
                                Optional<StreamResponse> parsedStreamResponseOpt;
                                if ((parsedStreamResponseOpt = tryParseJson(objectMapper, accumulatedData))
                                        .isPresent()) {
                                    StreamResponse parsedStreamResponse = parsedStreamResponseOpt.get();
                                    // Process the parsed JSON object
                                    JsonObject responseData = Json.createObject();
                                    responseData.put(
                                            "status",
                                            parsedStreamResponse.status().getMessage());
                                    getLogger().debug("Parsed JSON: " + parsedStreamResponse);
                                    devToolsInterface.send(
                                            Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_STATUS, responseData);

                                    completedSuccessfully.set(true);

                                    if (dataJson.hasKey(KEY_REQ_ID)) {
                                        responseData.put(KEY_REQ_ID, dataJson.getString(KEY_REQ_ID));
                                    }
                                    if (parsedStreamResponse.code() < 0) {
                                        responseData.put("code", parsedStreamResponse.code());
                                        responseData.put("message", parsedStreamResponse.message());
                                        devToolsInterface.send(
                                                Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
                                    } else if (parsedStreamResponse.status() == StreamResponseEnum.POST_PROCESS) {
                                        responseData.put("message", parsedStreamResponse.message());
                                        responseData.put(
                                                "changes",
                                                parsedStreamResponse.changes().keySet().stream()
                                                        .map(Json::create)
                                                        .collect(JsonUtils.asArray()));
                                        getLogger().debug("PostProcess finished");
                                        devToolsInterface.send(
                                                Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_OK, responseData);

                                        CopilotServerResponse response = new CopilotServerResponse(
                                                CopilotServerResponseCode.HILLA_REACT.getCode(),
                                                parsedStreamResponse.status().getMessage(),
                                                parsedStreamResponse.changes());

                                        handleQueryResponse(response);
                                    }
                                    jsonBuffer.set(new StringBuilder());
                                }
                            } catch (Exception e) {
                                getLogger()
                                        .error("Error parsing JSON: "
                                                + e.getMessage() + " - Accumulated data: "
                                                + accumulatedData);

                                JsonObject responseData = Json.createObject();
                                responseData.put(KEY_REQ_ID, dataJson.getString(KEY_REQ_ID));
                                devToolsInterface.send(
                                        Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
                                jsonBuffer.set(new StringBuilder(accumulatedData));
                            }
                        },
                        error -> getLogger().error("Error: " + error),
                        () -> {
                            if (!completedSuccessfully.get()) {
                                JsonObject responseData = Json.createObject();
                                responseData.put(KEY_REQ_ID, dataJson.getString(KEY_REQ_ID));
                                getLogger().error("Stream did not completed successfully.");
                                devToolsInterface.send(
                                        Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
                            } else {
                                disposables.remove(dataJson.getString(KEY_REQ_ID));
                                // Stream completed successfully
                                getLogger().debug("Stream completed successfully.");
                            }
                        });
        disposables.put(dataJson.getString(KEY_REQ_ID), disposable);
        getLogger()
                .debug(
                        "Disposable object created for prompt {} - object: {} - RequestId: {}",
                        json,
                        disposable,
                        dataJson.getString(KEY_REQ_ID));
    }

    // Utility method to attempt parsing JSON from the given string, returns
    // null if parsing fails or JSON is incomplete
    private Optional<StreamResponse> tryParseJson(ObjectMapper objectMapper, String data) {
        try {
            // ObjectMapper attempts to parse the data, may throw if incomplete
            // or invalid
            return Optional.ofNullable(objectMapper.readValue(data, StreamResponse.class));
        } catch (IOException e) {
            return Optional.empty(); // Return empty to indicate parsing was
            // unsuccessful or
            // data is incomplete
        }
    }

    private void handleQueryResponse(CopilotServerResponse response) throws IOException {
        if (response.code() == CopilotServerResponseCode.ERROR.getCode()) {
            getLogger()
                    .error("Copilot server returned error because an internal error."
                            + " The reason could be a malformed request or a timeout.");
            return;
        } else if (response.code() == CopilotServerResponseCode.ERROR_REQUEST.getCode()) {
            getLogger()
                    .error("Copilot server returned error because an internal error."
                            + " The reason could be a malformed request or a timeout.");
            return;
        } else if (response.code() == CopilotServerResponseCode.NOTHING.getCode()) {
            getLogger().debug("Copilot server returned no changes");
            return;
        } else if (response.code() == CopilotServerResponseCode.HILLA_REACT.getCode()) {
            getLogger().debug("Copilot server returned Hilla/React changes");
        } else if (response.code() == CopilotServerResponseCode.FLOW.getCode()) {
            getLogger().debug("Copilot server returned Flow changes");
        } else if (response.code() == CopilotServerResponseCode.LOCAL.getCode()) {
            getLogger().debug("Copilot server returned Local changes");
        } else if (response.code() < 0) {
            getLogger().debug("Copilot server returned Internal error code: {}", response.code());
            return;
        } else {
            getLogger().debug("Copilot server returned unknown response code: {}", response.code());
            return;
        }

        for (Map.Entry<String, String> change : response.changes().entrySet()) {
            try {
                projectManager.writeFile(
                        projectManager.makeAbsolute(change.getKey()),
                        AIConstants.COPILOT_AI_FILE_UPDATE_UNDO_LABEL,
                        change.getValue());
            } catch (IOException e) {
                throw new IOException(
                        "Unable to write file (" + change.getKey() + ") with data from copilot server response", e);
            }
        }
    }

    ProKey getProKey() {
        return LocalProKey.get();
    }

    String getMachineId() {
        return MachineId.get();
    }

    public Map<String, Disposable> getDisposables() {
        return disposables;
    }
}
