package com.vaadin.copilot.javarewriter;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConstructorAnalyzer {

    record ConstructorCall(String classNameSignature, String descriptor, List<Integer> params) {}

    public static final int ASM_API = Opcodes.ASM9;
    private static ConstructorAnalyzer INSTANCE = new ConstructorAnalyzer();

    private final Map<String, Map<Integer, String>> constructorParameterMappings = new HashMap<>();

    public static ConstructorAnalyzer get() {
        return INSTANCE;
    }

    ConstructorAnalyzer() {
        // Singleton
    }

    static class ConstructorVisitor extends ClassVisitor {
        private final String constructor;
        private final ConstructorMethodVisitor visitor = new ConstructorMethodVisitor();
        Map<String, String> constructorsCalled = new HashMap<>();

        public ConstructorVisitor(String constructor) {
            super(ASM_API);
            this.constructor = constructor;
        }

        @Override
        public MethodVisitor visitMethod(
                int access, String name, String constructorDescriptor, String signature, String[] exceptions) {
            if ("<init>".equals(name) && this.constructor.equals(constructorDescriptor)) {
                return visitor;
            }
            return null;
        }

        public Map<Integer, String> getMappings() {
            return visitor.mappings;
        }

        ConstructorCall getConstructorCall() {
            return visitor.constructorCall;
        }
    }

    static class ConstructorMethodVisitor extends MethodVisitor {
        private final List<Integer> loaded = new ArrayList<>();
        private final Map<Integer, String> mappings = new HashMap<>();
        private ConstructorCall constructorCall = null;

        public ConstructorMethodVisitor() {
            super(ASM_API);
        }

        @Override
        public void visitVarInsn(int opcode, int var) {
            super.visitVarInsn(opcode, var);
            if (opcode == Opcodes.ALOAD) {
                loaded.add(var - 1); // Put 'this' as -1 and others starting
                // from 0
            }
        }

        @Override
        public void visitMethodInsn(int opcode, String owner, String name, String descriptor, boolean isInterface) {
            if (opcode == Opcodes.INVOKESPECIAL && "<init>".equals(name)) {
                if (loaded.size() > 1 && loaded.get(0) == -1) {
                    // Calling another constructor and passing at least one
                    // argument
                    constructorCall =
                            new ConstructorCall(owner, descriptor, new ArrayList<>(loaded.subList(1, loaded.size())));
                }
            } else if (name.startsWith("set")) {
                if (loaded.size() == 2 && loaded.get(0) == -1) {
                    // Two arguments, the first is -1 == this, the second is the
                    // index of the value
                    mappings.put(loaded.get(1), name);
                }
            }
            loaded.clear();
        }
    }

    public Map<Integer, String> getMappings(Constructor<?> constructor) {
        String constructorDescriptor = getConstructorDescriptor(constructor);
        return getMappings(constructor.getDeclaringClass().getName(), constructorDescriptor);
    }

    private Map<Integer, String> getMappings(String className, String constructorDescriptor) {
        String mapId = className + "#" + constructorDescriptor;
        if (!constructorParameterMappings.containsKey(mapId)) {
            constructorParameterMappings.put(mapId, getMappingsForConstructor(className, constructorDescriptor));
        }
        return constructorParameterMappings.get(mapId);
    }

    private Map<Integer, String> getMappingsForConstructor(String className, String constructorDescriptor) {
        ClassReader cr;
        try {
            cr = new ClassReader(className);
        } catch (IOException e) {
            getLogger().error("Unable to read class " + className, e);
            return Collections.emptyMap();
        }

        ConstructorVisitor visitor = new ConstructorVisitor(constructorDescriptor);
        cr.accept(visitor, 0);

        if (visitor.getConstructorCall() != null) {
            String otherClass = visitor.getConstructorCall().classNameSignature;
            Map<Integer, String> otherMappings = getMappings(otherClass, visitor.getConstructorCall().descriptor);

            for (int toIndex = 0; toIndex < visitor.getConstructorCall().params.size(); toIndex++) {
                int fromIndex = visitor.getConstructorCall().params.get(toIndex);
                String property = otherMappings.get(toIndex);
                if (property != null) {
                    visitor.getMappings().put(fromIndex, property);
                }
            }
        }
        return visitor.getMappings();
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    private static String getConstructorDescriptor(Constructor<?> constructor) {
        Class<?>[] parameterTypes = constructor.getParameterTypes();
        Type[] types = new Type[parameterTypes.length];

        for (int i = 0; i < parameterTypes.length; i++) {
            types[i] = Type.getType(parameterTypes[i]);
        }

        return Type.getMethodDescriptor(Type.VOID_TYPE, types);
    }
}
