package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.Statement;

/** Represents a point in the source code where new code can be inserted. */
public class InsertionPoint {
    private final BlockStmt block;
    private final ClassOrInterfaceDeclaration declaration;
    private int index;

    /**
     * Creates a new InsertionPoint instance.
     *
     * @param block the block where to insert the code
     * @param index the index where to insert the code
     */
    public InsertionPoint(BlockStmt block, int index) {
        this.block = block;
        this.index = index;
        this.declaration = null;
    }

    public InsertionPoint(ClassOrInterfaceDeclaration declaration, int index) {
        this.declaration = declaration;
        this.index = index;
        this.block = null;
    }

    /**
     * Returns a free variable name based on the given base name, available in the scope where code will be inserted.
     *
     * @param baseName the base name for the variable
     * @return a free variable name
     */
    public String getFreeVariableName(String baseName) {
        if (getBlock() != null) {
            return JavaRewriterUtil.findFreeVariableName(baseName, getBlock());
        } else if (getDeclaration() != null) {
            return JavaRewriterUtil.findFreeVariableName(baseName, getDeclaration());
        }
        return null;
    }

    /**
     * Adds a statement to the insertion point.
     *
     * @param statement the statement to add
     */
    public void add(Statement statement) {
        if (block != null) {
            block.addStatement(index++, statement);
        } else {
            throw new IllegalArgumentException("Cannot add statement to a class or interface declaration");
        }
    }

    public BlockStmt getBlock() {
        return block;
    }

    public int getIndex() {
        return index;
    }

    public ClassOrInterfaceDeclaration getDeclaration() {
        return declaration;
    }
}
