package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.vaadin.flow.dom.Style;
import com.vaadin.flow.shared.util.SharedUtil;
import java.util.Collections;
import java.util.List;

public class JavaStyleRewriter {

    public record StyleInfo(String property, String value, boolean isDashSeparatedProperty) {}

    /**
     * Gets the (active) styles of a component.
     *
     * @param componentInfo the component to get the styles of
     * @return the styles, as a list of style names and values
     */
    public static List<StyleInfo> getStyles(ComponentInfo componentInfo) {
        return JavaRewriterUtil.findCalls(Style.class, componentInfo).stream()
                .map(JavaStyleRewriter::extractStyle)
                .toList();
    }

    /**
     * Sets the given inline style on the given component, replacing an existing style property if present.
     *
     * @param componentInfo the component to set the style on
     * @param dashSeparatedProperty the style property to set
     * @param value the style value to set or null to remove the style
     */
    public static void setStyle(ComponentInfo componentInfo, String dashSeparatedProperty, String value) {
        List<MethodCallExpr> styleCalls = JavaRewriterUtil.findCalls(Style.class, componentInfo);
        List<MethodCallExpr> existingSetters = styleCalls.stream()
                .filter(methodCallExpr -> {
                    StyleInfo style = extractStyle(methodCallExpr);
                    if (style.isDashSeparatedProperty) {
                        return style.property().equals(dashSeparatedProperty);
                    } else {
                        return SharedUtil.camelCaseToDashSeparated(style.property())
                                .equals(dashSeparatedProperty);
                    }
                })
                .toList();
        if (existingSetters.isEmpty()) {
            if (value == null) {
                throw new IllegalArgumentException(
                        "Unable to remove non-existing style " + dashSeparatedProperty + " from " + componentInfo);
            }
            MethodCallExpr newStylesCall = createStylesCall(dashSeparatedProperty, value);

            if (!styleCalls.isEmpty()) {
                // Add a new style call by chaining to the last existing
                // style call
                MethodCallExpr lastStylesCall = styleCalls.get(styleCalls.size() - 1);

                JavaRewriterUtil.findAncestorOrThrow(lastStylesCall, ExpressionStmt.class)
                        .setExpression(newStylesCall);
                newStylesCall.setScope(lastStylesCall);
            } else {
                // Add a new getStyle() call and use that to set the style
                MethodCallExpr getStyleCall =
                        JavaRewriterUtil.addFunctionCall(componentInfo, "getStyle", Collections.emptyList());
                newStylesCall.setScope(JavaRewriterUtil.clone(getStyleCall));
                getStyleCall.replace(newStylesCall);
            }

        } else {
            // Replace the value in the last existing setter
            updateStyle(existingSetters.get(existingSetters.size() - 1), value);
        }
    }

    private static MethodCallExpr createStylesCall(String dashSeparatedProperty, String value) {
        String camelCaseProperty = SharedUtil.dashSeparatedToCamelCase(dashSeparatedProperty);
        String setter = JavaRewriterUtil.getSetterName(camelCaseProperty, Style.class, false);
        if (JavaRewriterUtil.hasSetterForType(Style.class, setter, String.class)) {
            return new MethodCallExpr(null, setter).addArgument(JavaRewriterUtil.toExpression(value));
        }
        return new MethodCallExpr(null, "set")
                .addArgument(JavaRewriterUtil.toExpression(dashSeparatedProperty))
                .addArgument(JavaRewriterUtil.toExpression(value));
    }

    /**
     * Extracts the style information from a method call to a method inside Style.
     *
     * <p>Note that this only processes the method call and not any chained calls.
     *
     * @param methodCallExpr a method call that refers to a method inside Style.
     * @return The style info for the method call.
     */
    private static StyleInfo extractStyle(MethodCallExpr methodCallExpr) {
        String setter = methodCallExpr.getNameAsString();
        if (setter.equals("set")) {
            String dashProperty = String.valueOf(JavaRewriterUtil.fromExpression(methodCallExpr.getArgument(0), null));
            String value = String.valueOf(JavaRewriterUtil.fromExpression(methodCallExpr.getArgument(1), null));
            return new StyleInfo(dashProperty, value, true);
        } else {
            int argCount = methodCallExpr.getArguments().size();
            if (argCount != 1) {
                throw new IllegalArgumentException(
                        "Expected styles method call expression to have one argument but was " + argCount + " for "
                                + methodCallExpr);
            }
            Expression argument = methodCallExpr.getArgument(0);
            String value;
            if (argument.isFieldAccessExpr()) {
                String fieldName = argument.asFieldAccessExpr().getNameAsString();
                // This is copied from Styles.java
                value = fieldName.replace("_", "-").toLowerCase();
            } else if (argument.isStringLiteralExpr()) {
                value = argument.asStringLiteralExpr().getValue();
            } else {
                throw new IllegalArgumentException("Unexpected argument type in style call: " + methodCallExpr);
            }
            String property = JavaRewriterUtil.getPropertyName(setter);
            return new StyleInfo(property, value, false);
        }
    }

    static void updateStyle(MethodCallExpr methodCallExpr, Object value) {
        if (value == null) {
            JavaRewriterUtil.removeFromChainedStyleCall(methodCallExpr);
            return;
        }
        if (methodCallExpr.getNameAsString().equals("set")) {
            methodCallExpr.getArgument(1).replace(JavaRewriterUtil.toExpression(value));
        } else {
            methodCallExpr.getArgument(0).replace(JavaRewriterUtil.toExpression(value));
        }
    }
}
