package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.Parameter;
import com.github.javaparser.ast.body.RecordDeclaration;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ThisExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithSimpleName;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt;
import java.util.Optional;

/**
 * Class/Record definition related functions for Java Rewriter. This class generates a record to host label/value pairs
 * for components that show simple data item
 *
 * <p>ListBox Checkbox RadioButton
 */
public class LabelValuePairRecordRewriter {
    private static final String DEFAULT_LABEL_AND_VALUE_RECORD_NAME = "LabelAndValue";
    private static final String LABEL_ARG_NAME = "label";
    private static final String VALUE_ARG_NAME = "value";
    private static final String ENABLED_ARG_NAME = "enabled";
    private final CompilationUnit compilationUnit;

    public LabelValuePairRecordRewriter(CompilationUnit compilationUnit) {
        this.compilationUnit = compilationUnit;
    }

    /**
     * Finds the Record Declaration from given complication unit that has Label, Value and Enabled parameters. Creates
     * one with LabelAndValue name if not found.
     *
     * @return of the record name
     */
    public String addOrGetLabelValueEnabledRecordName() {
        return getLabelValueRecord().orElseGet(this::addLabelValueRecord);
    }

    private Optional<String> getLabelValueRecord() {
        Optional<ClassOrInterfaceDeclaration> classOrInterfaceDeclarationOptional =
                compilationUnit.findFirst(ClassOrInterfaceDeclaration.class);
        if (classOrInterfaceDeclarationOptional.isEmpty()) {
            return Optional.empty();
        }
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = classOrInterfaceDeclarationOptional.get();
        return classOrInterfaceDeclaration.getChildNodes().stream()
                .filter(RecordDeclaration.class::isInstance)
                .map(RecordDeclaration.class::cast)
                .filter(recordDeclaration -> {
                    Optional<Parameter> label = recordDeclaration.getParameterByName(LABEL_ARG_NAME);
                    if (label.isEmpty()) {
                        return false;
                    }
                    Optional<Parameter> value = recordDeclaration.getParameterByName(VALUE_ARG_NAME);
                    if (value.isEmpty()) {
                        return false;
                    }
                    Optional<Parameter> enabled = recordDeclaration.getParameterByName(ENABLED_ARG_NAME);
                    return enabled.isPresent();
                    // TODO look for default constructor ?
                })
                .map(NodeWithSimpleName::getNameAsString)
                .findFirst();
    }

    private String addLabelValueRecord() {
        RecordDeclaration recordDeclaration = new RecordDeclaration();
        recordDeclaration.setPublic(true);
        recordDeclaration.setName(DEFAULT_LABEL_AND_VALUE_RECORD_NAME);
        Parameter labelParam = new Parameter();
        labelParam.setType("String");
        labelParam.setName(LABEL_ARG_NAME);
        Parameter valueParam = new Parameter();
        valueParam.setType("String");
        valueParam.setName(VALUE_ARG_NAME);

        Parameter enabledParam = new Parameter();
        enabledParam.setType("boolean");
        enabledParam.setName(ENABLED_ARG_NAME);
        recordDeclaration.setParameters(new NodeList<>(labelParam, valueParam, enabledParam));
        ConstructorDeclaration constructorDeclaration = recordDeclaration.addConstructor();
        constructorDeclaration.addParameter(labelParam);
        constructorDeclaration.addParameter(valueParam);
        BlockStmt body = constructorDeclaration.getBody();

        body.addStatement(new ExplicitConstructorInvocationStmt(
                true,
                new ThisExpr(),
                new NodeList<>(
                        new NameExpr(LABEL_ARG_NAME), new NameExpr(VALUE_ARG_NAME), new BooleanLiteralExpr(true))));
        Optional<ClassOrInterfaceDeclaration> classOptional =
                compilationUnit.findFirst(ClassOrInterfaceDeclaration.class);
        if (classOptional.isEmpty()) {
            throw new IllegalArgumentException("Could not find the class to add record");
        }
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = classOptional.get();
        classOrInterfaceDeclaration.getMembers().add(recordDeclaration);
        return DEFAULT_LABEL_AND_VALUE_RECORD_NAME;
    }
}
