package com.vaadin.copilot.plugins.accessibilitychecker;

import static com.github.javaparser.StaticJavaParser.parseName;

import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.github.javaparser.ast.expr.Name;
import com.github.javaparser.ast.expr.SingleMemberAnnotationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.plugins.themeeditor.Editor;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.router.PageTitle;
import com.vaadin.flow.server.VaadinSession;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AccessibilityJavaSourceModifier extends Editor {

    private final ProjectManager projectManager;
    private final ErrorHandler errorHandler;
    private final SuccessHandler successHandler;

    public AccessibilityJavaSourceModifier(
            ProjectManager projectManager, ErrorHandler errorHandler, SuccessHandler successHandler) {
        this.projectManager = projectManager;
        this.errorHandler = errorHandler;
        this.successHandler = successHandler;
    }

    public void setPageTitle(DevToolsInterface devToolsInterface, Integer uiId, String pageTitle) {
        assert uiId != null && pageTitle != null;
        runAsAccess(devToolsInterface, (session) -> {
            Component currentView = session.getUIById(uiId).getCurrentView();
            List<Modification> modifications = new ArrayList<>();
            ComponentTracker.Location createLocation = getCreateLocation(currentView);
            File sourceFile = projectManager.getFileForClass(currentView.getClass());
            int sourceOffset = modifyClass(sourceFile, cu -> {
                Optional<ClassOrInterfaceDeclaration> classOrInterfaceDeclaration =
                        cu.getClassByName(currentView.getClass().getSimpleName());
                classOrInterfaceDeclaration.ifPresent(node -> {
                    Name name = parseName(PageTitle.class.getSimpleName());
                    Optional<AnnotationExpr> pageTitleAnnotationOptional = node.getAnnotations().stream()
                            .filter(expr -> name.equals(expr.getName()))
                            .findFirst();
                    SingleMemberAnnotationExpr normalAnnotationExpr =
                            new SingleMemberAnnotationExpr(name, new StringLiteralExpr(pageTitle));
                    if (pageTitleAnnotationOptional.isPresent()) {
                        getLogger().debug("update the page title {}", pageTitle);
                        modifications.add(Modification.remove(pageTitleAnnotationOptional.get()));
                        node.addAnnotation(normalAnnotationExpr);
                        modifications.add(Modification.insertLineBefore(node, normalAnnotationExpr));
                    } else {
                        getLogger().debug("insert the page title {}", pageTitle);

                        node.addAnnotation(normalAnnotationExpr);
                        modifications.add(Modification.addImport(
                                cu, new ImportDeclaration(PageTitle.class.getName(), false, false)));
                        modifications.add(Modification.insertLineBefore(node, normalAnnotationExpr));
                    }
                });
                return modifications;
            });

            if (sourceOffset != 0) {
                ComponentTracker.refreshLocation(createLocation, sourceOffset);
            }
        });
    }

    private void runAsAccess(DevToolsInterface devToolsInterface, Consumer<VaadinSession> runnable) {
        VaadinSession session = getSession();
        session.access(() -> {
            try {
                runnable.accept(session);
                successHandler.sendSuccess(devToolsInterface);
            } catch (Exception ex) {
                getLogger().error("Error during the execution", ex);
                errorHandler.sendError(devToolsInterface, ex.getMessage());
            }
        });
    }

    protected ComponentTracker.Location getCreateLocation(Component c) {
        ComponentTracker.Location location = ComponentTracker.findCreate(c);
        if (location == null) {
            throw new AccessibilityCheckerException("Unable to find the location where the component "
                    + c.getClass().getName() + " was created");
        }
        return location;
    }

    protected VaadinSession getSession() {
        return VaadinSession.getCurrent();
    }

    protected Component getComponent(VaadinSession session, int uiId, int nodeId) {
        Element element = session.findElement(uiId, nodeId);
        Optional<Component> c = element.getComponent();
        if (!c.isPresent()) {
            throw new AccessibilityCheckerException(
                    "Only component locations are tracked. The given node id refers to an element and not a component.");
        }
        return c.get();
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(AccessibilityJavaSourceModifier.class);
    }

    public static interface ErrorHandler {
        void sendError(DevToolsInterface devToolsInterface, String errorMessage);
    }

    public static interface SuccessHandler {
        void sendSuccess(DevToolsInterface devToolsInterface);
    }
}
