package com.vaadin.copilot.plugins.themeeditor;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.plugins.themeeditor.handlers.ClassNameHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.ComponentMetadataHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.LoadPreviewHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.LoadRulesHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.LocalClassNameHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.OpenCssHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.RulesHandler;
import com.vaadin.copilot.plugins.themeeditor.handlers.SetFontHandler;
import com.vaadin.copilot.plugins.themeeditor.messages.BaseResponse;
import com.vaadin.copilot.plugins.themeeditor.messages.ErrorResponse;
import com.vaadin.copilot.plugins.themeeditor.messages.ThemeNotFoundError;
import com.vaadin.copilot.plugins.themeeditor.utils.HasSourceModifier;
import com.vaadin.copilot.plugins.themeeditor.utils.HasThemeModifier;
import com.vaadin.copilot.plugins.themeeditor.utils.MessageHandler;
import com.vaadin.copilot.plugins.themeeditor.utils.ThemeEditorException;
import elemental.json.Json;
import elemental.json.JsonObject;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handler for ThemeEditor debug window communication messages. Responsible for preparing data for
 * {@link com.vaadin.copilot.plugins.themeeditor.ThemeModifier} and {@link JavaSourceModifier}.
 */
public class ThemeEditorMessageHandler implements HasSourceModifier, HasThemeModifier, CopilotCommand {

    private final ObjectMapper objectMapper = new ObjectMapper();

    private final JavaSourceModifier sourceModifier;

    private final ThemeModifier themeModifier;

    private final Set<MessageHandler> handlers = new HashSet<>();

    private boolean missingThemeNotificationSent = false;

    public ThemeEditorMessageHandler(ProjectManager projectManager) {
        this.sourceModifier = new JavaSourceModifier(projectManager);
        ThemeModifier modifier = null;
        if (projectManager.getThemeFolder().isPresent()) {
            modifier = new ThemeModifier(projectManager);
            registerHandlers();
        }
        this.themeModifier = modifier;
    }

    @Override
    public JavaSourceModifier getSourceModifier() {
        return sourceModifier;
    }

    @Override
    public ThemeModifier getThemeModifier() {
        return themeModifier;
    }

    /**
     * Checks if given command can be handled by ThemeEditor.
     *
     * @param command command to be verified if supported
     * @param data data object to be verified if is of proper structure
     * @return true if it can be handled, false otherwise
     */
    public boolean canHandle(String command, JsonObject data) {
        return !missingThemeNotificationSent
                && command != null
                && data != null
                && data.hasKey(KEY_REQ_ID)
                && getHandler(command).isPresent();
    }

    /**
     * Handles debug message command and performs given action.
     *
     * @param command Command name
     * @param data Command data
     * @return response in form of JsonObject
     */
    public BaseResponse handleDebugMessageData(String command, JsonObject data) {
        String requestId = data.getString(KEY_REQ_ID);
        try {
            BaseResponse response = getHandler(command).map(h -> h.handle(data)).orElseGet(ErrorResponse::new);
            response.setReqId(requestId);
            return response;
        } catch (ThemeEditorException ex) {
            getLogger().debug(ex.getMessage(), ex);
            return new ErrorResponse(requestId, ex.getMessage());
        }
    }

    private ErrorResponse themeNotFoundError(JsonObject data) {
        String requestId = data.getString(KEY_REQ_ID);
        return new ThemeNotFoundError(requestId);
    }

    private Optional<MessageHandler> getHandler(String command) {
        return handlers.stream().filter(h -> h.getCommandName().equals(command)).findFirst();
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ThemeEditorMessageHandler.class.getName());
    }

    protected void registerHandlers() {
        this.handlers.add(new ComponentMetadataHandler(this));
        this.handlers.add(new RulesHandler(this));
        this.handlers.add(new LocalClassNameHandler(this, this));
        this.handlers.add(new LoadRulesHandler(this));
        this.handlers.add(new LoadPreviewHandler(this));
        this.handlers.add(new OpenCssHandler(this));
        this.handlers.add(new ClassNameHandler(this));
        this.handlers.add(new SetFontHandler(this));
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {

        if (!data.hasKey(KEY_REQ_ID)) {
            return false;
        }

        // sent notification and do not process other messages
        if (themeModifier == null && !missingThemeNotificationSent) {
            devToolsInterface.send(ThemeEditorCommand.RESPONSE.getValue(), toJsonObject(themeNotFoundError(data)));
            missingThemeNotificationSent = true;
            return true;
        }

        if (!canHandle(command, data)) {
            return false;
        }

        BaseResponse resultData = handleDebugMessageData(command, data);
        devToolsInterface.send(ThemeEditorCommand.RESPONSE.getValue(), toJsonObject(resultData));

        return true;
    }

    private JsonObject toJsonObject(Object object) {
        try {
            String resultAsString = objectMapper.writeValueAsString(object);
            return Json.parse(resultAsString);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }
}
