package com.vaadin.copilot;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.routes.AccessRequirement;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinService;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.auth.AnonymousAllowed;
import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import jakarta.annotation.security.DenyAll;
import jakarta.annotation.security.PermitAll;
import jakarta.annotation.security.RolesAllowed;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

/**
 * Provides server side route information to the client.
 */
public class RouteHandler implements CopilotCommand {
    private final ProjectManager projectManager;
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final VaadinServletContext context;

    /**
     * Creates a new route handler.
     *
     * @param projectManager
     *            the project manager to use
     * @param context
     *            the servlet context to use
     */
    public RouteHandler(ProjectManager projectManager,
            VaadinServletContext context) {
        this.projectManager = projectManager;
        this.context = context;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (command.equals("get-routes")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            boolean securityEnabled = isViewSecurityEnabled();
            returnData.put("securityEnabled", securityEnabled);
            Future<Void> future = VaadinSession.getCurrent().access(() -> {
                RouteRegistry routeRegistry = VaadinService.getCurrent()
                        .getRouter().getRegistry();

                JsonArray serverRoutes = routeRegistry.getRegisteredRoutes()
                        .stream().map(routeData -> {
                            JsonObject route = Json.createObject();
                            Class<? extends Component> target = routeData
                                    .getNavigationTarget();
                            route.put("path",
                                    addInitialSlash(routeData.getTemplate()));
                            route.put("navigationTarget", target.getName());
                            if (securityEnabled) {
                                try {
                                    AccessRequirement req = getAccessRequirement(
                                            target);

                                    route.put("accessRequirement",
                                            JsonUtils.beanToJson(req));
                                } catch (Exception e) {
                                    getLogger().error(
                                            "Unable to determine access requirement",
                                            e);
                                }
                            }
                            route.put("filename", projectManager
                                    .getFileForClass(
                                            routeData.getNavigationTarget())
                                    .getAbsolutePath());
                            return route;
                        }).collect(JsonUtils.asArray());
                returnData.put("routes", serverRoutes);
            });

            try {
                future.get();
                devToolsInterface.send("server-routes", returnData);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            } catch (ExecutionException e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command,
                        returnData, "Error fetching routes", e);
            }

            return true;
        }
        return false;
    }

    private boolean isViewSecurityEnabled() {
        return SpringBridge.isSpringAvailable()
                && SpringBridge.isViewSecurityEnabled(context);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteHandler.class);
    }

    private String addInitialSlash(String template) {
        if (!template.startsWith("/")) {
            return "/" + template;
        }
        return template;
    }

    private AccessRequirement getAccessRequirement(
            Class<?> annotatedClassOrMethod) {
        // Based on AccessAnnotationChecker.hasAccess
        if (annotatedClassOrMethod.isAnnotationPresent(DenyAll.class)) {
            return new AccessRequirement(AccessRequirement.Type.DENY_ALL);
        }
        if (annotatedClassOrMethod
                .isAnnotationPresent(AnonymousAllowed.class)) {
            return new AccessRequirement(
                    AccessRequirement.Type.ANONYMOUS_ALLOWED);
        }
        RolesAllowed rolesAllowed = annotatedClassOrMethod
                .getAnnotation(RolesAllowed.class);
        if (rolesAllowed != null) {
            return new AccessRequirement(AccessRequirement.Type.ROLES_ALLOWED,
                    rolesAllowed.value());
        } else if (annotatedClassOrMethod
                .isAnnotationPresent(PermitAll.class)) {
            return new AccessRequirement(AccessRequirement.Type.PERMIT_ALL);
        }
        return new AccessRequirement(AccessRequirement.Type.DENY_ALL);

    }
}
