package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.experimental.FeatureFlags;
import com.vaadin.flow.server.VaadinContext;
import com.vaadin.pro.licensechecker.BuildType;
import com.vaadin.pro.licensechecker.LicenseChecker;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.ProKey;
import com.vaadin.pro.licensechecker.Product;
import com.vaadin.pro.licensechecker.VaadinComIntegration;
import elemental.json.Json;
import elemental.json.JsonObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles getting information about the user (pro key based) and communicating
 * with the server about feature access.
 */
public class UserInfoHandler implements CopilotCommand {

    public static final String PRODUCT_VAADIN_COPILOT = "vaadin-copilot";
    public static final int LOGIN_TIMEOUT_SECONDS = 15 * 60;
    private final VaadinContext context;

    private final UserInfoServerClient client = new UserInfoServerClient();

    public UserInfoHandler(VaadinContext context) {
        this.context = context;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data,
            DevToolsInterface devToolsInterface) {
        if (command.equals("get-user-info")) {
            ProKey proKey = LocalProKey.get();
            JsonObject userInfo = Json.createObject();
            if (proKey != null) {
                UserInfo info = client.getUserInfo(proKey.getProKey());
                userInfo.put("validLicense", hasValidLicense());
                userInfo.put("vaadiner", info != null && info.vaadiner());
            }

            userInfo.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
            userInfo.put("flowFeatureFlag", FeatureFlags.get(context)
                    .isEnabled(FeatureFlags.COPILOT_FLOW));
            userInfo.put("copilotI18nFeatureFlag", FeatureFlags.get(context)
                    .isEnabled(FeatureFlags.COPILOT_I18N));
            userInfo.put("copilotExperimentalFeatureFlag", FeatureFlags
                    .get(context).isEnabled(FeatureFlags.COPILOT_EXPERIMENTAL));
            devToolsInterface.send("copilot-user-info", userInfo);
            return true;
        } else if (command.equals("log-in")) {
            ProKey proKey = LocalProKey.get();
            JsonObject response = Json.createObject();
            response.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
            if (proKey == null) {
                // Create a url so we can download the pro key. We don't really
                // care if the check succeeds or fails
                // as the client side will recheck the license after the login
                LicenseChecker.checkLicenseAsync(PRODUCT_VAADIN_COPILOT,
                        CopilotVersion.getVersion(), BuildType.DEVELOPMENT,
                        new LicenseChecker.Callback() {
                            @Override
                            public void ok() {
                                devToolsInterface.send(
                                        "copilot-prokey-received",
                                        Json.createObject());
                            }

                            @Override
                            public void failed(Exception e) {
                                getLogger().error("Unable to fetch pro key", e);
                            }
                        }, loginUrl -> {
                            // This is where we always should end up as there is
                            // no pro key
                            response.put("loginUrl", loginUrl);
                            response.put("reason", "no-pro-key");
                            devToolsInterface.send("log-in-resp", response);
                        }, LOGIN_TIMEOUT_SECONDS);
            } else {
                // There is a pro key, so probably the user does not have an
                // appropriate subscription. Use the default URL
                response.put("loginUrl",
                        VaadinComIntegration.constructUrl(null,
                                new Product(PRODUCT_VAADIN_COPILOT,
                                        CopilotVersion.getVersion())));
                response.put("reason", "no-subscription");
                devToolsInterface.send("log-in-resp", response);
            }
            return true;
        }
        return false;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    private boolean hasValidLicense() {
        try {
            LicenseChecker.checkLicense(PRODUCT_VAADIN_COPILOT,
                    CopilotVersion.getVersion(), BuildType.DEVELOPMENT, null);
            return true;
        } catch (LicenseException e) {
            if (!ignoreWhenDoingBackgroundCheck(e)) {
                getLogger().error("Problem checking the subscription status",
                        e);
            }
        }

        return false;
    }

    private boolean ignoreWhenDoingBackgroundCheck(LicenseException e) {
        String msg = e.getMessage();
        if (msg.contains("The provided license key does not allow usage")) {
            // There is a key but it does not allow access
            return true;
        }
        if (msg.contains("could not be reached and no offline key was found")) {
            // Offline
            return true;
        }

        return false;
    }

}
