package com.vaadin.copilot.javarewriter;

import com.vaadin.flow.shared.util.SharedUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.BiFunction;

/**
 * This class contains hacks related to Flow components and how they work.
 */
public final class FlowComponentQuirks {

    private static final String AVATAR_GROUP = "com.vaadin.flow.component.avatar.AvatarGroup";
    private static final String AVATAR_GROUP_ITEM = AVATAR_GROUP
            + ".AvatarGroupItem";
    private static final String MESSAGE_LIST = "com.vaadin.flow.component.messages.MessageList";
    private static final String MESSAGE_LIST_ITEM = "com.vaadin.flow.component.messages.MessageListItem";

    private FlowComponentQuirks() {
        // Utility class
    }

    /**
     * Returns the Java class name for a given HTML tag.
     *
     * @param tag
     *            the HTML tag
     * @return the Java class name
     */
    public static String getClassForTag(String tag) {
        if (tag.equalsIgnoreCase("A")) {
            tag = "Anchor";
        } else if (tag.equalsIgnoreCase("P")) {
            tag = "Paragraph";
        } else if (tag.equalsIgnoreCase("OL")) {
            tag = "OrderedList";
        } else if (tag.equalsIgnoreCase("ul")) {
            tag = "UnorderedList";
        } else if (tag.equalsIgnoreCase("dl")) {
            tag = "DescriptionList";
        } else if (tag.equalsIgnoreCase("dt")) {
            tag = "DescriptionList.Term";
        } else if (tag.equalsIgnoreCase("dd")) {
            tag = "DescriptionList.Description";
        } else if (tag.equalsIgnoreCase("LI")) {
            tag = "ListItem";
        } else if (tag.equalsIgnoreCase("img")) {
            tag = "Image";
        }
        String capitalizedTag = SharedUtil.capitalize(tag);
        String lowercaseTag = tag.toLowerCase(Locale.ENGLISH);
        String lowercaseFirstPart = removeInitialDash(
                SharedUtil.camelCaseToDashSeparated(tag)).split("-", 2)[0]
                .toLowerCase(Locale.ENGLISH);

        String[] probes = new String[] {
                // H1 and other html elements
                "com.vaadin.flow.component.html." + capitalizedTag,
                // Button and other Vaadin components
                "com.vaadin.flow.component." + lowercaseTag + "."
                        + capitalizedTag,
                // AvatarGroup is avatar.AvatarGroup
                "com.vaadin.flow.component." + lowercaseFirstPart + "."
                        + capitalizedTag,
                "com.vaadin.flow.component.orderedlayout." + capitalizedTag,
                "com.vaadin.flow.component.messages." + capitalizedTag,
                "com.vaadin.flow.component.sidenav." + capitalizedTag,
                "com.vaadin.flow.component.textfield." + capitalizedTag,
                "com.vaadin.flow.component.combobox." + capitalizedTag,
                "com.vaadin.flow.component.applayout." + capitalizedTag,
                "com.vaadin.flow.component.charts." + capitalizedTag,
                "com.vaadin.flow.component.tabs." + capitalizedTag, };
        for (String probe : probes) {
            if (exists(probe)) {
                return probe;
            }
        }
        if (tag.equalsIgnoreCase("BoardRow")) {
            return "com.vaadin.flow.component.board.Row";
        }
        throw new IllegalArgumentException(
                "Unable to find Java class for <" + tag + ">");
    }

    private static String removeInitialDash(String s) {
        if (s.startsWith("-")) {
            return s.substring(1);
        }
        return s;
    }

    private static boolean exists(String className) {
        try {
            JavaRewriterUtil.getClass(className);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * Rewrites the value of a property for a specific component type.
     *
     * @param componentType
     *            the component type
     * @param prop
     *            the property name
     * @param value
     *            the property value
     * @return the rewritten value or the original value if no rewriting is
     *         needed
     */
    public static Object componentSpecificValueMapping(Class<?> componentType,
            String prop, Object value) {
        String itemType = getListSetterType(componentType, prop);
        if (itemType != null) {
            return mapToClass((List) value, itemType, null);
        }

        return value;
    }

    private static String getListSetterType(Class<?> componentType,
            String prop) {
        // These could be detected using reflection
        if (componentType.getName().equals(AVATAR_GROUP)
                && prop.equals("items")) {
            return AVATAR_GROUP_ITEM;
        } else if (componentType.getName().equals(MESSAGE_LIST)
                && prop.equals("items")) {
            return MESSAGE_LIST_ITEM;
        }
        return null;
    }

    private static List<JavaRewriter.JavaComponent> mapToClass(
            List<Map<String, Object>> value, String itemClass,
            BiFunction<String, Object, Object> customMapper) {
        return value.stream().map(item -> {
            Map<String, Object> props = new HashMap<>();
            for (Map.Entry<String, Object> entry : item.entrySet()) {
                Object propValue = entry.getValue();
                if (customMapper != null) {
                    propValue = customMapper.apply(entry.getKey(), propValue);
                }
                props.put(entry.getKey(), propValue);
            }

            return new JavaRewriter.JavaComponent(null, itemClass, props,
                    new ArrayList<>());
        }).toList();
    }

    /**
     * Converts a React property name to a Java setter method name.
     * <p>
     * Used to map React properties to Java setters when the react property API
     * does not match the Java API.
     *
     * @param property
     *            the property name
     * @param type
     *            the component type
     * @return the Java setter method name or {@code null} if no mapping is
     *         found
     */
    public static String convertReactPropertyToJavaSetter(String property,
            Class<?> type) {
        if (property.equals("theme")) {
            if (JavaRewriterUtil.hasMethod(type, "setThemeName")) {
                return "setThemeName";
            }
            return "getElement().getThemeList().add";
        }

        String typeName = type.getName();
        if (property.equals("checked") && typeName
                .equals("com.vaadin.flow.component.checkbox.Checkbox")) {
            return "setValue";
        }
        if (property.equals("text") && (typeName
                .equals("com.vaadin.flow.component.tabs.Tab")
                || typeName.equals(
                        "com.vaadin.flow.component.sidenav.SideNavItem"))) {
            return "setLabel";
        }
        if (property.equals("summary") && typeName
                .equals("com.vaadin.flow.component.accordion.AccordionPanel")) {
            return "setSummaryText";
        }
        if (property.equals("img")
                && typeName.equals("com.vaadin.flow.component.avatar.Avatar")) {
            return "setImage";
        }
        if (property.equals("abbr")
                && typeName.equals("com.vaadin.flow.component.avatar.Avatar")) {
            return "setAbbreviation";
        }
        if (property.equals("nodrop")
                && typeName.equals("com.vaadin.flow.component.upload.Upload")) {
            return "setDropAllowed";
        }

        return null;
    }

    public static boolean isInvertedBoolean(String property, Class<?> type) {
        if (property.equals("nodrop") && type.getName()
                .equals("com.vaadin.flow.component.upload.Upload")) {
            return true;
        }
        return false;
    }
}
