package com.vaadin.copilot.javarewriter;

import com.vaadin.copilot.ProjectManager;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.vaadin.copilot.JavaRewriteHandler.UNDO_LABEL;

public class JavaBatchRewriter {
    private final ProjectManager projectManager;
    private final Map<File, JavaRewriter> rewriters = new HashMap<>();
    private final Map<JavaRewriter.ComponentTypeAndSourceLocation, JavaRewriter.ComponentInfo> components;

    public JavaBatchRewriter(ProjectManager projectManager,
            List<JavaRewriter.ComponentTypeAndSourceLocation> componentLocations) {
        this.projectManager = projectManager;

        // Parse all needed Java files
        rewriters.putAll(componentLocations.stream()
                .map(JavaRewriter.ComponentTypeAndSourceLocation::javaFile)
                .distinct().collect(Collectors.toMap(file -> file, file -> {
                    try {
                        return new JavaRewriter(projectManager.readFile(file));
                    } catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                })));

        // Convert line number -> ast nodes
        this.components = componentLocations.stream()
                .collect(Collectors.toMap(component -> component, component -> {
                    JavaRewriter rewriter = rewriters.get(component.javaFile());
                    return rewriter.findComponentInfo(component);
                }));

        // Filter out components which are not in the source
        filterComponents(
                component -> component.getValue().objectCreationExpr() != null);

    }

    private void filterComponents(
            Predicate<? super Map.Entry<JavaRewriter.ComponentTypeAndSourceLocation, JavaRewriter.ComponentInfo>> filter) {
        this.components.entrySet().removeIf(entry -> !filter.test(entry));
    }

    public void deleteAll() {
        forEachComponent((source, component, rewriter) -> component.rewriter()
                .delete(component));
    }

    public void duplicateAll() {
        forEachComponent((source, component, rewriter) -> component.rewriter()
                .duplicate(component));
    }

    public interface Callback {
        void accept(JavaRewriter.ComponentTypeAndSourceLocation source,
                JavaRewriter.ComponentInfo component, JavaRewriter rewriter);
    }

    public void forEachComponent(Callback cb) {
        components.forEach((source, component) -> {
            JavaRewriter rewriter = rewriters.get(source.javaFile());
            cb.accept(source, component, rewriter);
        });
    }

    public Map<File, String> getResults() {
        return rewriters.keySet().stream().collect(Collectors
                .toMap(file -> file, file -> rewriters.get(file).getResult()));
    }

    public void writeResult() {
        getResults().forEach((file, result) -> {
            try {
                projectManager.writeFile(file, UNDO_LABEL, result);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }
}
