package com.vaadin.copilot.javarewriter;

import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.printer.lexicalpreservation.LexicalPreservingPrinter;
import com.github.javaparser.resolution.UnsolvedSymbolException;
import com.github.javaparser.resolution.declarations.ResolvedConstructorDeclaration;
import com.github.javaparser.resolution.types.ResolvedType;
import com.github.javaparser.symbolsolver.JavaSymbolSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.CombinedTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ReflectionTypeSolver;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.shared.util.SharedUtil;
import elemental.json.JsonArray;
import elemental.json.JsonBoolean;
import elemental.json.JsonNull;
import elemental.json.JsonNumber;
import elemental.json.JsonObject;
import elemental.json.JsonString;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Rewrites Java source code to add or replace constructor parameters, method
 * invocations and more.
 */
public class JavaRewriter {

    private final String source;

    /**
     * Information about a component in the source code.
     */
    public record ComponentTypeAndSourceLocation(
            Class<? extends Component> type, File javaFile,
            ComponentTracker.Location createLocation,
            ComponentTracker.Location attachLocation,
            ComponentTypeAndSourceLocation parent,
            List<ComponentTypeAndSourceLocation> children) {
    }

    public record ComponentInfo(Class<? extends Component> type,
            ObjectCreationExpr objectCreationExpr,
            BlockStmt componentCreateScope, MethodCallExpr attachCall,
            BlockStmt componentAttachScope,
            VariableDeclarator localVariableDeclarator,
            AssignExpr assignmentExpression, FieldDeclaration fieldDeclaration,
            FieldDeclaration fieldDeclarationAndAssignment,
            String localVariableName, String fieldName,
            ConstructorDeclaration routeConstructor, JavaRewriter rewriter) {
    }

    /**
     * A code snippet to be inserted into the source code.
     *
     * @param code
     *            the code snippet
     */
    public record Code(String code) {

    }

    /**
     * Holder for a setter name and associated value
     */
    public record SetterAndValue(String setter, Object value) {
    }

    /**
     * Information about extracting an inline variable to local variable
     */
    public record ExtractInlineVariableResult(BlockStmt blockStmt,
            String newVariableName, int index) {
    }

    /**
     * Represents a Java component to be added to the source code.
     *
     * @param tag       the tag of the component, used to determine the Java class
     * @param className the class name if known, or null to use the tag to look up the class
     * @param props     the properties of the component
     * @param children  the child components
     */
    public record JavaComponent(String tag, String className, Map<String, Object> props, List<JavaComponent> children) {

        private static Object propertyValueFromJson(Object jsonValue) {
            if (jsonValue instanceof JsonBoolean jsonboolean) {
                return jsonboolean.getBoolean();
            } else if (jsonValue instanceof JsonString jsonString) {
                return jsonString.getString();
            } else if (jsonValue instanceof JsonNull) {
                return null;
            } else if (jsonValue instanceof JsonNumber number) {
                if (number.asString().equals("" + (int) number.asNumber())) {
                    // Integer
                    return (int) ((JsonNumber) jsonValue).getNumber();
                }
                return number.asNumber();
            } else if (jsonValue instanceof JsonObject jsonObject) {
                Map<String, Object> values = new HashMap<>();
                for (String key : jsonObject.keys()) {
                    values.put(key, propertyValueFromJson(jsonObject.get(key)));
                }
                return values;
            } else if (jsonValue instanceof JsonArray jsonArray) {
                List<Object> values = new ArrayList<>();
                for (int i = 0; i < jsonArray.length(); i++) {
                    values.add(propertyValueFromJson(jsonArray.get(i)));
                }
                return values;
            } else {
                throw new IllegalArgumentException("Unsupported JSON value: " + jsonValue);
            }
        }

        /**
         * Creates a new JavaComponent instance from a JSON object.
         *
         * @param json the JSON object
         * @return the JavaComponent instance
         */
        public static JavaComponent componentFromJson(JsonObject json) {
            String tag = json.hasKey("tag") ? json.getString("tag") : null;
            String className = json.hasKey("className") ? json.getString("className") : null;
            Map<String, Object> props = new HashMap<>();
            JsonObject jsonProps = json.getObject("props");
            if (jsonProps != null) {
                for (String key : jsonProps.keys()) {
                    props.put(key, propertyValueFromJson(jsonProps.get(key)));
                }
            }
            List<JavaComponent> children = new ArrayList<>();
            JsonArray jsonChildren = json.getArray("children");
            if (jsonChildren != null) {
                for (int i = 0; i < jsonChildren.length(); i++) {
                    JavaComponent child = JavaComponent.componentFromJson(jsonChildren.getObject(i));
                    if ("prefix".equals(child.props.get("slot"))) {
                        child.props.remove("slot");
                        props.put("prefixComponent", child);
                    } else if ("add-button".equals(child.props.get("slot"))) {
                        child.props.remove("slot");
                        props.put("uploadButton", child);
                    } else {
                        children.add(child);
                    }
                }
            }
            return new JavaComponent(tag, className, props, children);
        }

        /**
         * Creates a new JavaComponent instance from a JSON array.
         *
         * @param template the JSON array
         * @return the JavaComponent instances
         */
        public static List<JavaRewriter.JavaComponent> componentsFromJson(
                JsonArray template) {
            return JsonUtils.stream(template).map(JsonObject.class::cast)
                    .map(JavaRewriter.JavaComponent::componentFromJson).toList();
        }
    }

    /**
     * Where to add a component
     */
    public enum Where {
        BEFORE, APPEND
    }

    public enum AlignmentMode {
        ALIGN_ITEMS, SELF_HORIZONTALLY, SELF_VERTICALLY
    }

    /**
     * Represents a point in the source code where new code can be inserted.
     */
    public static class InsertionPoint {
        private final BlockStmt block;
        private int index;

        /**
         * Creates a new InsertionPoint instance.
         *
         * @param block
         *            the block where to insert the code
         * @param index
         *            the index where to insert the code
         */
        public InsertionPoint(BlockStmt block, int index) {
            this.block = block;
            this.index = index;
        }

        /**
         * Returns a free variable name based on the given base name, available
         * in the scope where code will be inserted.
         *
         * @param baseName
         *            the base name for the variable
         * @return a free variable name
         */
        public String getFreeVariableName(String baseName) {
            return JavaRewriterUtil.findFreeVariableName(baseName, block);
        }

        /**
         * Adds a statement to the insertion point.
         *
         * @param statement
         *            the statement to add
         */
        public void add(Statement statement) {
            block.addStatement(index++, statement);
        }

        public BlockStmt getBlock() {
            return block;
        }

        public int getIndex() {
            return index;
        }
    }

    private final ParserConfiguration parserConfiguration = new ParserConfiguration();
    protected CompilationUnit compilationUnit;

    /**
     * Creates a new JavaRewriter instance.
     *
     * @param source
     *            the Java source code to rewrite
     */
    public JavaRewriter(String source) {
        this.source = source;
        parseSource(source);
    }

    private void parseSource(String source) {
        parserConfiguration
                .setLanguageLevel(ParserConfiguration.LanguageLevel.JAVA_17);
        CombinedTypeSolver combinedTypeSolver = new CombinedTypeSolver();
        combinedTypeSolver.add(new ReflectionTypeSolver(false));
        JavaSymbolSolver symbolSolver = new JavaSymbolSolver(
                combinedTypeSolver);
        parserConfiguration.setSymbolResolver(symbolSolver);

        StaticJavaParser.setConfiguration(parserConfiguration);
        this.compilationUnit = LexicalPreservingPrinter
                .setup(StaticJavaParser.parse(source));
    }

    public String getResult() {
        return LexicalPreservingPrinter.print(compilationUnit);
    }

    public int getFirstModifiedRow() {
        int row = 1;
        for (int i = 0; i < source.length(); i++) {
            if (source.charAt(i) != getResult().charAt(i)) {
                return row;
            }
            if (source.charAt(i) == '\n') {
                row++;
            }
        }
        return -1;
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) or a function call in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to replace or add, if the constructor
     *            parameter is not found
     * @param value
     *            the new value for the constructor parameter or function call
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    public boolean replaceFunctionCall(ComponentInfo componentInfo,
            String function, Object value) {
        if (replaceConstructorParam(componentInfo, function, value)) {
            return true;
        }
        return replaceOrAddCall(componentInfo, function, value);
    }

    /**
     * Replaces a constructor parameter (if it is mapped to the given setter
     * function) in the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the setter function
     * @param value
     *            the new value for the constructor parameter
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    private boolean replaceConstructorParam(ComponentInfo componentInfo,
            String function, Object value) {
        ObjectCreationExpr objectCreationExpr = componentInfo
                .objectCreationExpr();

        // Find constructor based on number of arguments
        Optional<Constructor<?>> constructor = findConstructor(
                componentInfo.type(), objectCreationExpr);
        Optional<Map<Integer, String>> mappings = constructor
                .map(c -> ConstructorAnalyzer.get().getMappings(c));
        if (mappings.isPresent()) {
            Optional<Map.Entry<Integer, String>> mapping = mappings.get()
                    .entrySet().stream()
                    .filter(entry -> entry.getValue().equals(function))
                    .findFirst();
            if (mapping.isPresent()) {
                objectCreationExpr.setArgument(mapping.get().getKey(),
                        JavaRewriterUtil.toExpression(value));
                return true;
            }
        }
        return false;
    }

    private String getMappedProperty(Constructor<?> c, int propertyIndex) {
        return ConstructorAnalyzer.get().getMappings(c).entrySet().stream()
                .filter(entry -> entry.getKey() == propertyIndex)
                .map(Map.Entry::getValue).findFirst().orElse(null);
    }

    private Optional<Constructor<?>> findConstructor(
            Class<? extends Component> componentType,
            ObjectCreationExpr objectCreationExpr) {
        try {
            ResolvedConstructorDeclaration resolved = objectCreationExpr
                    .resolve();

            List<Constructor<?>> candidates = Arrays
                    .stream(componentType.getDeclaredConstructors())
                    .filter(c -> c.getParameterCount() == resolved
                            .getNumberOfParams())
                    .filter(c -> {
                        for (int i = 0; i < resolved.getNumberOfParams(); i++) {
                            ResolvedType paramType = resolved.getParam(i)
                                    .getType();
                            if (!JavaRewriterUtil.typesEqual(paramType,
                                    c.getParameterTypes()[i])) {
                                return false;
                            }
                        }
                        return true;
                    }).toList();
            if (candidates.isEmpty()) {
                return Optional.empty();
            } else if (candidates.size() > 1) {
                throw new IllegalStateException(
                        "Multiple constructors found for "
                                + componentType.getName() + ": " + candidates);
            } else {
                return Optional.of(candidates.get(0));
            }
        } catch (UnsolvedSymbolException e) {
            throw new IllegalStateException("Unable to resolve constructor for "
                    + componentType.getName() + ": " + objectCreationExpr, e);
        }
    }

    /**
     * Adds a function call to the source code.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function to add
     * @param parameters
     *            parameters for the function
     * @return {@code true} if the addition was successful, {@code false}
     *         otherwise
     */
    public boolean addCall(ComponentInfo componentInfo, String function,
            Object... parameters) {
        return doReplaceOrAddCall(componentInfo, function, false, parameters);
    }

    /**
     * Replaces a function call in the source code, if found, otherwise adds the
     * function call.
     *
     * @param componentInfo
     *            the component to modify
     * @param function
     *            the name of the function call to add or replace
     * @param parameters
     *            new parameters for the function
     * @return {@code true} if the replacement was successful, {@code false}
     *         otherwise
     */
    public boolean replaceOrAddCall(ComponentInfo componentInfo,
            String function, Object... parameters) {
        return doReplaceOrAddCall(componentInfo, function, true, parameters);
    }

    private boolean doReplaceOrAddCall(ComponentInfo componentInfo,
            String function, boolean replace, Object... parameters) {
        List<Expression> parameterExpressions = new ArrayList<>();
        for (Object parameter : parameters) {
            parameterExpressions.add(JavaRewriterUtil.toExpression(parameter));
        }

        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(componentInfo);

        MethodCallExpr functionCall = functionCalls.stream()
                .filter(m -> m.getNameAsString().equals(function)).findFirst()
                .orElse(null);
        if (replace && functionCall != null) {
            if (parameterExpressions.size() == 1) {
                functionCall.setArgument(0, parameterExpressions.get(0));
            }

            return true;
        }

        if (!functionCalls.isEmpty() && JavaRewriterUtil
                .addAfterLastFunctionCall(functionCalls, function,
                        parameterExpressions.toArray(new Expression[0]))) {
            return true;
        }

        int addAfterStatementIndex = -1;
        BlockStmt scope = componentInfo.componentCreateScope;

        if (componentInfo.localVariableDeclarator != null) {
            // Local variable but no calls found, add after the declaration
            addAfterStatementIndex = JavaRewriterUtil.findBlockStatementIndex(
                    componentInfo.localVariableDeclarator);
        } else if (componentInfo.fieldDeclaration != null
                && componentInfo.fieldDeclarationAndAssignment == null) {
            // Field defined but assignment happens elsewhere -> add after that
            // assignment
            addAfterStatementIndex = JavaRewriterUtil.findBlockStatementIndex(
                    componentInfo.assignmentExpression);
        } else if (componentInfo.fieldDeclarationAndAssignment != null) {
            // Field defined and assigned in the same place -> add before attach
            addAfterStatementIndex = JavaRewriterUtil
                    .findBlockStatementIndex(componentInfo.attachCall) - 1;
            scope = componentInfo.componentAttachScope;
        }

        if (addAfterStatementIndex >= 0) {
            NameExpr variable = new NameExpr(
                    JavaRewriterUtil.getFieldOrVariableName(componentInfo));
            MethodCallExpr newCall = new MethodCallExpr(variable, function);
            parameterExpressions.forEach(newCall::addArgument);
            scope.addStatement(addAfterStatementIndex + 1,
                    new ExpressionStmt(newCall));
            return true;
        }

        if (JavaRewriterUtil.inlineAssignment(componentInfo)) {
            // Inline assignment, like add(new Button("foo"))
            // Extract to a local variable, add the call and replace the inline
            // add
            ExtractInlineVariableResult extractInlineVariableResult = JavaRewriterUtil
                    .extractInlineVariableToLocalVariable(componentInfo);
            if (extractInlineVariableResult != null) {

                // input.setFoo("bar")
                MethodCallExpr newCall = new MethodCallExpr(
                        new NameExpr(
                                extractInlineVariableResult.newVariableName()),
                        function);
                parameterExpressions.forEach(newCall::addArgument);

                extractInlineVariableResult.blockStmt.addStatement(
                        extractInlineVariableResult.index + 1, newCall);
                return true;

            }
        }
        throw new IllegalStateException(
                "Unable to determine where to add function call");
    }

    /**
     * Gets the (active) value of a property of a component.
     * <p>
     * The property value is determined by looking for a setter method call in
     * the source code. If the property is not set using a setter, the
     * constructor is checked.
     * <p>
     * If the property is not set using a setter or in the constructor,
     * {@code null} is returned.
     * <p>
     * If the property is set using a method call, the method call expression is
     * returned.
     *
     * @param componentInfo
     *            the component to get the property value from
     * @param property
     *            the property name
     * @return the property value, or null if the property is not set
     */
    public Object getPropertyValue(ComponentInfo componentInfo,
            String property) {
        String setterName = JavaRewriterUtil.getSetterName(property,
                componentInfo.type(), false);
        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(componentInfo);
        List<MethodCallExpr> candidates = functionCalls.stream()
                .filter(m -> m.getNameAsString().equals(setterName)).toList();
        if (!candidates.isEmpty()) {
            // If there would be multiple calls, the last one is the effective
            // one
            MethodCallExpr setterCall = candidates.get(candidates.size() - 1);
            Expression arg = setterCall.getArguments().get(0);
            return JavaRewriterUtil.fromExpression(arg,
                    setterCall.resolve().getParam(0).getType());
        }

        // Try constructor
        ObjectCreationExpr createExpression = componentInfo
                .objectCreationExpr();
        if (createExpression != null) {
            Optional<Constructor<?>> maybeConstructor = findConstructor(
                    componentInfo.type(), createExpression);
            if (maybeConstructor.isPresent()) {
                Constructor<?> constructor = maybeConstructor.get();
                for (int i = 0; i < constructor.getParameterCount(); i++) {
                    String mappedProperty = getMappedProperty(constructor, i);
                    if (setterName.equals(mappedProperty)) {
                        return JavaRewriterUtil.fromExpression(
                                createExpression.getArgument(i),
                                createExpression.resolve().getParam(i)
                                        .getType());
                    }
                }
            }
        }
        return null;
    }

    /**
     * Finds a component in the source code.
     * <p>
     * Note that this will create a constructor if the component is a route
     * class and there are no constructors.
     *
     * @param typeAndSourceLocation
     *            the type and source location of the component
     * @return the component info
     */
    public ComponentInfo findComponentInfo(
            ComponentTypeAndSourceLocation typeAndSourceLocation) {
        List<ObjectCreationExpr> objectCreationExprs = JavaRewriterUtil
                .findNodesOfType(compilationUnit,
                        typeAndSourceLocation.createLocation().lineNumber(),
                        ObjectCreationExpr.class,
                        node -> node.getType().asClassOrInterfaceType()
                                .getName().asString()
                                .equals(typeAndSourceLocation.type()
                                        .getSimpleName()));

        if (objectCreationExprs.size() > 1) {
            throw new IllegalArgumentException(
                    "There are multiple components created on the given line and we are unable to determine which one to modify");
        }

        Optional<ObjectCreationExpr> objectCreationExpr = Optional
                .ofNullable(objectCreationExprs.isEmpty() ? null
                        : objectCreationExprs.get(0));

        boolean routeClass = false;
        if (objectCreationExpr.isEmpty()) {
            routeClass = isRouteClass(typeAndSourceLocation);
            if (!routeClass) {
                throw new IllegalArgumentException(
                        "Constructor not found at the expected location: "
                                + typeAndSourceLocation.createLocation());
            }
        }
        Optional<BlockStmt> componentCreateScope = objectCreationExpr
                .map(JavaRewriterUtil::findBlock);

        Optional<MethodCallExpr> attachCall = JavaRewriterUtil.findNodeOfType(
                compilationUnit,
                typeAndSourceLocation.attachLocation().lineNumber(),
                MethodCallExpr.class);
        if (attachCall.isEmpty() && !routeClass) {
            throw new IllegalArgumentException(
                    "Attach not found at the expected location: "
                            + typeAndSourceLocation.attachLocation());
        }
        Optional<BlockStmt> componentAttachScope = attachCall
                .map(JavaRewriterUtil::findBlock);

        Optional<VariableDeclarator> localVariableDeclarator = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr,
                        VariableDeclarator.class));
        Optional<AssignExpr> assignmentExpression = objectCreationExpr.map(
                expr -> JavaRewriterUtil.findAncestor(expr, AssignExpr.class));
        Optional<FieldDeclaration> fieldDeclarationAndAssignment = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr,
                        FieldDeclaration.class));
        Optional<FieldDeclaration> fieldDeclaration = fieldDeclarationAndAssignment;
        if (localVariableDeclarator.isPresent()
                && fieldDeclarationAndAssignment.isPresent()) {
            // A variable declarator is found also for assignments for fields
            // but we want to differentiate between the two cases later on
            localVariableDeclarator = Optional.empty();
        }

        String localVariableName = null;
        String fieldName = null;
        if (localVariableDeclarator.isPresent()) {
            // TextField foo = new TextField();
            localVariableName = localVariableDeclarator.get().getNameAsString();
        } else if (fieldDeclarationAndAssignment.isPresent()) {
            // private TextField foo = new TextField();
            fieldName = fieldDeclarationAndAssignment.get().getVariable(0)
                    .getNameAsString();
        } else if (assignmentExpression.isPresent()) {
            // foo = new TextField();
            // Here foo can be a local variable or field
            String localVariableOrFieldName = assignmentExpression.get()
                    .getTarget().asNameExpr().getNameAsString();
            if (componentCreateScope.isPresent() && JavaRewriterUtil
                    .findLocalVariableDeclarator(localVariableOrFieldName,
                            componentCreateScope.get()) != null) {
                localVariableName = localVariableOrFieldName;
            } else {
                fieldName = localVariableOrFieldName;
                fieldDeclaration = Optional
                        .ofNullable(JavaRewriterUtil.findFieldDeclaration(
                                assignmentExpression.get(), fieldName));
            }
        }
        Optional<ConstructorDeclaration> routeConstructor = Optional.empty();
        if (routeClass) {
            routeConstructor = JavaRewriterUtil.findNodeOfType(compilationUnit,
                    typeAndSourceLocation.createLocation().lineNumber(),
                    ConstructorDeclaration.class);

            if (routeConstructor.isEmpty()) {
                // Potentially no constructor
                List<ConstructorDeclaration> constructors = compilationUnit
                        .findAll(ConstructorDeclaration.class);
                if (constructors.isEmpty()) {
                    // Need to create a constructor
                    routeConstructor = Optional.of(new ConstructorDeclaration()
                            .setName(typeAndSourceLocation.type()
                                    .getSimpleName())
                            .setPublic(true));
                    String className = typeAndSourceLocation.type()
                            .getSimpleName();
                    ClassOrInterfaceDeclaration classDeclaration = compilationUnit
                            .findAll(ClassOrInterfaceDeclaration.class).stream()
                            .filter(c -> c.getNameAsString().equals(className))
                            .findFirst()
                            .orElseThrow(() -> new IllegalArgumentException(
                                    "Class " + className + "not found"));
                    classDeclaration.addMember(routeConstructor.get());
                } else {
                    throw new IllegalArgumentException(
                            "Route class has constructors but none at the expected location: "
                                    + typeAndSourceLocation.createLocation());
                }
            }
        }
        return new ComponentInfo(typeAndSourceLocation.type(),
                objectCreationExpr.orElse(null),
                componentCreateScope.orElse(null), attachCall.orElse(null),
                componentAttachScope.orElse(null),
                localVariableDeclarator.orElse(null),
                assignmentExpression.orElse(null),
                fieldDeclaration.orElse(null),
                fieldDeclarationAndAssignment.orElse(null), localVariableName,
                fieldName, routeConstructor.orElse(null), this);
    }

    private boolean isRouteClass(
            ComponentTypeAndSourceLocation typeAndSourceLocation) {
        if (!typeAndSourceLocation.createLocation
                .equals(typeAndSourceLocation.attachLocation)) {
            return false;
        }
        return typeAndSourceLocation.attachLocation().methodName()
                .equals("<init>");
    }

    /**
     * Deletes a component from the source code.
     *
     * @param componentInfo
     *            the component to delete
     * @return {@code true} if the deletion was successful, {@code false}
     *         otherwise
     */
    public boolean delete(ComponentInfo componentInfo) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(componentInfo);
        List<MethodCallExpr> otherMethodCalls = JavaRewriterUtil
                .findMethodCallNonStatements(componentInfo);

        functionCalls.forEach(expr -> JavaRewriterUtil
                .findAncestorOrThrow(expr, Statement.class).remove());
        if (componentInfo.fieldDeclaration != null) {
            componentInfo.fieldDeclaration.remove();
            if (componentInfo.fieldDeclarationAndAssignment == null) {
                // Instance created elsewhere, i.e.
                // TextField foo;
                // ...
                // foo = new TextField();
                JavaRewriterUtil
                        .removeStatement(componentInfo.objectCreationExpr());
            }
        } else if (componentInfo.localVariableDeclarator != null) {
            JavaRewriterUtil
                    .removeStatement(componentInfo.localVariableDeclarator);

        }

        otherMethodCalls.forEach(methodCall -> {
            if (!JavaRewriterUtil.removeFromStringConcatenation(methodCall)) {
                JavaRewriterUtil.removeStatement(methodCall);
            }
        });

        removeAttachCall(componentInfo);

        List<Expression> parameterUsage = JavaRewriterUtil
                .findParameterUsage(componentInfo);

        parameterUsage.forEach(expr -> {
            // This might find usage that has been handled in a better way
            // earlier on, like string concatenation
            if (JavaRewriterUtil.hasAncestor(expr, Statement.class)) {
                JavaRewriterUtil.removeStatement(expr);
            }

        });
        return true;
    }

    private Optional<Expression> removeAttachCall(ComponentInfo componentInfo) {
        Optional<Expression> addArgument = JavaRewriterUtil
                .getAttachArgument(componentInfo);
        addArgument.ifPresent(Expression::remove);
        if (componentInfo.attachCall() != null
                && componentInfo.attachCall().getArguments().isEmpty()) {
            JavaRewriterUtil.removeStatement(componentInfo.attachCall());
        }
        return addArgument;
    }

    /**
     * Moves a component in the source code.
     *
     * @param component
     *            the component to move
     * @param container
     *            the new container for the component, if where is Where.APPEND.
     * @param reference
     *            the reference component to move the component before, if where
     *            is Where.BEFORE.
     * @param where
     *            where to move the component
     */
    public void moveComponent(ComponentInfo component, ComponentInfo container,
            ComponentInfo reference, Where where) {
        if (container == null) {
            throw new IllegalArgumentException(
                    "Container component must be non-null");
        }
        if (component.equals(container) || component.equals(reference)
                || container.equals(reference)) {
            throw new IllegalArgumentException(
                    "Component, container and reference must be different");
        }
        removeAttachCall(component);

        List<MethodCallExpr> containerFunctionCalls = JavaRewriterUtil
                .findMethodCallStatements(container);
        String componentFieldOrVariableName = JavaRewriterUtil
                .getFieldOrVariableName(component);
        // For inline we add the full expression, otherwise the name reference
        var toAdd = componentFieldOrVariableName == null
                ? component.objectCreationExpr()
                : new NameExpr(componentFieldOrVariableName);

        if (where == Where.APPEND) {
            if (reference != null) {
                throw new IllegalArgumentException(
                        "Reference component must be null when appending");
            }

            // Find the last add statement for the container and then add the
            // component after that
            boolean added = JavaRewriterUtil.addAfterLastFunctionCall(
                    containerFunctionCalls, "add", toAdd);
            if (!added) {
                throw new IllegalArgumentException(
                        "No add call found for the container component");
            }
        } else if (where == Where.BEFORE) {
            if (reference == null) {
                throw new IllegalArgumentException(
                        "Reference component must be non-null when moving before");
            }

            Optional<Expression> referenceAddArgument = JavaRewriterUtil
                    .findReference(reference.attachCall().getArguments(),
                            reference);

            if (referenceAddArgument.isPresent()) {
                int refAddIndex = reference.attachCall().getArguments()
                        .indexOf(referenceAddArgument.get());
                reference.attachCall().getArguments().add(refAddIndex, toAdd);

                // We can now end up with having "add" before the component
                // constructor and setters, e.g.

                // Input input1 = new Input();
                // add(input3, input1);
                // Input input2 = new Input();
                // add(input2);
                // Input input3 = new Input();

                // We move the add call down after all functions
                // BUT there can also be add calls between the current
                // add location and where it should end up.
                // If so, they need to be moved also
                // We then get

                // Input input1 = new Input();
                // Input input2 = new Input();
                // Input input3 = new Input();
                // add(input3, input1);
                // add(input2);

                List<MethodCallExpr> componentFunctionCalls = JavaRewriterUtil
                        .findMethodCallStatements(component);
                if (!componentFunctionCalls.isEmpty()) {
                    MethodCallExpr lastCall = componentFunctionCalls
                            .get(componentFunctionCalls.size() - 1);
                    BlockStmt insertBlock = JavaRewriterUtil
                            .findAncestorOrThrow(lastCall, BlockStmt.class);
                    int finalAddLocation = JavaRewriterUtil
                            .findBlockStatementIndex(lastCall) + 1;

                    int attachCallIndex = JavaRewriterUtil
                            .findBlockStatementIndex(reference.attachCall());

                    List<MethodCallExpr> allAddCalls = containerFunctionCalls
                            .stream()
                            .filter(m -> m.getNameAsString().equals(reference
                                    .attachCall().getName().asString()))
                            .toList();
                    for (MethodCallExpr addCall : allAddCalls) {
                        int addCallIndex = JavaRewriterUtil
                                .findBlockStatementIndex(addCall);
                        if (addCallIndex >= attachCallIndex
                                && addCallIndex < finalAddLocation) {
                            Statement statement = JavaRewriterUtil
                                    .findAncestorOrThrow(addCall,
                                            Statement.class);
                            statement.remove();
                            insertBlock.addStatement(finalAddLocation - 1,
                                    statement);
                        }
                    }
                }

            } else {
                throw new IllegalArgumentException(
                        "Reference component not found in the add call");
            }
        } else {
            throw new IllegalArgumentException("Unknown where: " + where);
        }
    }

    /**
     * Duplicates a component in the source code.
     *
     * @param component
     *            the component to duplicate
     */

    public void duplicate(ComponentInfo component) {
        if (component.routeConstructor() != null) {
            throw new IllegalArgumentException(
                    "Cannot duplicate a route class");
        }

        // Insert new component after the original component
        InsertionPoint insertionPoint = findInsertionPointForAppend(component);

        String duplicatedName = null;
        if (component.localVariableName() != null) {
            duplicatedName = JavaRewriterUtil.findFreeVariableName(
                    component.localVariableName(), insertionPoint.block);
            VariableDeclarator newLocalVariable = JavaRewriterUtil
                    .clone(component.localVariableDeclarator());
            newLocalVariable.setName(duplicatedName);
            insertionPoint.add(new ExpressionStmt(
                    new VariableDeclarationExpr(newLocalVariable)));
        } else if (component.fieldName() != null) {
            duplicatedName = JavaRewriterUtil.findFreeVariableName(
                    component.fieldName(), insertionPoint.block);
            FieldDeclaration newField;
            if (component.fieldDeclarationAndAssignment() != null) {
                newField = JavaRewriterUtil
                        .clone(component.fieldDeclarationAndAssignment());
            } else {
                newField = JavaRewriterUtil.clone(component.fieldDeclaration());

                // Assignment, e.g. button = new Button();
                AssignExpr newAssignment = JavaRewriterUtil
                        .clone(component.assignmentExpression());
                newAssignment.setTarget(new NameExpr(duplicatedName));
                insertionPoint.add(new ExpressionStmt(newAssignment));
            }
            newField.getVariable(0).setName(duplicatedName);
            JavaRewriterUtil.addFieldAfter(newField,
                    component.fieldDeclaration());
        } else {
            // Inline

        }

        // Duplicate method calls
        if (duplicatedName != null) {
            List<MethodCallExpr> calls = JavaRewriterUtil
                    .findMethodCallStatements(component);
            for (MethodCallExpr call : calls) {
                MethodCallExpr newCall = JavaRewriterUtil.clone(call);
                JavaRewriterUtil.setNameExprScope(newCall,
                        new NameExpr(duplicatedName));
                insertionPoint.add(new ExpressionStmt(newCall));
            }

            List<Expression> parameterUsages = JavaRewriterUtil
                    .findParameterUsage(component);
            // excluding attach call
            Optional<Range> attachCallRangeOptional = component.attachCall
                    .getRange();
            if (attachCallRangeOptional.isPresent()) {
                Range attachRange = attachCallRangeOptional.get();
                parameterUsages = parameterUsages.stream()
                        .filter(parameter -> parameter.getRange().isPresent()
                                && !parameter.getRange().get()
                                        .overlapsWith(attachRange))
                        .toList();
            }
            for (Expression parameterUsage : parameterUsages) {
                MethodCallExpr methodCallExpr = JavaRewriterUtil
                        .findAncestorOrThrow(parameterUsage,
                                MethodCallExpr.class);
                int argumentPosition = methodCallExpr
                        .getArgumentPosition(parameterUsage);
                boolean addAsArg = false;
                Optional<Expression> scope = methodCallExpr.getScope();
                if (scope.isEmpty() || scope.get().isThisExpr()) {
                    Optional<String> classNameOpt = JavaRewriterUtil
                            .findAncestor(parameterUsage,
                                    ClassOrInterfaceDeclaration.class)
                            .getFullyQualifiedName();
                    if (classNameOpt.isEmpty()) {
                        throw new IllegalArgumentException(
                                "Could not find source class");
                    }
                    addAsArg = JavaRewriterUtil.isArrayArgument(
                            classNameOpt.get(),
                            methodCallExpr.getNameAsString(), argumentPosition);
                }
                if (addAsArg) {
                    methodCallExpr.getArguments().add(argumentPosition + 1,
                            new NameExpr(duplicatedName));
                } else {
                    MethodCallExpr clone = JavaRewriterUtil
                            .clone(methodCallExpr);
                    clone.setArgument(argumentPosition,
                            new NameExpr(duplicatedName));
                    insertionPoint.add(new ExpressionStmt(clone));
                }

            }

        }
        // Attach the new component
        Expression componentAddArgument = JavaRewriterUtil
                .getAttachArgumentOrThrow(component);

        int addIndex = component.attachCall()
                .getArgumentPosition(componentAddArgument) + 1;
        if (duplicatedName != null) {
            component.attachCall().getArguments().add(addIndex,
                    new NameExpr(duplicatedName));
        } else {
            component.attachCall().getArguments().add(addIndex,
                    JavaRewriterUtil.clone(component.objectCreationExpr()));

        }

    }

    /**
     * Adds the given code snippet to the source code either before the
     * reference component (Where.BEFORE) or by appending to the layout
     * (Where.APPEND).
     *
     * @param referenceComponent
     *            the reference component to add the code before, or null if the
     *            code should be appended to the layout
     * @param layout
     *            the layout to append the code to
     * @param where
     *            where to add the code
     * @param template
     *            the code to add, as JSON array of objects with "tag", "props"
     *            and "children"
     */
    public void addComponentUsingTemplate(ComponentInfo referenceComponent,
            ComponentInfo layout, Where where, List<JavaComponent> template) {

        if (where == Where.APPEND) {
            if (referenceComponent != null || layout == null) {
                throw new IllegalArgumentException(
                        "Reference component must be null and layout must be non-null when appending");
            }
            //
            // HorizontalLayout layout = new HorizontalLayout(button);
            // layout.setThis();
            // layout.addThat();
            // ...
            // add(layout);
            InsertionPoint insertionPoint = findInsertionPointForAppend(layout);
            if (layout.routeConstructor() != null) {
                createComponentStatements(insertionPoint, null, template,
                        "this", null);
            } else {
                createComponentStatements(insertionPoint, null, template,
                        JavaRewriterUtil.getFieldOrVariableName(layout), null);
            }

        } else if (where == Where.BEFORE) {
            // HorizontalLayout hl = new HorizontalLayout();
            // button = new Button("Hello");
            // button.setThis();
            // button.addThat();
            // hl.add(button);
            //
            // Insert constructor + setters before the reference constructor and
            // add the component so that it is added right before the reference
            BlockStmt insertBlock = referenceComponent.componentCreateScope();
            int insertIndex = JavaRewriterUtil.findBlockStatementIndex(
                    referenceComponent.objectCreationExpr());
            InsertionPoint insertionPoint = new InsertionPoint(insertBlock,
                    insertIndex);
            createComponentStatements(insertionPoint, null, template,
                    JavaRewriterUtil.getFieldOrVariableName(referenceComponent),
                    referenceComponent);

        } else {
            throw new IllegalArgumentException("Unknown where: " + where);
        }

    }

    private InsertionPoint findInsertionPointForAppend(
            JavaRewriter.ComponentInfo component) {
        List<MethodCallExpr> functionCalls = JavaRewriterUtil
                .findMethodCallStatements(component);
        if (!functionCalls.isEmpty()) {
            // There are some component.setThis() or component.somethingElse()
            // calls, add a new component after those
            MethodCallExpr lastCall = functionCalls
                    .get(functionCalls.size() - 1);
            return JavaRewriterUtil.findLocationAfter(lastCall);
        } else if (component.routeConstructor() != null) {
            return JavaRewriterUtil
                    .findLocationAtEnd(component.routeConstructor().getBody());
        }

        // There are no component.anything() calls, add before the
        // add(component) call
        return JavaRewriterUtil.findLocationBefore(component.attachCall());
    }

    public void setAlignment(ComponentInfo component,
            AlignmentMode alignmentMode, boolean selected,
            List<String> lumoClasses) {
        String[] lumoUtilityInnerClassNames = alignmentMode == JavaRewriter.AlignmentMode.ALIGN_ITEMS
                ? new String[] { "AlignItems", "JustifyContent" }
                : new String[] { "AlignSelf" };
        LumoRewriterUtil.removeClassNameArgs(component,
                lumoUtilityInnerClassNames);
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);
        if (selected) {
            List<Expression> parameterToAddList = new ArrayList<>();
            for (String lumoUtilityInnerClassName : lumoUtilityInnerClassNames) {
                parameterToAddList
                        .addAll(LumoRewriterUtil.getLumoMethodArgExpressions(
                                lumoUtilityInnerClassName, lumoClasses));
            }
            boolean added = LumoRewriterUtil.addClassNameWithArgs(component,
                    parameterToAddList);
            if (!added) {
                replaceOrAddCall(component, "addClassNames",
                        parameterToAddList.toArray(new Object[0]));
            }
        }
    }

    /**
     * Sets gap to selected component
     *
     * @param component
     *            component to set gap
     * @param newValue
     *            lumo utility gap variable literal.
     */
    public void setGap(ComponentInfo component, String newValue) {
        LumoRewriterUtil.removeClassNameArgs(component, "Gap", "Gap.Column",
                "Gap.Row");
        LumoRewriterUtil.addLumoUtilityImport(compilationUnit);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil
                .findMethodCallStatements(component);
        methodCallStatements.stream()
                .filter(methodCallExpr -> StringUtils
                        .equals(methodCallExpr.getNameAsString(), "setSpacing"))
                .findAny().ifPresent(JavaRewriterUtil::removeStatement);
        LumoRewriterUtil.removeThemeArgStartsWith(methodCallStatements,
                "spacing");

        if (StringUtils.isNotEmpty(newValue)) {
            List<Expression> gapExpressions = LumoRewriterUtil
                    .getLumoMethodArgExpressions("Gap", List.of(newValue));
            boolean added = LumoRewriterUtil.addClassNameWithArgs(component,
                    gapExpressions);
            if (!added) {
                replaceOrAddCall(component, "addClassNames",
                        gapExpressions.toArray(new Object[0]));
            }
        } else {
            boolean added = JavaRewriterUtil.addAfterLastFunctionCall(
                    methodCallStatements, "setSpacing",
                    new BooleanLiteralExpr(false));
            if (!added) {
                replaceOrAddCall(component, "setSpacing", false);
            }

        }
    }

    /**
     * Merges all the components and wraps them using the given component and
     * places the result in place of the first component.
     *
     * @param components
     *            The components to merge. The first component will be replaced
     *            with the wrapper component
     * @param wrapperComponent
     *            The component to wrap the merged components in.
     */
    public void mergeAndReplace(List<ComponentInfo> components,
            JavaComponent wrapperComponent) {
        ComponentInfo firstComponent = components.get(0);
        if (firstComponent.componentAttachScope == null) {
            throw new IllegalArgumentException("Routes cannot be wrapped");
        }
        for (int i = 1; i < components.size(); i++) {
            ComponentInfo component = components.get(i);
            if (component.componentAttachScope != firstComponent.componentAttachScope) {
                throw new IllegalArgumentException(
                        "Only components attached in the same block are currently supported");
            }
        }

        // Add wrapper init code where the first component is attached
        int wrapperInsertIndex = JavaRewriterUtil
                .findBlockStatementIndex(firstComponent.attachCall);
        InsertionPoint firstComponentAttachLocation = new InsertionPoint(
                firstComponent.componentAttachScope, wrapperInsertIndex);

        List<VariableDeclarator> statements = createComponentStatements(
                firstComponentAttachLocation, null, wrapperComponent, false,
                null, null);

        // Remove all attach calls for the components and attach them to the
        // wrapper instead. Replace the first component add with an add for the
        // wrapper
        NameExpr wrapperVariable = new NameExpr(
                statements.get(0).getNameAsString());
        MethodCallExpr addCall = new MethodCallExpr(wrapperVariable, "add");

        Node wrapperAdd = null;
        for (int i = 0; i < components.size(); i++) {
            ComponentInfo component = components.get(i);
            Optional<Expression> componentAddArgument;
            if (i == 0) {
                componentAddArgument = Optional.of(
                        JavaRewriterUtil.getAttachArgumentOrThrow(component));
                // Attach wrapper in place of the first component
                Optional<Node> maybeParent = componentAddArgument
                        .flatMap(Node::getParentNode);
                if (maybeParent.isEmpty()) {
                    throw new IllegalArgumentException(
                            "No add argument found for the first component");
                }

                wrapperAdd = maybeParent.get();
                wrapperAdd.replace(componentAddArgument.get(), wrapperVariable);
            } else {
                componentAddArgument = removeAttachCall(component);
            }
            componentAddArgument.ifPresent(addCall::addArgument);
        }
        // Add the wrapper.add(components) call
        if (wrapperAdd == null) {
            throw new IllegalArgumentException("Wrapper was never added");
        }
        firstComponentAttachLocation.block.addStatement(
                JavaRewriterUtil.findBlockStatementIndex(wrapperAdd), addCall);

    }

    private void createComponentStatements(InsertionPoint insertionPoint,
            JavaComponent parent, List<JavaComponent> template,
            String layoutVariableName, ComponentInfo referenceComponent) {
        for (JavaComponent javaComponent : template) {
            createComponentStatements(insertionPoint, parent, javaComponent,
                    true, layoutVariableName, referenceComponent);
        }
    }

    private List<VariableDeclarator> createComponentStatements(
            InsertionPoint insertionPoint, JavaComponent parent,
            JavaComponent javaComponent, boolean attach,
            String layoutVariableName, ComponentInfo referenceComponent) {

        List<VariableDeclarator> createdVariables = new ArrayList<>();

        String componentClassName = javaComponent.className() != null
                ? javaComponent.className()
                : FlowComponentQuirks.getClassForTag(javaComponent.tag());

        ClassOrInterfaceType fullType = StaticJavaParser
                .parseClassOrInterfaceType(componentClassName);

        Map<String, Object> setters = new HashMap<>();
        Class<?> componentType = JavaRewriterUtil
                .getClass(fullType.getNameWithScope());
        javaComponent.props().forEach((prop, value) -> {
            SetterAndValue setterAndValue = JavaRewriterUtil
                    .getSetterAndValue(componentType, prop, value);
            setters.put(setterAndValue.setter(), setterAndValue.value());
        });

        // Import
        JavaRewriterUtil.addImport(compilationUnit,
                fullType.getNameWithScope());
        ClassOrInterfaceType type = fullType.clone().removeScope();

        // Constructor call
        ObjectCreationExpr constructor = new ObjectCreationExpr(null, type,
                new NodeList<>());
        getSingleParamConstructor(fullType, setters.keySet())
                .ifPresent(constructorProp -> {
                    Object value = setters.remove(constructorProp);
                    getParameterList(insertionPoint, value)
                            .forEach(constructor::addArgument);
                });

        String variableBaseName = type.getNameAsString()
                .toLowerCase(Locale.ENGLISH);
        if (javaComponent.props().containsKey("text")) {
            // Use the text for the variable name for more readable code
            variableBaseName = SharedUtil
                    .firstToLower(SharedUtil.dashSeparatedToCamelCase(
                            ((String) javaComponent.props().get("text"))
                                    .replace(' ', '-')));
            variableBaseName = JavaRewriterUtil
                    .getJavaIdentifier(variableBaseName);

        }
        String variableName = insertionPoint
                .getFreeVariableName(variableBaseName);
        NameExpr variableNameExpr = new NameExpr(variableName);

        VariableDeclarator decl = new VariableDeclarator(type, variableName,
                constructor);
        VariableDeclarationExpr declarationExpr = new VariableDeclarationExpr(
                decl);
        createdVariables.add(decl);
        insertionPoint.add(new ExpressionStmt(declarationExpr));

        // Properties setters
        for (Map.Entry<String, Object> setter : setters.entrySet()) {
            Object value = setter.getValue();
            if (value instanceof JavaComponent javaComponentValue) {
                value = createSubComponentStatements(insertionPoint,
                        javaComponent, List.of(javaComponentValue)).get(0);
            }
            insertSetter(insertionPoint, variableNameExpr, setter.getKey(),
                    value);
        }

        // Child components
        createComponentStatements(insertionPoint, javaComponent,
                javaComponent.children(), variableName, null);

        if (attach) {
            attachComponent(insertionPoint, parent, layoutVariableName,
                    referenceComponent, variableNameExpr, variableName);
        }

        return createdVariables;
    }

    private void attachComponent(InsertionPoint insertionPoint,
            JavaComponent parent, String layoutVariableName,
            ComponentInfo referenceComponent, NameExpr variableNameExpr,
            String variableName) {
        // Attach component
        if (referenceComponent != null) {
            MethodCallExpr referenceAttach = referenceComponent.attachCall();
            if (referenceAttach.getArguments().size() == 1) {
                // add(reference)
                ExpressionStmt addCall = createAddCall(parent,
                        referenceAttach.getScope().orElse(null),
                        variableNameExpr);
                BlockStmt block = JavaRewriterUtil
                        .findAncestorOrThrow(referenceAttach, BlockStmt.class);
                block.addStatement(JavaRewriterUtil
                        .findBlockStatementIndex(referenceAttach), addCall);
            } else {
                Expression referenceArgument;
                // add(..., reference,...)
                if (layoutVariableName == null) {
                    // Reference is an inline call, like add(new Button()
                    referenceArgument = referenceComponent.objectCreationExpr();
                } else {
                    referenceArgument = referenceAttach.getArguments().stream()
                            .filter(arg -> arg instanceof NameExpr nameExpr
                                    && nameExpr.getNameAsString()
                                            .equals(layoutVariableName))
                            .findFirst().orElse(null);
                }
                if (referenceArgument == null) {
                    throw new IllegalArgumentException(
                            "Did not find reference argument ('" + variableName
                                    + "') in " + referenceAttach);
                }
                int index = referenceAttach.getArguments()
                        .indexOf(referenceArgument);
                referenceAttach.getArguments().add(index, variableNameExpr);
            }
        } else if (layoutVariableName != null) {
            NameExpr scope = null;
            if (!layoutVariableName.equals("this")) {
                scope = new NameExpr(layoutVariableName);
            }
            ExpressionStmt addCall = createAddCall(parent, scope,
                    variableNameExpr);
            // Just adding to the given layout
            insertionPoint.add(addCall);
        } else {
            throw new IllegalArgumentException(
                    "Either layoutVariableName or referenceAttach must be given to attach a component");
        }
    }

    private NodeList<Expression> getParameterList(InsertionPoint insertionPoint,
            Object value) {
        if (value instanceof List<?> list) {
            if (list.isEmpty()) {
                return new NodeList<>();
            }
            if (list.get(0) instanceof JavaComponent) {
                return JavaRewriterUtil.toExpressionList(
                        createSubComponentStatements(insertionPoint, null,
                                (List<JavaComponent>) list));
            }
        }
        return JavaRewriterUtil.toExpressionList(value);
    }

    private void insertSetter(InsertionPoint insertionPoint, Expression owner,
            String setterName, Object value) {
        if (setterName.equals("setStyle")) {
            insertStyles(insertionPoint, owner, (Map<String, String>) value);
        } else if (setterName.equals("setSlot")) {
            // getElement().setAttribute("slot", value)
            MethodCallExpr getElement = new MethodCallExpr(owner, "getElement");
            MethodCallExpr setAttributeCall = new MethodCallExpr(getElement,
                    "setAttribute",
                    new NodeList<>(JavaRewriterUtil.toExpression("slot"),
                            JavaRewriterUtil.toExpression(value)));
            insertionPoint.add(new ExpressionStmt(setAttributeCall));
        } else {
            NodeList<Expression> parameterExpression = getParameterList(
                    insertionPoint, value);
            MethodCallExpr setterCall = new MethodCallExpr(owner, setterName,
                    parameterExpression);
            insertionPoint.add(new ExpressionStmt(setterCall));
        }
    }

    private void insertStyles(InsertionPoint insertionPoint, Expression owner,
            Map<String, String> styles) {
        // button.getStyle().set("p1","value1").set("p2","value2");
        MethodCallExpr finalCall = new MethodCallExpr(owner, "getStyle");
        for (Map.Entry<String, String> entry : styles.entrySet()) {
            finalCall = new MethodCallExpr(finalCall, "set",
                    new NodeList<>(
                            JavaRewriterUtil.toExpression(entry.getKey()),
                            JavaRewriterUtil.toExpression(entry.getValue())));
        }
        insertionPoint.add(new ExpressionStmt(finalCall));

    }

    private NodeList<Expression> createSubComponentStatements(
            InsertionPoint insertionPoint, JavaComponent parent,
            List<JavaComponent> components) {
        List<VariableDeclarator> variables = new ArrayList<>();
        for (JavaComponent javaComponent : components) {
            variables.addAll(createComponentStatements(insertionPoint, parent,
                    javaComponent, false, null, null));
        }
        return new NodeList<>(
                variables.stream().map(VariableDeclarator::getName)
                        .map(NameExpr::new).toArray(NameExpr[]::new));

    }

    private Optional<String> getSingleParamConstructor(
            ClassOrInterfaceType type, Set<String> setters) {
        Class<?> componentType = JavaRewriterUtil
                .getClass(type.getNameWithScope());
        Optional<Constructor<?>> constructor = Arrays
                .stream(componentType.getDeclaredConstructors())
                .filter(c -> c.getParameterCount() == 1
                        && c.getParameterTypes()[0] == String.class)
                .findFirst();
        if (constructor.isEmpty()) {
            return Optional.empty();
        }
        String mappedProperty = getMappedProperty(constructor.get(), 0);
        if (mappedProperty != null && setters.contains(mappedProperty)) {
            return Optional.of(mappedProperty);
        }
        return Optional.empty();
    }

    private ExpressionStmt createAddCall(JavaComponent parent, Expression scope,
            Expression toAdd) {
        String methodName = "add";
        if (parent != null && parent.tag().equals("SideNav")) {
            methodName = "addItem";
        }
        return new ExpressionStmt(
                new MethodCallExpr(scope, methodName, new NodeList<>(toAdd)));
    }

    /**
     * Returns the source code.
     *
     * @return the source code
     */
    protected String getSource() {
        return source;
    }
}
