package com.vaadin.copilot.javarewriter;

import com.github.javaparser.Range;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.DataKey;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.BodyDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.BinaryExpr;
import com.github.javaparser.ast.expr.BooleanLiteralExpr;
import com.github.javaparser.ast.expr.DoubleLiteralExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.IntegerLiteralExpr;
import com.github.javaparser.ast.expr.LongLiteralExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.NullLiteralExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithSimpleName;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.Type;
import com.github.javaparser.resolution.types.ResolvedReferenceType;
import com.github.javaparser.resolution.types.ResolvedType;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.shared.util.SharedUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * Util methods for rewriting Java
 */
public class JavaRewriterUtil {

    private static final Set<String> javaKeywords = Set.of("abstract", "assert",
            "boolean", "break", "byte", "case", "catch", "char", "class",
            "const", "continue", "default", "double", "do", "else", "enum",
            "extends", "false", "final", "finally", "float", "for", "goto",
            "if", "implements", "import", "instanceof", "int", "interface",
            "long", "native", "new", "null", "package", "private", "protected",
            "public", "return", "short", "static", "strictfp", "super",
            "switch", "synchronized", "this", "throw", "throws", "transient",
            "true", "try", "void", "volatile", "while");
    private static final int MAX_VARIABLE_NAME_LENGTH = 20;

    private static final Map<Class<?>, Class<?>> WRAPPER_TYPE_MAP;

    static {
        WRAPPER_TYPE_MAP = new HashMap<>(16);
        WRAPPER_TYPE_MAP.put(Integer.class, int.class);
        WRAPPER_TYPE_MAP.put(Byte.class, byte.class);
        WRAPPER_TYPE_MAP.put(Character.class, char.class);
        WRAPPER_TYPE_MAP.put(Boolean.class, boolean.class);
        WRAPPER_TYPE_MAP.put(Double.class, double.class);
        WRAPPER_TYPE_MAP.put(Float.class, float.class);
        WRAPPER_TYPE_MAP.put(Long.class, long.class);
        WRAPPER_TYPE_MAP.put(Short.class, short.class);
        WRAPPER_TYPE_MAP.put(Void.class, void.class);
    }

    private JavaRewriterUtil() {
        // Only static helpers
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaRewriterUtil.class);
    }

    static boolean addAfterLastFunctionCall(List<MethodCallExpr> functionCalls,
            String function, Expression... parameterExpressions) {
        ArrayList<MethodCallExpr> reversedFunctionCalls = new ArrayList<>(
                functionCalls);
        Collections.reverse(reversedFunctionCalls);

        Optional<MethodCallExpr> maybeLastCall = reversedFunctionCalls.stream()
                .filter(call -> hasAncestor(call, Statement.class)
                        && hasAncestor(call, BlockStmt.class))
                .findFirst();
        if (maybeLastCall.isEmpty()) {
            return false;
        }
        MethodCallExpr lastCall = maybeLastCall.get();
        Optional<Expression> maybeScope = lastCall.getScope();
        Expression scope = maybeScope.orElse(null);

        Statement statement = findAncestorOrThrow(lastCall, Statement.class);
        BlockStmt block = findAncestorOrThrow(lastCall, BlockStmt.class);
        MethodCallExpr newCall = new MethodCallExpr(scope, function);
        for (Expression parameterExpression : parameterExpressions) {
            newCall.addArgument(parameterExpression);
        }
        block.addStatement(block.getStatements().indexOf(statement) + 1,
                new ExpressionStmt(newCall));
        return true;

    }

    static List<MethodCallExpr> findMethodCallStatementsInClass(
            Expression expressionInsideClass, String fieldName) {
        return (List) findUsageInClass(expressionInsideClass,
                body -> findMethodCallStatements(fieldName, body).stream());
    }

    static List<MethodCallExpr> findMethodCallNonStatementsInClass(
            Expression expressionInsideClass, String fieldName) {
        return (List) findUsageInClass(expressionInsideClass,
                body -> findMethodCallNonStatements(fieldName, body).stream());
    }

    static List<Expression> findUsageInClass(Expression expressionInsideClass,
            Function<BlockStmt, Stream<? extends Expression>> statementFinder) {
        ClassOrInterfaceDeclaration classDeclaration = findAncestor(
                expressionInsideClass, ClassOrInterfaceDeclaration.class);
        if (classDeclaration == null) {
            return Collections.emptyList();
        }

        List<BlockStmt> blocks = classDeclaration.findAll(BlockStmt.class);
        return blocks.stream().flatMap(statementFinder).toList();
    }

    static VariableDeclarator findLocalVariableDeclarator(String variableName,
            BlockStmt block) {
        return block
                .findFirst(VariableDeclarator.class,
                        v -> v.getNameAsString().equals(variableName))
                .orElse(null);
    }

    static NodeList<Expression> toExpressionList(Object parameter) {
        if (parameter instanceof List<?> list) {
            return new NodeList<>(
                    list.stream().map(JavaRewriterUtil::toExpression).toList());
        } else {
            return new NodeList<>(toExpression(parameter));
        }
    }

    static Expression toExpression(Object parameter) {
        if (parameter == null) {
            return new NullLiteralExpr();
        } else if (parameter instanceof Expression e) {
            return e;
        } else if (parameter instanceof String s) {
            StringLiteralExpr expr = new StringLiteralExpr();
            expr.setString(s);
            return expr;
        } else if (parameter instanceof Long l) {
            return new LongLiteralExpr(String.valueOf(l));
        } else if (parameter instanceof Double d) {
            return new DoubleLiteralExpr(d);
        } else if (parameter instanceof Boolean b) {
            return new BooleanLiteralExpr(b);
        } else if (parameter instanceof Integer d) {
            return new IntegerLiteralExpr(d + "");
        } else if (parameter instanceof JavaRewriter.Code code) {
            return StaticJavaParser.parseExpression(code.code());
        } else {
            throw new IllegalArgumentException(
                    "Unknown type: " + parameter.getClass().getName());
        }
    }

    static boolean equalsByNameAsString(Expression e1, Expression e2) {
        if (e1 == null && e2 == null) {
            return true;
        }
        if (e1 == null) {
            return false;
        }
        if (e2 == null) {
            return false;
        }
        if (e1.isNullLiteralExpr() && e2.isNullLiteralExpr()) {
            return true;
        }
        if (e1.isStringLiteralExpr() && e2.isStringLiteralExpr()) {
            return e1.asStringLiteralExpr().toString()
                    .equals(e2.asStringLiteralExpr().toString());
        }
        if (e1.isFieldAccessExpr() && e2.isFieldAccessExpr()) {
            return e1.toString().equals(e2.toString());
        }
        if (e1 instanceof NodeWithSimpleName<?>
                && e2 instanceof NodeWithSimpleName<?>) {
            return ((NodeWithSimpleName<?>) e1).getNameAsString()
                    .equals(((NodeWithSimpleName<?>) e2).getNameAsString());
        }
        return false;
    }

    /**
     * Parses the given expression and returns the object it represents.
     * <p>
     * If the argument is a null literal, it is returned as is. Method call
     * expressions are also returned as is.
     *
     * @param arg
     *            the expression to parse
     * @param expectedType
     *            the expected type of the object or null if unknown
     * @return the object represented by the expression
     */
    public static Object fromExpression(Expression arg,
            ResolvedType expectedType) {
        if (arg instanceof StringLiteralExpr literalExpression) {
            return literalExpression.asString();
        } else if (arg instanceof IntegerLiteralExpr integerLiteralExpr) {
            return integerLiteralExpr.asNumber();
        } else if (arg instanceof DoubleLiteralExpr doubleLiteralExpr) {
            if (expectedType != null && expectedType.isPrimitive()
                    && expectedType.asPrimitive()
                            .getBoxTypeClass() == Float.class) {
                return (float) doubleLiteralExpr.asDouble();
            }
            return doubleLiteralExpr.asDouble();
        } else if (arg instanceof BooleanLiteralExpr booleanLiteralExpr) {
            return booleanLiteralExpr.getValue();
        } else if (arg instanceof NullLiteralExpr) {
            return arg;
        } else if (arg instanceof MethodCallExpr methodCallExpr) {
            return methodCallExpr;
        } else {
            throw new IllegalArgumentException(
                    "Unknown expression type: " + arg.getClass().getName());
        }
    }

    /**
     * Find all method calls that are statements, i.e. calls like foo.bar() or
     * baz.qux("zug")). Does not find usage of the variable in other
     * expressions, like otherFunction("something" + foo.bar())
     */
    static List<MethodCallExpr> findMethodCallStatements(String variableName,
            BlockStmt scope) {
        return findMethodCalls(variableName, scope)
                .filter(n -> isParentNode(n, ExpressionStmt.class)).toList();
    }

    /**
     * Finds usage of the variable outside of method call statements, i.e. all
     * usage not reported by
     * {@link #findMethodCallStatements(String, BlockStmt)}.
     */
    static List<MethodCallExpr> findMethodCallNonStatements(String variableName,
            BlockStmt scope) {
        return findMethodCalls(variableName, scope)
                .filter(n -> !isParentNode(n, ExpressionStmt.class)).toList();
    }

    private static Stream<MethodCallExpr> findMethodCalls(String variableName,
            BlockStmt scope) {
        if (scope == null) {
            return Stream.empty();
        }
        return scope
                .findAll(MethodCallExpr.class, m -> scopeIs(m, variableName))
                .stream();
    }

    private static Stream<Expression> findNameReferences(String variableName,
            BlockStmt scope) {
        if (scope == null) {
            return Stream.empty();
        }
        return scope.findAll(Expression.class,
                n -> (n.isNameExpr() || n.isFieldAccessExpr())
                        && nameMatches(n, variableName))
                .stream();
    }

    public static List<MethodCallExpr> findMethodCallStatements(
            JavaRewriter.ComponentInfo componentDefinition) {
        if (componentDefinition.fieldName() != null) {
            // If we have a field, we need to go through the whole class because
            // any method anywhere can run functions on it

            // FIXME This returns all methods call anywhere in the class but
            // what the logic would want in most cases
            // is a list of methods called when calling the class constructor or
            // other init methods
            return findMethodCallStatementsInClass(
                    componentDefinition.objectCreationExpr(),
                    componentDefinition.fieldName());
        } else if (componentDefinition.routeConstructor() != null) {
            // Only consider the constructor for now. For full support, would
            // need to traverse all
            // methods called from the constructor in order. And also consider
            // other init methods like @PostConstruct
            return JavaRewriterUtil.findMethodCallStatements(null,
                    componentDefinition.routeConstructor().getBody());
        } else if (componentDefinition.localVariableName() != null) {
            ArrayList<MethodCallExpr> functionCalls = new ArrayList<>();
            functionCalls.addAll(JavaRewriterUtil.findMethodCallStatements(
                    componentDefinition.localVariableName(),
                    componentDefinition.componentCreateScope()));
            if (componentDefinition
                    .componentCreateScope() != componentDefinition
                            .componentAttachScope()) {
                functionCalls.addAll(JavaRewriterUtil.findMethodCallStatements(
                        componentDefinition.localVariableName(),
                        componentDefinition.componentAttachScope()));
            }
            return functionCalls;
        }
        return Collections.emptyList();
    }

    public static List<MethodCallExpr> findMethodCallNonStatements(
            JavaRewriter.ComponentInfo componentDefinition) {
        if (componentDefinition.fieldName() != null) {
            // If we have a field, we need to go through the whole class because
            // any method anywhere can run functions on it
            return findMethodCallNonStatementsInClass(
                    componentDefinition.objectCreationExpr(),
                    componentDefinition.fieldName());
        } else {
            ArrayList<MethodCallExpr> usage = new ArrayList<>();
            usage.addAll(JavaRewriterUtil.findMethodCallNonStatements(
                    componentDefinition.localVariableName(),
                    componentDefinition.componentCreateScope()));
            if (componentDefinition
                    .componentCreateScope() != componentDefinition
                            .componentAttachScope()) {
                usage.addAll(JavaRewriterUtil.findMethodCallNonStatements(
                        componentDefinition.localVariableName(),
                        componentDefinition.componentAttachScope()));
            }
            return usage;
        }
    }

    public static List<Expression> findParameterUsage(
            JavaRewriter.ComponentInfo componentDefinition) {
        List<Expression> refs;
        if (componentDefinition.fieldName() != null) {
            refs = findUsageInClass(componentDefinition.objectCreationExpr(),
                    body -> findNameReferences(componentDefinition.fieldName(),
                            body));
        } else if (componentDefinition.localVariableName() != null) {
            refs = findNameReferences(componentDefinition.localVariableName(),
                    componentDefinition.componentCreateScope()).toList();
        } else {
            refs = Collections.emptyList();
        }
        return refs.stream().filter(JavaRewriterUtil::isMethodCallArgument)
                .toList();
    }

    /**
     * Extracts an inline variable to local variable with a new variable name.
     *
     * @param componentInfo
     *            Component info
     * @return Block, new variable name and index, null if attach call is not
     *         found.
     */
    public static JavaRewriter.ExtractInlineVariableResult extractInlineVariableToLocalVariable(
            JavaRewriter.ComponentInfo componentInfo) {
        Type type = componentInfo.objectCreationExpr().getType();
        BlockStmt block = JavaRewriterUtil.findAncestorOrThrow(
                componentInfo.objectCreationExpr(), BlockStmt.class);
        String newLocalVariable = JavaRewriterUtil
                .findFreeVariableName(componentInfo, block);

        int index = JavaRewriterUtil
                .findBlockStatementIndex(componentInfo.attachCall());
        if (index >= -1) {
            Optional<Node> parentNode = componentInfo.objectCreationExpr()
                    .getParentNode();
            if (parentNode.isEmpty()) {
                throw new IllegalStateException(
                        "Parent for object creation does not exist");
            }

            // add(new Input(...)) -> add(input)
            parentNode.get().replace(componentInfo.objectCreationExpr(),
                    new NameExpr(newLocalVariable));

            // Input input = new Input(...);
            VariableDeclarator declarator = new VariableDeclarator(type,
                    newLocalVariable, componentInfo.objectCreationExpr());
            block.addStatement(index, new VariableDeclarationExpr(declarator));

            return new JavaRewriter.ExtractInlineVariableResult(block,
                    newLocalVariable, index);
        }
        return null;
    }

    public static boolean inlineAssignment(
            JavaRewriter.ComponentInfo componentInfo) {
        return componentInfo.localVariableDeclarator() == null
                && componentInfo.assignmentExpression() == null
                && componentInfo.fieldDeclarationAndAssignment() == null
                && JavaRewriterUtil.onSameLine(
                        componentInfo.objectCreationExpr(),
                        componentInfo.attachCall());
    }

    private static boolean isMethodCallArgument(Expression ref) {
        Optional<Node> parent = ref.getParentNode();
        if (parent.isEmpty()) {
            return false;
        }
        return parent.get() instanceof Expression e && e.isMethodCallExpr()
                && e.asMethodCallExpr().getArguments().contains(ref);
    }

    static boolean isParentNode(Node node, Class<?> parentType) {
        return node.getParentNode()
                .filter(value -> parentType == value.getClass()).isPresent();
    }

    static boolean hasAncestor(Node node, Class<?> parentType) {
        return findAncestor(node, parentType) != null;
    }

    static boolean scopeIs(MethodCallExpr m, String variableName) {
        Optional<Expression> maybeScope = m.getScope();
        if (maybeScope.isEmpty() || maybeScope.get().isThisExpr()) {
            return variableName == null;
        }
        Expression scope = maybeScope.get();
        if (scope instanceof MethodCallExpr) {
            return scopeIs((MethodCallExpr) scope, variableName);
        }
        return nameMatches(scope, variableName);
    }

    private static boolean nameMatches(Expression expression,
            String variableName) {
        if (expression.isNameExpr()) {
            return expression.asNameExpr().getNameAsString()
                    .equals(variableName);
        } else if (expression.isFieldAccessExpr()) {
            return expression.asFieldAccessExpr().getNameAsString()
                    .equals(variableName);
        }

        getLogger().debug("Unknown type of scope expression {}: {}",
                expression.getClass().getName(), expression);
        return false;

    }

    static BlockStmt findBlock(Node node) {
        return findAncestor(node, BlockStmt.class);
    }

    static BlockStmt findBlockOrThrow(Node node) {
        return findAncestorOrThrow(node, BlockStmt.class);
    }

    static <T> T findAncestor(Node node, Class<T> type) {
        return (T) findAncestor(node, n -> type.isAssignableFrom(n.getClass()));
    }

    @Nonnull
    static <T> T findAncestorOrThrow(Node node, Class<T> type)
            throws IllegalArgumentException {
        T ancestor = findAncestor(node, type);
        if (ancestor == null) {
            throw new IllegalArgumentException(
                    "Ancestor of type " + type.getName() + " not found");
        }
        return ancestor;
    }

    static Node findAncestor(Node node, Predicate<Node> filter) {
        if (filter.test(node)) {
            return node;
        }
        return node.getParentNode().map(value -> findAncestor(value, filter))
                .orElse(null);
    }

    static <T extends Node> Optional<T> findNodeOfType(Node node,
            int lineNumber, Class<T> type) {
        return findNodeOfType(node, lineNumber, type, (n) -> true);
    }

    static <T extends Node> Optional<T> findNodeOfType(Node node,
            int lineNumber, Class<T> type, Predicate<T> filter) {
        return Optional.ofNullable((T) findNode(node, lineNumber,
                n -> type.isAssignableFrom(n.getClass())
                        && filter.test((T) n)));
    }

    static <T extends Node> List<T> findNodesOfType(Node node, int lineNumber,
            Class<T> type, Predicate<T> filter) {
        return (List<T>) findNodes(node, lineNumber,
                n -> type.isAssignableFrom(n.getClass()) && filter.test((T) n));
    }

    static Node findNode(Node startFrom, int lineNumber,
            Predicate<Node> filter) {
        for (Node node : startFrom.getChildNodes()) {
            if (coversLine(node, lineNumber)) {
                Node found = findNode(node, lineNumber, filter);
                if (found != null) {
                    return found;
                }
            }
        }
        if (filter.test(startFrom)) {
            return startFrom;
        }

        return null;
    }

    static List<Node> findNodes(Node startFrom, int lineNumber,
            Predicate<Node> filter) {
        List<Node> allFound = new ArrayList<>();
        for (Node node : startFrom.getChildNodes()) {
            if (coversLine(node, lineNumber)) {
                List<Node> found = findNodes(node, lineNumber, filter);
                allFound.addAll(found);
            }
        }
        if (filter.test(startFrom)) {
            allFound.add(startFrom);
        }

        return allFound;
    }

    static boolean coversLine(Node node, int lineNumber) {
        return node.getRange().filter(value -> value.begin.line <= lineNumber
                && value.end.line >= lineNumber).isPresent();
    }

    /**
     * Checks if the given component type has a method with the given name,
     * taking one parameter.
     *
     * @param type
     *            The component type
     * @param func
     *            The method name
     * @return {@code true} if the method exists, {@code false} otherwise
     */
    public static boolean hasSingleParameterMethod(
            Class<? extends Component> type, String func) {
        return Arrays.stream(type.getMethods())
                .anyMatch(method -> method.getName().equals(func)
                        && method.getParameterCount() == 1);
    }

    /**
     * Finds the index of the statement that wraps the given node in the closest
     * block statement.
     *
     * @param node
     *            The node to find the wrapping statement for
     * @return The index of the statement in the block statement
     */
    public static int findBlockStatementIndex(Node node) {
        BlockStmt block = findAncestorOrThrow(node, BlockStmt.class);
        while (true) {
            Optional<Node> maybeParentNode = node.getParentNode();
            if (maybeParentNode.isEmpty()) {
                return -1;
            }
            Node parentNode = maybeParentNode.get();
            if (parentNode == block) {
                // Not using indexOf here as it uses equals, and there might be
                // two equal statements in the list
                for (int i = 0; i < block.getStatements().size(); i++) {
                    if (block.getStatement(i) == node) {
                        return i;
                    }
                }
            }
            node = parentNode;
        }
    }

    /**
     * Finds a free variable name based on the component type.
     *
     * @param componentInfo
     *            The component info
     * @param block
     *            The block the variable will be used in
     * @return A free variable name
     */
    public static String findFreeVariableName(
            JavaRewriter.ComponentInfo componentInfo, BlockStmt block) {
        String base = componentInfo.type().getSimpleName()
                .toLowerCase(Locale.ENGLISH);
        return findFreeVariableName(base, block);

    }

    /**
     * Finds a free variable name based on the given base name.
     *
     * @param base
     *            The base name
     * @param block
     *            The block the variable will be used in
     * @return A free variable name
     */
    public static String findFreeVariableName(String base, BlockStmt block) {
        Set<String> localVariables = block.findAll(VariableDeclarator.class)
                .stream().map(VariableDeclarator::getNameAsString)
                .collect(java.util.stream.Collectors.toSet());
        Set<String> fieldDeclarations = findAncestorOrThrow(block,
                ClassOrInterfaceDeclaration.class)
                .findAll(FieldDeclaration.class).stream()
                .flatMap(fieldDeclaration -> fieldDeclaration.getVariables()
                        .stream())
                .map(NodeWithSimpleName::getNameAsString)
                .collect(java.util.stream.Collectors.toSet());

        base = base.length() > MAX_VARIABLE_NAME_LENGTH
                ? base.substring(0, MAX_VARIABLE_NAME_LENGTH)
                : base;

        if (base.isEmpty()) {
            base = "unknown";
        }
        String name = base;
        int i = 2;
        while (localVariables.contains(name)
                || fieldDeclarations.contains(name)) {
            if (base.matches("[0-9]$")) {
                name = base + "_" + i;
            } else {
                name = base + i;
            }
            i++;
        }
        return name;
    }

    /**
     * Removes the statement that wraps the given node.
     *
     * @param node
     *            the node to remove
     */
    public static void removeStatement(Node node) {
        JavaRewriterUtil.findAncestorOrThrow(node, Statement.class).remove();
    }

    /**
     * Removes the given node from a string concatenation expression.
     *
     * @param node
     *            the node to remove
     * @return {@code true} if the node was removed, {@code false} otherwise
     */
    public static boolean removeFromStringConcatenation(Node node) {
        Optional<Node> parent = node.getParentNode();
        if (parent.isEmpty()) {
            return false;
        }

        if (parent.get() instanceof BinaryExpr binaryExpr) {
            Expression otherExpr = binaryExpr.getLeft() == node
                    ? binaryExpr.getRight()
                    : binaryExpr.getLeft();
            if (otherExpr instanceof StringLiteralExpr) {
                binaryExpr.replace(otherExpr);
                return true;
            }
        }
        return false;
    }

    /**
     * Finds a node among the nodes which refer to the given component.
     *
     * @param nodes
     *            the nodes to search
     * @param componentDefinition
     *            the component definition
     * @return the first node that refers to the component, if any
     */
    public static Optional<Expression> findReference(NodeList<Expression> nodes,
            JavaRewriter.ComponentInfo componentDefinition) {
        return nodes.stream().filter(node -> {
            if (componentDefinition.localVariableName() != null) {
                return isName(node, componentDefinition.localVariableName());
            } else if (componentDefinition.fieldName() != null) {
                return isName(node, componentDefinition.fieldName())
                        || isFieldReference(node,
                                componentDefinition.fieldName());
            } else {
                // Inline
                return (node == componentDefinition.objectCreationExpr());
            }
        }).findFirst();
    }

    private static boolean isFieldReference(Expression node, String s) {
        return node.isFieldAccessExpr()
                && node.asFieldAccessExpr().getNameAsString().equals(s);
    }

    private static boolean isName(Expression node, String s) {
        return node.isNameExpr()
                && node.asNameExpr().getNameAsString().equals(s);
    }

    /**
     * Finds the field declaration for the given field name in the class of the
     * given node.
     *
     * @param nodeInClass
     *            the node in the class
     * @param fieldName
     *            the field name
     * @return the field declaration
     * @throws IllegalArgumentException
     *             if the field is not found
     */
    public static FieldDeclaration findFieldDeclaration(Node nodeInClass,
            String fieldName) {
        ClassOrInterfaceDeclaration classDeclaration = findAncestorOrThrow(
                nodeInClass, ClassOrInterfaceDeclaration.class);

        return classDeclaration.getFieldByName(fieldName)
                .orElseThrow(() -> new IllegalArgumentException(
                        "No field found with anme " + fieldName));
    }

    /**
     * Checks if the two nodes are on the same line.
     *
     * @param node1
     *            the first node
     * @param node2
     *            the second node
     * @return {@code true} if the nodes are on the same line, {@code false}
     *         otherwise
     */
    public static boolean onSameLine(Node node1, Node node2) {
        Optional<Range> r1 = node1.getRange();
        Optional<Range> r2 = node2.getRange();
        if (r1.isEmpty() || r2.isEmpty()) {
            return false;
        }
        return r1.get().begin.line == r2.get().begin.line;
    }

    /**
     * Gets the setter name and value for the given component type, property and
     * value.
     *
     * @param componentType
     *            the component type
     * @param property
     *            the property
     * @param value
     *            the value
     * @return the setter name and value
     */
    public static JavaRewriter.SetterAndValue getSetterAndValue(
            Class<?> componentType, String property, Object value) {
        String setterName = getSetterName(property, componentType, true);

        if (FlowComponentQuirks.isInvertedBoolean(property, componentType)) {
            return new JavaRewriter.SetterAndValue(setterName,
                    !Boolean.TRUE.equals(value));
        }
        Object propertyValue = getPropertyValue(componentType, setterName,
                property, value);
        return new JavaRewriter.SetterAndValue(setterName, propertyValue);
    }

    /**
     * Gets the setter name for the given property.
     *
     * @param property
     *            the property
     * @param type
     *            the component type
     * @param includeReactConversions
     *            whether to include React property name conversions in the
     *            setter name
     * @return the setter name
     */
    public static String getSetterName(String property, Class<?> type,
            boolean includeReactConversions) {
        if (includeReactConversions) {
            String setterName = FlowComponentQuirks
                    .convertReactPropertyToJavaSetter(property, type);
            if (setterName != null) {
                return setterName;
            }
        }
        return "set" + SharedUtil
                .capitalize(SharedUtil.dashSeparatedToCamelCase(property));
    }

    /**
     * Gets the field or local variable name for the given component.
     *
     * @param componentInfo
     *            the component info
     * @return the field or local variable name
     */
    public static String getFieldOrVariableName(
            JavaRewriter.ComponentInfo componentInfo) {
        if (componentInfo.localVariableName() != null) {
            return componentInfo.localVariableName();
        } else if (componentInfo.fieldName() != null) {
            return componentInfo.fieldName();
        } else if (componentInfo.routeConstructor() != null) {
            return "this";
        }
        return null;
    }

    /**
     * Adds an import to the given compilation unit if it is not already
     * imported.
     *
     * @param compilationUnit
     *            the compilation unit
     * @param qualifiedName
     *            the qualified name of the import
     */
    public static void addImport(CompilationUnit compilationUnit,
            String qualifiedName) {
        if (compilationUnit.getImports().stream()
                .anyMatch(i -> i.getNameAsString().equals(qualifiedName))) {
            return;
        }
        compilationUnit.addImport(qualifiedName);
    }

    /**
     * Converts the given string into a valid Java identifier.
     *
     * @param str
     *            the string
     * @return the Java identifier
     */
    public static String getJavaIdentifier(String str) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < str.length(); i++) {
            if (Character.isJavaIdentifierStart(str.charAt(i))
                    || (sb.length() > 0
                            && Character.isJavaIdentifierPart(str.charAt(i)))) {
                sb.append(str.charAt(i));
            }
        }
        String result = sb.toString();
        if (isReservedJavaKeyword(result)) {
            result += "_";
        }
        return result;
    }

    private static boolean isReservedJavaKeyword(String identifier) {
        return javaKeywords.contains(identifier);
    }

    /**
     * Checks if the given component type has the given method.
     *
     * @param type
     *            the component type
     * @param methodName
     *            the method to check for
     * @return {@code true} if the component has the method, {@code false}
     *         otherwise
     */
    public static boolean hasMethod(Class<?> type, String methodName) {
        return Arrays.stream(type.getMethods())
                .anyMatch(method -> method.getName().equals(methodName));
    }

    private static Object getPropertyValue(Class<?> componentType,
            String setterName, String prop, Object value) {
        value = FlowComponentQuirks.componentSpecificValueMapping(componentType,
                prop, value);
        if (value == null) {
            return null;
        }

        if (setterName.startsWith("getElement().getThemeList()")) {
            return value;
        }
        if (setterName.equals("setStyle")) {
            return value;
        }
        if (hasSetterForType(componentType, setterName, value.getClass())) {
            return value;
        }
        if (hasSetterForType(componentType, setterName, Component.class)
                && value instanceof JavaRewriter.JavaComponent) {
            return value;
        }
        List<Method> setters = findSetters(componentType, setterName);
        for (Method setter : setters) {
            Class<?> setterType = getSetterType(setter);
            if (setterType == Instant.class) {
                Instant instant = Instant.now();
                if (value.equals("yesterday")) {
                    instant = instant.minusSeconds(60 * 60 * 24L);
                } else if (!value.equals("right now")) {
                    instant = Instant.parse((String) value);
                }

                NameExpr scope = new NameExpr(Instant.class.getName());
                return new MethodCallExpr(scope, "ofEpochSecond")
                        .addArgument(toExpression(instant.getEpochSecond()));
            }
        }
        throw new IllegalArgumentException("Unable to find suitable setter for "
                + componentType.getName() + "." + setterName
                + " for value of type " + value.getClass().getName());
    }

    /**
     * Finds the class for the given source type.
     *
     * @param name
     *            the class name
     * @return the class for the given name
     * @throws IllegalArgumentException
     *             if the class is not found
     */
    public static Class<?> getClass(String name)
            throws IllegalArgumentException {
        try {
            return Class.forName(name);
        } catch (ClassNotFoundException e) {
            getLogger().debug("Class " + name + " not found", e);
            if (name.contains(".")) {
                int lastDot = name.lastIndexOf('.');
                return getClass(name.substring(0, lastDot) + "$"
                        + name.substring(lastDot + 1));
            } else {
                throw new IllegalArgumentException(
                        "Class " + name + " not found", e);
            }
        }
    }

    private static boolean hasSetterForType(Class<?> componentType,
            String setterName, Class<?> valueType) {
        return findSetters(componentType, setterName).stream()
                .anyMatch(method -> {
                    if (getSetterType(method).isAssignableFrom(valueType)) {
                        return true;
                    }

                    return WRAPPER_TYPE_MAP.containsKey(valueType)
                            && getSetterType(method).isAssignableFrom(
                                    WRAPPER_TYPE_MAP.get(valueType));
                });
    }

    private static List<Method> findSetters(Class<?> componentType,
            String setterName) {
        return Arrays.stream(componentType.getMethods())
                .filter(method -> method.getName().equals(setterName))
                .filter(method -> method.getParameterCount() == 1).toList();
    }

    private static Class<?> getSetterType(Method setter) {
        return setter.getParameterTypes()[0];
    }

    public static void removeArgumentCalls(List<MethodCallExpr> methods,
            List<? extends Expression> argumentsToRemove,
            boolean removeMethodIfNoArgs) {
        for (MethodCallExpr methodCall : methods) {
            removeArgumentCalls(methodCall, argumentsToRemove,
                    removeMethodIfNoArgs);
        }
    }

    public static boolean removeArgumentCalls(MethodCallExpr methodCallExpr,
            List<? extends Expression> argumentsToRemove,
            boolean removeMethodIfNoArgs) {

        List<Expression> willRemoveArg = new ArrayList<>();
        for (Expression argument : methodCallExpr.getArguments()) {
            for (Expression argToRemove : argumentsToRemove) {
                if (equalsByNameAsString(argument, argToRemove)) {
                    willRemoveArg.add(argument);
                    break;
                }
            }
        }
        for (Expression argument : willRemoveArg) {
            boolean remove = argument.remove();
            if (!remove) {
                throw new IllegalArgumentException(
                        "Argument remove has failed for "
                                + methodCallExpr.getNameAsString() + ", "
                                + argument);
            }
        }
        if (removeMethodIfNoArgs && methodCallExpr.getArguments().isEmpty()) {
            removeStatement(methodCallExpr);
        }
        return true;
    }

    public static JavaRewriter.InsertionPoint findLocationBefore(
            Expression expr) {
        JavaRewriter.InsertionPoint loc = findLocationAfter(expr);
        return new JavaRewriter.InsertionPoint(loc.getBlock(),
                loc.getIndex() - 1);
    }

    public static JavaRewriter.InsertionPoint findLocationAfter(
            Expression expr) {
        BlockStmt insertBlock = findBlockOrThrow(expr);
        int insertIndex = JavaRewriterUtil.findBlockStatementIndex(expr) + 1;
        return new JavaRewriter.InsertionPoint(insertBlock, insertIndex);
    }

    public static JavaRewriter.InsertionPoint findLocationAtEnd(
            Statement statement) {
        BlockStmt insertBlock = findBlockOrThrow(statement);
        int insertIndex = insertBlock.getStatements().size();
        return new JavaRewriter.InsertionPoint(insertBlock, insertIndex);
    }

    public static void addFieldAfter(FieldDeclaration newField,
            FieldDeclaration reference) {
        ClassOrInterfaceDeclaration classDeclaration = JavaRewriterUtil
                .findAncestorOrThrow(reference,
                        ClassOrInterfaceDeclaration.class);
        NodeList<BodyDeclaration<?>> members = classDeclaration.getMembers();
        int referenceIndex = members.indexOf(reference);
        members.add(referenceIndex + 1, newField);
    }

    public static <T extends Node> T clone(T node) {
        T newNode = (T) node.clone();

        clearData(newNode);
        return newNode;
    }

    private static void clearData(Node node) {
        HashSet<DataKey<?>> dataKeys = new HashSet<>(node.getDataKeys());
        dataKeys.forEach(node::removeData);
        for (Node child : node.getChildNodes()) {
            clearData(child);
        }
    }

    public static Optional<Expression> getAttachArgument(
            JavaRewriter.ComponentInfo component) {
        if (component.attachCall() == null) {
            return Optional.empty();
        }
        return findReference(component.attachCall().getArguments(), component);
    }

    public static Expression getAttachArgumentOrThrow(
            JavaRewriter.ComponentInfo component) {
        return getAttachArgument(component)
                .orElseThrow(() -> new IllegalArgumentException(
                        "No attach argument found for the component"));
    }

    /**
     * Sets the name expression scope for the given method call. For a simple
     * call like `foo.setBar()` this changes `foo` and for nested calls like
     * `foo.baz().fo().a().b()` it also changes `foo`.
     *
     * @param newCall
     *            the method call to change
     * @param nameExpr
     *            the new scope
     * @return {@code true} if the scope was changed, {@code false} otherwise
     */
    public static boolean setNameExprScope(MethodCallExpr newCall,
            NameExpr nameExpr) {
        Optional<Expression> scope = newCall.getScope();
        if (scope.isEmpty()) {
            return false;
        }
        if (scope.get().isMethodCallExpr()) {
            return setNameExprScope(scope.get().asMethodCallExpr(), nameExpr);
        }
        newCall.setScope(nameExpr);
        return true;
    }

    /**
     * Checks if the given type is equal to the given reflection type.
     *
     * @param javaParserType
     *            the Java parser type
     * @param javaReflectionType
     *            the Java reflection type
     * @return {@code true} if the types are equal, {@code false} otherwise
     */
    public static boolean typesEqual(ResolvedType javaParserType,
            Class<?> javaReflectionType) {
        if (javaParserType.isReferenceType()) {
            ResolvedReferenceType refType = javaParserType.asReferenceType();
            return refType.getQualifiedName()
                    .equals(javaReflectionType.getName().replace("$", "."));
        }
        getLogger().debug("Do not know how to compare type {} with {}",
                javaParserType, javaReflectionType);
        return false;
    }

    /**
     * Uses reflection API for finding class name, method and the argument to
     * find whether argument is array.
     *
     * @param className
     *            full class name of a class.
     * @param methodName
     *            method name
     * @param argumentIndex
     *            argument to look up. It can exceed the method parameters, in
     *            that case the last argument is taken into account
     * @return {@code true} if argument is an array, {@code false} otherwise
     */
    public static boolean isArrayArgument(String className, String methodName,
            int argumentIndex) {
        try {

            Class<?> clazz = getClass(className);
            return Arrays.stream(clazz.getMethods())
                    .filter(f -> f.getName().equals(methodName))
                    .anyMatch(method -> {
                        Parameter[] parameters = method.getParameters();
                        Parameter seekingParam;
                        if (argumentIndex < parameters.length) {
                            seekingParam = parameters[argumentIndex];
                        } else {
                            seekingParam = parameters[parameters.length - 1];
                        }
                        return seekingParam.getType().isArray();
                    });
        } catch (Exception ex) {
            getLogger().debug("Could not find argument index of {}",
                    className + "/" + methodName, ex);
        }
        return false;
    }
}
