package com.vaadin.copilot;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.WeakHashMap;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.DevToolsMessageHandler;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.JsonObject;

/**
 * This is the entry point for copilot.
 *
 * <p>
 * It is a singleton with the main purpose to forward messages to the correct
 * CopilotSession instance, based on which session the message is related to.
 */
public class Copilot implements DevToolsMessageHandler {

    public static final String PREFIX = "copilot-";

    public static final String ISSUE_TRACKER = "https://github.com/vaadin/copilot/issues";

    private static Boolean enable = null;

    private final Map<VaadinSession, CopilotSession> copilotSessions = Collections.synchronizedMap(new WeakHashMap<>());

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        VaadinSession vaadinSession = VaadinSession.getCurrent();
        if (vaadinSession == null || !isEnabled(vaadinSession)) {
            return;
        }
        copilotSessions.computeIfAbsent(vaadinSession, newSession -> {
            try {
                return new CopilotSession(newSession, devToolsInterface);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).handleConnect(devToolsInterface);
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        VaadinSession vaadinSession = VaadinSession.getCurrent();
        if (vaadinSession == null || !isEnabled(vaadinSession)) {
            return false;
        }

        if (!command.startsWith(Copilot.PREFIX)) {
            return false;
        }
        command = command.substring(Copilot.PREFIX.length());

        return copilotSessions.get(vaadinSession).handleMessage(command, data, devToolsInterface);
    }

    private static boolean isEnabled(VaadinSession vaadinSession) {
        if (enable == null) {
            enable = CopilotStatus.isEnabled(getContext(vaadinSession));
        }
        return enable;
    }

    static VaadinServletContext getContext(VaadinSession vaadinSession) {
        return (VaadinServletContext) vaadinSession.getService().getContext();
    }

    /**
     * Check if copilot is running in development mode i.e. is being developed.
     * <p>
     * Unrelated to Vaadin's production mode.
     *
     * @return true if copilot is running in development mode
     */
    public static boolean isDevelopmentMode() {
        return Boolean.getBoolean("copilot.development");
    }
}
