package com.vaadin.copilot;

import jakarta.annotation.security.DenyAll;
import jakarta.annotation.security.PermitAll;
import jakarta.annotation.security.RolesAllowed;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.stream.Collectors;

import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.flow.server.auth.AnonymousAllowed;
import com.vaadin.flow.shared.util.SharedUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Handles creation of views for new routes. */
public class RouteCreator {

    private final ProjectManager projectManager;

    public RouteCreator(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteCreator.class);
    }

    /**
     * Creates a Flow view for the given route using the given access requirement.
     *
     * @param route
     *            the path to create a view for
     * @param accessRequirement
     *            the access requirement for the view
     * @param layoutClass
     *            the layout class to use, or {@code null} to not use a layout
     * @throws IOException
     *             if the view file could not be created
     */
    public void createFlowView(String route, AccessRequirement accessRequirement, Class<?> layoutClass)
            throws IOException, RouteDuplicatedException {
        getLogger().debug("Creating Flow view for route {}", route);

        throwIfInvalidRoute(route);

        String lastRouteSegment = getLastRouteSegment(route);
        String className = getFlowViewName(route);
        if (!projectManager.sanitizeFilename(className).equals(className)) {
            throw new IllegalArgumentException("Invalid filename " + className);
        }

        String viewsPackage = projectManager.getFlowViewsPackage();
        File viewFile = getFlowViewsFolder(viewsPackage);
        if (existsRoute(route)) {
            throw new RouteDuplicatedException(route);
        }
        viewFile = new File(viewFile, className + ".java");
        if (viewFile.exists()) {
            throw new RouteDuplicatedException(route);
        }
        viewFile.getParentFile().mkdirs();

        String title = Util.titleCase(SharedUtil.dashSeparatedToCamelCase(lastRouteSegment));

        String content = getFlowRouteTemplate(viewsPackage, route, title, className, accessRequirement, layoutClass);
        projectManager.writeFile(viewFile, CopilotIDEPlugin.undoLabel("Add route"), content);
    }

    private void throwIfInvalidRoute(String route) {
        if (route.isEmpty()) {
            return;
        }
        if (!route.matches("^[a-zA-Z0-9-/]*$")) {
            throw new IllegalArgumentException(
                    "Routes can only contain letters, numbers, dashes and separators (/): " + route);
        }
        if (!route.matches(".*[a-zA-Z0-9]+.*")) {
            throw new IllegalArgumentException("Route must contain at least one letter or number: " + route);
        }
        if (route.contains("//")) {
            throw new IllegalArgumentException("Route must not contain consecutive slashes: " + route);
        }
    }

    private static String getLastRouteSegment(String route) {
        if (route.contains("/")) {
            return route.substring(route.lastIndexOf('/') + 1);
        }
        return route;
    }

    static String getFlowViewName(String route) {
        String[] parts = route.split("/");
        String filename = parts[parts.length - 1];

        String identifier;
        if (filename.isEmpty() || filename.endsWith("/")) {
            identifier = "Main";
        } else {
            identifier = SharedUtil.capitalize(JavaRewriterUtil.getJavaIdentifier(filename, 100));
        }

        return identifier;
    }

    File getFlowViewsFolder(String viewsPackage) {
        File viewFile = projectManager.getJavaSourceFolder();
        for (String folder : viewsPackage.split("\\.")) {
            viewFile = new File(viewFile, folder);
        }
        return viewFile;
    }

    /**
     * Creates a Hilla view for the given route using the given access requirement.
     *
     * @param route
     *            the path to create a view for
     * @param accessRequirement
     *            the access requirement for the view
     */
    public void createHillaView(String route, AccessRequirement accessRequirement)
            throws IOException, RouteDuplicatedException {
        getLogger().debug("Creating Hilla view for route {}", route);

        throwIfInvalidRoute(route);
        // Assumes FS router will set up routing
        String filenameWithPath = route;
        if (filenameWithPath.isEmpty() || filenameWithPath.endsWith("/")) {
            filenameWithPath += "@index";
        }

        if (!projectManager.sanitizeFilename(filenameWithPath).equals(filenameWithPath.replace('/', '_'))) {
            throw new IllegalArgumentException("Invalid filename " + filenameWithPath);
        }
        File viewFile = projectManager.getHillaViewsFolder();

        String[] parts = filenameWithPath.split("/");
        for (String part : parts) {
            viewFile = new File(viewFile, part);
        }
        if (existsRoute(route)) {
            throw new RouteDuplicatedException(route);
        }
        viewFile = addExtension(viewFile, ".tsx");
        if (viewFile.exists()) {
            throw new RouteDuplicatedException(route);
        }
        viewFile.getParentFile().mkdirs();
        String viewName = viewFile.getName().replace(".tsx", "");
        if (viewName.startsWith("@")) {
            viewName = "";
        }
        String title = Util.titleCase(SharedUtil.dashSeparatedToCamelCase(viewName));

        String content = getHillaRouteTemplate(parts[parts.length - 1], title, accessRequirement);
        projectManager.writeFile(viewFile, CopilotIDEPlugin.undoLabel("Add route"), content);
    }

    private String getHillaRouteTemplate(String routeName, String title, AccessRequirement accessRequirement) {
        String functionName = getValidReactComponentFunctionName(
                SharedUtil.capitalize(SharedUtil.dashSeparatedToCamelCase(routeName)));
        String accessControl;
        switch (accessRequirement.getType()) {
        case PERMIT_ALL:
            accessControl = "loginRequired: true,";
            break;
        case ANONYMOUS_ALLOWED:
            accessControl = "";
            break;
        case DENY_ALL:
            accessControl = "rolesAllowed: [],";
            break;
        case ROLES_ALLOWED:
            accessControl = "rolesAllowed: [" + Arrays.stream(accessRequirement.getRoles())
                    .map(role -> "\"" + role + "\"").collect(Collectors.joining(", ")) + "],";
            break;
        default:
            throw new IllegalArgumentException("Unknown access requirement: " + accessRequirement.getType());
        }
        return """
                import type { ViewConfig } from '@vaadin/hilla-file-router/types.js';

                export const config: ViewConfig = {
                  title: 'TITLE',
                  ACCESS_CONTROL
                };

                export default function FUNCTION_NAME() {
                    return (
                        <div className="flex flex-col">
                            <h1>ROUTE_NAME</h1>
                        </div>
                    );
                }
                """.replace("TITLE", title).replace("FUNCTION_NAME", functionName).replace("ROUTE_NAME", routeName)
                .replace("ACCESS_CONTROL", accessControl);
    }

    private String getValidReactComponentFunctionName(String functionName) {
        if (functionName.equals("@index")) {
            return "Index";
        }
        if (functionName.matches("^[^A-Z].*")) {
            // React component names must start with a capital letter
            functionName = "A" + functionName;
        }

        return functionName.replaceAll("[^a-zA-Z0-9]", "");
    }

    String getFlowRouteTemplate(String packageName, String route, String title, String className,
            AccessRequirement accessRequirement, Class<?> layoutClass) {
        Class<? extends Annotation> accessClass;
        String accessParam = null;
        switch (accessRequirement.getType()) {
        case PERMIT_ALL:
            accessClass = PermitAll.class;
            break;
        case ANONYMOUS_ALLOWED:
            accessClass = AnonymousAllowed.class;
            break;
        case DENY_ALL:
            accessClass = DenyAll.class;
            break;
        case ROLES_ALLOWED:
            accessClass = RolesAllowed.class;
            accessParam = "{" + Arrays.stream(accessRequirement.getRoles()).map(role -> "\"" + role + "\"")
                    .collect(Collectors.joining(", ")) + "}";
            break;
        default:
            throw new IllegalArgumentException("Unknown access requirement: " + accessRequirement.getType());
        }

        String accessAnnotation = accessClass.getSimpleName();
        if (accessParam != null) {
            accessAnnotation += "(" + accessParam + ")";
        }

        String layoutImport = "";
        String layoutAnnotationValue = "";
        if (layoutClass != null) {
            layoutImport = "import " + layoutClass.getName() + ";";
            layoutAnnotationValue = ", layout = " + layoutClass.getSimpleName() + ".class";
        }

        return """
                package VIEW_PACKAGE;

                import com.vaadin.flow.component.html.H1;
                import com.vaadin.flow.component.orderedlayout.HorizontalLayout;
                import com.vaadin.flow.router.Route;
                import com.vaadin.flow.router.PageTitle;
                import com.vaadin.flow.server.auth.AnonymousAllowed;
                import ACCESS_CLASS_NAME;
                LAYOUT_IMPORT

                @Route(value = "ROUTE"ROUTE_LAYOUT)
                @PageTitle("TITLE")
                @ACCESS_ANNOTATION
                public class CLASS_NAME extends HorizontalLayout {

                    public CLASS_NAME() {
                        add(new H1("This is CLASS_NAME"));
                    }

                }
                """.replace("LAYOUT_IMPORT", layoutImport).replace("ROUTE_LAYOUT", layoutAnnotationValue)
                .replace("VIEW_PACKAGE", packageName).replace("ROUTE", route)
                .replace("ACCESS_CLASS_NAME", accessClass.getName()).replace("CLASS_NAME", className)
                .replace("TITLE", title).replace("ACCESS_ANNOTATION", accessAnnotation);
    }

    private static File addExtension(File viewFile, String ext) {
        return new File(viewFile.getParentFile(), viewFile.getName() + ext);
    }

    private boolean existsRoute(String route) {
        try {
            RouteHandler.getServerRoutes(projectManager.getVaadinSession()).stream()
                    .filter(routeData -> routeData.getTemplate().equals(route)).findAny().ifPresent(routeData -> {
                        throw new RuntimeException("Route already exists: " + route);
                    });
        } catch (RuntimeException e) {
            return true;
        }
        return false;
    }
}
