package com.vaadin.copilot.javarewriter;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.component.Component;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;

public record ComponentInfo(Class<? extends Component> type, ObjectCreationExpr objectCreationExpr,
        BlockStmt componentCreateScope, AttachExpression attachCall, BlockStmt componentAttachScope,
        VariableDeclarator localVariableDeclarator, AssignExpr assignmentExpression, FieldDeclaration fieldDeclaration,
        FieldDeclaration fieldDeclarationAndAssignment, String localVariableName, String fieldName,
        ConstructorDeclaration routeConstructor, boolean containerComposite, boolean isAnonymousComponent,
        boolean isReturnValue, JavaRewriter rewriter) {

    /**
     * Creates ComponentInfo of the given component.
     *
     * @param typeAndSourceLocation
     *            Type and source location of a component
     * @param javaRewriter
     *            JavaRewriter class of the given component.
     * @return Component info if every argument is found well. Throws
     *         {@code IllegalArgumentException} for unknown statements.
     */
    public static ComponentInfo find(ComponentTypeAndSourceLocation typeAndSourceLocation, JavaRewriter javaRewriter) {
        CompilationUnit compilationUnit = javaRewriter.compilationUnit;
        List<ObjectCreationExpr> objectCreationExprs = new ArrayList<>();
        for (Class clazz : typeAndSourceLocation.inheritanceChain()) {
            objectCreationExprs = JavaRewriterUtil.findNodesOfType(compilationUnit,
                    typeAndSourceLocation.createLocation().lineNumber(), ObjectCreationExpr.class,
                    node -> node.getType().asClassOrInterfaceType().getName().asString().equals(clazz.getSimpleName()));
            if (objectCreationExprs.size() > 0) {
                break;
            }
        }

        if (objectCreationExprs.size() > 1) {
            throw new IllegalArgumentException(
                    "There are multiple components created on the given line and we are unable to determine which one to modify");
        }

        Optional<ObjectCreationExpr> objectCreationExpr = Optional
                .ofNullable(objectCreationExprs.isEmpty() ? null : objectCreationExprs.get(0));

        boolean routeClass = false;
        boolean compositeContainer = false;
        boolean isAnonymousComponent = false;
        boolean isReturnValue = false;
        if (objectCreationExpr.isEmpty()) {
            routeClass = JavaRewriterUtil.isRouteClass(typeAndSourceLocation, compilationUnit);
            if (!routeClass) {
                throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                        "Constructor not found at the expected location: " + typeAndSourceLocation.createLocation());
            }
        } else {
            ObjectCreationExpr n = objectCreationExpr.orElse(null);
            Node parent = n.getParentNode().orElse(null);
            if (n != null && parent != null && !(parent instanceof VariableDeclarator || parent instanceof AssignExpr
                    || parent instanceof FieldDeclaration)) {
                isAnonymousComponent = true;
            }
            if (n != null && parent != null && (parent instanceof ReturnStmt)) {
                isReturnValue = true;
            }
        }
        Optional<BlockStmt> componentCreateScope = objectCreationExpr.map(JavaRewriterUtil::findBlock);

        Optional<VariableDeclarator> localVariableDeclarator = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr, VariableDeclarator.class));
        Optional<AssignExpr> assignmentExpression = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr, AssignExpr.class));
        Optional<FieldDeclaration> fieldDeclarationAndAssignment = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr, FieldDeclaration.class));
        Optional<FieldDeclaration> fieldDeclaration = fieldDeclarationAndAssignment;
        if (localVariableDeclarator.isPresent() && fieldDeclarationAndAssignment.isPresent()) {
            // A variable declarator is found also for assignments for fields
            // but we want to differentiate between the two cases later on
            localVariableDeclarator = Optional.empty();
        }

        String localVariableName = null;
        String fieldName = null;
        if (localVariableDeclarator.isPresent()) {
            // TextField foo = new TextField();
            localVariableName = localVariableDeclarator.get().getNameAsString();
        } else if (fieldDeclarationAndAssignment.isPresent()) {
            // private TextField foo = new TextField();
            fieldName = fieldDeclarationAndAssignment.get().getVariable(0).getNameAsString();
        } else if (assignmentExpression.isPresent()) {
            // foo = new TextField();
            // Here foo can be a local variable or field
            String localVariableOrFieldName = assignmentExpression.get().getTarget().asNameExpr().getNameAsString();
            if (componentCreateScope.isPresent() && JavaRewriterUtil
                    .findLocalVariableDeclarator(localVariableOrFieldName, componentCreateScope.get()) != null) {
                localVariableName = localVariableOrFieldName;
            } else {
                fieldName = localVariableOrFieldName;
                fieldDeclaration = Optional
                        .ofNullable(JavaRewriterUtil.findFieldDeclaration(assignmentExpression.get(), fieldName));
            }
        }

        Optional<BlockStmt> componentAttachScope;
        AttachExpression attachCall;

        Optional<MethodCallExpr> attachMethodCall = JavaRewriterUtil.findNodeOfType(compilationUnit,
                typeAndSourceLocation.attachLocation().lineNumber(), MethodCallExpr.class);
        if (attachMethodCall.isEmpty() && !routeClass) {
            Optional<ObjectCreationExpr> attachObjectCreationCall = JavaRewriterUtil.findNodeOfType(compilationUnit,
                    typeAndSourceLocation.attachLocation().lineNumber(), ObjectCreationExpr.class);
            if (attachObjectCreationCall.isEmpty() && !routeClass) {
                String varName = localVariableName != null ? localVariableName : fieldName;

                Optional<AttachExpression> possibleAttachExpressionFromParentInfo = findPossibleAttachExpressionFromParentInfo(
                        typeAndSourceLocation, objectCreationExpr.orElse(null), varName, javaRewriter);

                if (possibleAttachExpressionFromParentInfo.isEmpty()) {
                    throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                            "Attach not found at the expected location: " + typeAndSourceLocation.attachLocation());
                }
                attachCall = possibleAttachExpressionFromParentInfo.get();
                componentAttachScope = Optional
                        .of(JavaRewriterUtil.findBlock(possibleAttachExpressionFromParentInfo.get().getNode()));
            } else {
                componentAttachScope = attachObjectCreationCall.map(JavaRewriterUtil::findBlock);
                attachCall = new AttachExpression(attachObjectCreationCall.orElse(null));
            }
        } else {
            componentAttachScope = attachMethodCall.map(JavaRewriterUtil::findBlock);
            attachCall = new AttachExpression(attachMethodCall.orElse(null));
        }

        Optional<ConstructorDeclaration> routeConstructor = Optional.empty();
        if (routeClass) {
            routeConstructor = JavaRewriterUtil.findNodeOfType(compilationUnit,
                    typeAndSourceLocation.createLocation().lineNumber(), ConstructorDeclaration.class);

            if (routeConstructor.isEmpty()) {
                // Potentially no constructor
                List<ConstructorDeclaration> constructors = compilationUnit.findAll(ConstructorDeclaration.class);
                if (constructors.isEmpty()) {
                    // Need to create a constructor
                    routeConstructor = Optional.of(new ConstructorDeclaration()
                            .setName(typeAndSourceLocation.type().getSimpleName()).setPublic(true));
                    String className = typeAndSourceLocation.type().getSimpleName();
                    ClassOrInterfaceDeclaration classDeclaration = compilationUnit
                            .findAll(ClassOrInterfaceDeclaration.class).stream()
                            .filter(c -> c.getNameAsString().equals(className)).findFirst()
                            .orElseThrow(() -> new IllegalArgumentException("Class " + className + "not found"));
                    classDeclaration.addMember(routeConstructor.get());
                } else {
                    throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                            "Route class has constructors but none at the expected location: "
                                    + typeAndSourceLocation.createLocation());
                }
            }
        }
        compositeContainer = isChildOfCompositeContainer(attachCall.expression());

        if (compositeContainer && attachCall.expression().isMethodCallExpr()) {
            MethodCallExpr methodCallExpr = attachCall.getMethodCallExpression();
            if (methodCallExpr != null && methodCallExpr.getNameAsString().equals("getContent")
                    && methodCallExpr.getParentNode().isPresent()) {
                attachCall = new AttachExpression((Expression) methodCallExpr.getParentNode().get());
            }
        }

        return new ComponentInfo(typeAndSourceLocation.type(), objectCreationExpr.orElse(null),
                componentCreateScope.orElse(null), attachCall, componentAttachScope.orElse(null),
                localVariableDeclarator.orElse(null), assignmentExpression.orElse(null), fieldDeclaration.orElse(null),
                fieldDeclarationAndAssignment.orElse(null), localVariableName, fieldName, routeConstructor.orElse(null),
                compositeContainer, isAnonymousComponent, isReturnValue, javaRewriter);
    }

    private static boolean isChildOfCompositeContainer(Expression attachCallExpr) {
        if (attachCallExpr == null) {
            return false;
        }
        return JavaRewriterUtil.isNodeInCompositeClass(attachCallExpr);
    }

    /**
     * To find dynamically generated component attach location.
     *
     * @param componentTypeAndSourceLocation
     *            component location
     * @param varName
     *            local or field variable name
     * @param objectCreationExpr
     *            creation expression is used for inline variables.
     * @return returns empty if parent is null or argument variables are not present
     */
    private static Optional<AttachExpression> findPossibleAttachExpressionFromParentInfo(
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation, ObjectCreationExpr objectCreationExpr,
            String varName, JavaRewriter javaRewriter) {
        if (componentTypeAndSourceLocation.parent() == null) {
            return Optional.empty();
        }
        List<Expression> possibleArgs = new ArrayList<>();
        if (varName != null) {
            possibleArgs.add(new NameExpr(varName));
        }
        if (objectCreationExpr != null) {
            possibleArgs.add(objectCreationExpr);
        }
        if (possibleArgs.isEmpty()) {
            return Optional.empty();
        }
        ComponentTypeAndSourceLocation parent = componentTypeAndSourceLocation.parent();
        ComponentInfo componentInfo = find(parent, javaRewriter);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(componentInfo);

        return methodCallStatements.stream().filter(
                f -> f.getNameAsString().equals("add") && f.getArguments().stream().anyMatch(possibleArgs::contains))
                .findFirst().map(AttachExpression::new);
    }
}
