package com.vaadin.copilot;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.JsonObject;

/** Finds the source location of a component in the project. */
public class ComponentSourceFinder {

    private ProjectManager projectManager;

    /**
     * Creates a new instance of the finder.
     *
     * @param projectManager
     *            the project manager to use
     */
    public ComponentSourceFinder(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    /**
     * Finds the source location of a component.
     *
     * @param component
     *            the component to find, defined by uiId and nodeId in the given
     *            JSON object
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(JsonObject component) {
        return findTypeAndSourceLocation(component, false);
    }

    /**
     * Finds the source location of a component.
     *
     * @param component
     *            the component to find, defined by uiId and nodeId in the given
     *            JSON object
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(JsonObject component, boolean includeChildren) {
        return findTypeAndSourceLocation((int) component.getNumber("uiId"), (int) component.getNumber("nodeId"),
                includeChildren);
    }

    /**
     * Finds the source location of a component.
     *
     * @param uiId
     *            the uiId of the component
     * @param nodeId
     *            the nodeId of the component
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(int uiId, int nodeId) {
        return findTypeAndSourceLocation(uiId, nodeId, false);
    }

    /**
     * Finds the source location of a component.
     *
     * @param uiId
     *            the uiId of the component
     * @param nodeId
     *            the nodeId of the component
     * @param includeChildren
     *            whether to include children in the search
     * @return the source location of the component
     */
    public ComponentTypeAndSourceLocation findTypeAndSourceLocation(int uiId, int nodeId, boolean includeChildren) {
        AtomicReference<ComponentTypeAndSourceLocation> info = new AtomicReference<>(null);

        VaadinSession vaadinSession = projectManager.getVaadinSession();
        vaadinSession.accessSynchronously(() -> {
            Element element = vaadinSession.findElement(uiId, nodeId);
            Optional<Component> c = element.getComponent();
            if (c.isPresent()) {
                ComponentTypeAndSourceLocation componentInfo = findTypeAndSourceLocation(c.get(), includeChildren);
                info.set(componentInfo);
            }
        });
        ComponentTypeAndSourceLocation res = info.get();
        if (res == null) {
            throw new IllegalArgumentException("Unable to find component in source");
        }
        return res;
    }

    private ComponentTypeAndSourceLocation findTypeAndSourceLocation(Component c, boolean includeChildren) {
        Optional<Component> parent = c.getParent();

        ComponentTypeAndSourceLocation parentInfo = parent.map(p -> getSourceLocation(p, null, null)).orElse(null);

        List<ComponentTypeAndSourceLocation> children = new ArrayList<>();
        if (includeChildren) {
            c.getChildren().forEach(child -> {
                ComponentTypeAndSourceLocation childInfo = findTypeAndSourceLocation(child, true);
                children.add(childInfo);
            });
        }
        return getSourceLocation(c, parentInfo, children);
    }

    private ComponentTypeAndSourceLocation getSourceLocation(Component component,
            ComponentTypeAndSourceLocation parentInfo, List<ComponentTypeAndSourceLocation> children) {
        ComponentTracker.Location createLocation = ComponentTracker.findCreate(component);
        ComponentTracker.Location attachLocation = ComponentTracker.findAttach(component);
        File javaFile = projectManager.getSourceFile(createLocation);

        List<Class> inheritanceChain = new ArrayList<>();
        Class superClass = component.getClass().getSuperclass();
        inheritanceChain.add(component.getClass());
        while (superClass != null) {
            inheritanceChain.add(superClass);
            superClass = superClass.getSuperclass();
        }

        return new ComponentTypeAndSourceLocation(component.getClass(), inheritanceChain, component, javaFile,
                createLocation, attachLocation, parentInfo, children);
    }
}
