package com.vaadin.copilot;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasElement;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.server.VaadinSession;

public class FlowUtil {

    /**
     * Finds all Java files that are used to create the current component tree for
     * the given ui.
     *
     * @param projectManager
     *            the project manager
     * @param uiId
     *            the id for the UI, inside the given session
     * @return a map of the locations and the corresponding Java files
     */
    public static Map<ComponentTracker.Location, File> findActiveJavaFiles(ProjectManager projectManager, int uiId) {
        VaadinSession session = projectManager.getVaadinSession();
        try {
            session.lock();
            UI ui = session.getUIById(uiId);
            Set<ComponentTracker.Location> allComponents = getAllComponentLocations(ui);
            Set<String> processedClasses = new HashSet<>();
            LinkedHashMap<ComponentTracker.Location, File> locations = new LinkedHashMap<>();
            for (ComponentTracker.Location location : allComponents) {
                if (processedClasses.contains(location.className())) {
                    continue;
                }
                File javaFile = projectManager.getFileForClass(location.className());
                // The component locator tracks the Java class and guesses the
                // Java file, so it's possible the Java file is not in the
                // project
                if (javaFile.exists()) {
                    processedClasses.add(location.className());
                    locations.put(location, javaFile);
                }
            }
            return locations;
        } finally {
            session.unlock();
        }
    }

    private static Set<ComponentTracker.Location> getAllComponentLocations(UI ui) {
        List<Component> componentList = new ArrayList<>();
        addComponents(ui, componentList);

        Set<ComponentTracker.Location> locations = new HashSet<>();

        for (Component component : componentList) {
            ComponentTracker.Location create = ComponentTracker.findCreate(component);
            ComponentTracker.Location attach = ComponentTracker.findAttach(component);
            if (create != null && attach != null) {
                // e.g. the UI has no create location, and we are not interested
                // in its attach
                locations.add(create);
                locations.add(attach);
            }
        }
        return locations;
    }

    /**
     * Returns the view chain for the given UI.
     *
     * @param session
     *            the vaadin session
     * @param uiId
     *            the id for the UI, inside the given session
     * @return a list of the view classes currently being shown in the UI
     */
    public static List<Class<? extends HasElement>> getViewChain(VaadinSession session, int uiId) {
        try {
            session.lock();
            UI ui = session.getUIById(uiId);
            if (ui == null) {
                return Collections.emptyList();
            }
            return (List) ui.getActiveRouterTargetsChain().stream().map(Object::getClass).toList();
        } finally {
            session.unlock();
        }
    }

    private static void addComponents(Component component, List<Component> componentList) {
        componentList.add(component);
        component.getChildren().forEach(c -> addComponents(c, componentList));
    }
}
