package com.vaadin.copilot;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Base64;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.frontend.FrontendUtils;
import com.vaadin.flow.server.frontend.ThemeUtils;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import org.apache.commons.io.FileUtils;

/** Handles reading and writing files in the project */
public class ProjectManager {
    final Path javaSourceFolder;
    final Path javaTestSourceFolder;
    final Path kotlinSourceFolder;
    final Path kotlinTestSourceFolder;
    private final Path projectRoot;
    private final ApplicationConfiguration applicationConfiguration;
    private final CopilotIDEPlugin idePlugin;
    private final VaadinSession vaadinSession;

    /**
     * Creates a new project manager
     *
     * @param applicationConfiguration
     *            the application configuration
     * @param vaadinSession
     *            the Vaadin session
     * @throws IOException
     *             if the project folder cannot be resolved
     */
    public ProjectManager(ApplicationConfiguration applicationConfiguration, VaadinSession vaadinSession)
            throws IOException {
        this.projectRoot = applicationConfiguration.getProjectFolder().toPath().toRealPath();
        this.javaSourceFolder = applicationConfiguration.getJavaSourceFolder().toPath();
        this.javaTestSourceFolder = Util.replaceFolderInPath(this.javaSourceFolder, "main", "test", "src");
        this.kotlinSourceFolder = Util.replaceFolderInPath(this.javaSourceFolder, "java", "kotlin", "main");
        this.kotlinTestSourceFolder = Util.replaceFolderInPath(this.kotlinSourceFolder, "main", "test", "src");
        this.applicationConfiguration = applicationConfiguration;
        this.vaadinSession = vaadinSession;
        this.idePlugin = CopilotIDEPlugin.getInstance();
    }

    public String readFile(String filename) throws IOException {
        return readFile(new File(filename));
    }

    public String readFile(Path filename) throws IOException {
        return readFile(filename.toFile());
    }

    public String readFile(File file) throws IOException {
        if (file == null) {
            throw new IllegalArgumentException("File cannot be null");
        }
        if (!file.exists()) {
            throw new FileNotFoundException(file.getAbsolutePath());
        }
        if (isFileInsideProject(file)) {
            try {
                return Files.readString(file.toPath(), StandardCharsets.UTF_8);
            } catch (MalformedInputException e) {
                return Files.readString(file.toPath(), StandardCharsets.ISO_8859_1);
            }
        } else {
            throw new IllegalArgumentException(
                    "File " + file.getPath() + " is not inside the project at " + projectRoot);
        }
    }

    public List<String> readLines(File file) throws IOException {
        if (!isFileInsideProject(file)) {
            throw new IllegalArgumentException(
                    "File " + file.getPath() + " is not inside the project at " + projectRoot);
        }
        return FileUtils.readLines(file, StandardCharsets.UTF_8);
    }

    boolean isFileInsideProject(File file) throws IOException {
        return isFileInside(file, projectRoot.toFile());
    }

    private boolean isFileInside(File toCheck, File container) throws IOException {
        if (toCheck.exists()) {
            Path pathToCheck = toCheck.toPath();
            return pathToCheck.toRealPath().startsWith(container.toPath().toString());
        } else {
            // New file
            return isFileInside(toCheck.getParentFile(), container);
        }
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param filename
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(String filename, String undoLabel, String content) throws IOException {
        writeFile(getAbsolutePath(filename), undoLabel, content);
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param file
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(File file, String undoLabel, String content) throws IOException {
        writeFile(file, undoLabel, content, false);
    }

    public void writeFile(Path file, String undoLabel, String content) throws IOException {
        writeFile(file.toFile(), undoLabel, content, false);
    }

    public File writeFileBase64(String filename, String undoLabel, String base64Content, boolean renameIfExists)
            throws IOException {
        var target = getAbsolutePath(filename);
        while (renameIfExists && target.exists()) {
            target = new File(target.getParentFile(), Util.increaseTrailingNumber(target.getName()));
        }
        writeFile(target, undoLabel, base64Content, true);
        return target;
    }

    private void writeFile(File file, String undoLabel, String content, boolean base64Encoded) throws IOException {
        if (isFileInsideProject(file)) {
            File folder = file.getParentFile();
            if (!folder.exists() && !folder.mkdirs()) {
                throw new IOException("Unable to create folder " + folder.getAbsolutePath());
            }

            if (idePlugin.isActive()) {
                if (base64Encoded && idePlugin.supports(CopilotIDEPlugin.Commands.WRITE_BASE64)) {
                    idePlugin.writeBase64File(file, undoLabel, content);
                    return;
                } else if (!base64Encoded && idePlugin.supports(CopilotIDEPlugin.Commands.WRITE)) {
                    idePlugin.writeFile(file, undoLabel, content);
                    return;
                }
            }
            writeFileIfNeeded(file, content, base64Encoded);
        } else {
            throw new IllegalArgumentException(
                    "File " + file.getPath() + " is not inside the project at " + projectRoot);
        }
    }

    private void writeFileIfNeeded(File file, String content, boolean base64Encoded) throws IOException {
        if (file.exists()) {
            String currentContent;
            if (base64Encoded) {
                byte[] data = FileUtils.readFileToByteArray(file);
                currentContent = base64Encode(data);
            } else {
                currentContent = FileUtils.readFileToString(file, StandardCharsets.UTF_8);
            }
            if (content.equals(currentContent)) {
                return;
            }
        }

        byte[] data;
        if (base64Encoded) {
            data = base64Decode(content);
        } else {
            data = content.getBytes(StandardCharsets.UTF_8);
        }
        if (file.exists()) {
            try (FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.WRITE)) {
                int written = channel.write(ByteBuffer.wrap(data));
                channel.truncate(written);
            }
        } else {
            Files.write(file.toPath(), data);
        }
        // If the file was a resource, we can directly copy it to target
        // to make it available
        copyResourceToTarget(file);
    }

    private void copyResourceToTarget(File resourceFile) throws IOException {
        if (!isFileInside(resourceFile, getJavaResourceFolder())) {
            // Not a resource
            return;
        }

        String relativeResourceName = getRelativeName(resourceFile, getJavaResourceFolder());
        File target = new File(getClassesFolder(), relativeResourceName);
        FileUtils.copyFile(resourceFile, target);
    }

    private File getBuildFolder() {
        String buildFolder = applicationConfiguration.getBuildFolder();
        File buildFolderFile = new File(buildFolder);
        if (buildFolderFile.isAbsolute()) {
            return buildFolderFile;
        }
        return new File(projectRoot.toFile(), buildFolder);
    }

    private File getClassesFolder() {
        return new File(getBuildFolder(), "classes");
    }

    private byte[] base64Decode(String content) {
        return Base64.getDecoder().decode(content);
    }

    private String base64Encode(byte[] data) {
        return Base64.getEncoder().encodeToString(data);
    }

    public String makeAbsolute(String projectRelativeFilename) throws IOException {
        Path resolved = projectRoot.resolve(projectRelativeFilename).toRealPath();
        if (!resolved.startsWith(projectRoot)) {
            throw new IllegalArgumentException(
                    "File " + projectRelativeFilename + " is not inside the project at " + projectRoot);
        }
        return resolved.toString();
    }

    public String makeRelative(String filename) throws IOException {
        Path absolutePath = new File(filename).toPath().toRealPath();
        // If file is not inside project root, throw
        if (!absolutePath.startsWith(projectRoot)) {
            throw new IllegalArgumentException("File " + filename + " is not inside the project at " + projectRoot);
        }
        return projectRoot.relativize(absolutePath).toString();
    }

    /**
     * Returns the name of the file, relative to the project root.
     *
     * @param projectFile
     *            the file
     * @return the relative name of the file
     */
    public String getProjectRelativeName(File projectFile) {
        return getRelativeName(projectFile, projectRoot.toFile());
    }

    /**
     * Returns the name of the file, relative to the given folder.
     *
     * @param projectFile
     *            the file
     * @param folder
     *            the folder to make relative to
     * @return the relative name of the file
     */
    public static String getRelativeName(File projectFile, File folder) {
        String relativeFilename = projectFile.getAbsolutePath().substring(folder.getAbsolutePath().length() + 1);
        while (relativeFilename.startsWith("./") || relativeFilename.startsWith(".\\")) {
            relativeFilename = relativeFilename.substring(2);
        }
        // for escaping path characters for Windows
        if (relativeFilename.contains("\\")) {
            relativeFilename = relativeFilename.replace("\\", "\\\\");
        }

        return relativeFilename;
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(Class<?> cls) {
        return getFileForClass(cls.getName());
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(String cls) {
        if (cls.contains("$")) {
            cls = cls.substring(0, cls.indexOf("$"));
        }
        cls = cls.replace(".", File.separator);

        for (Path path : new Path[] { javaSourceFolder, javaTestSourceFolder, kotlinSourceFolder,
                kotlinTestSourceFolder }) {
            for (String extension : new String[] { "java", "kt" }) {
                String filename = cls + "." + extension;
                File file = new File(path.toFile(), filename);
                if (file.exists()) {
                    return file;
                }
            }
        }

        // This is to be compatible with existing code which assumes a non-null return
        // value even if the file does not exist
        return new File(javaSourceFolder.toFile(), cls + ".java");
    }

    /**
     * Returns the Java file for the given component location.
     *
     * @param location
     *            the component location
     * @return the file for the class where the component is used
     */
    public File getSourceFile(ComponentTracker.Location location) {
        return getFileForClass(location.className());
    }

    /**
     * Returns the Java source folder.
     *
     * @return the Java source folder
     */
    public File getJavaSourceFolder() {
        return this.javaSourceFolder.toFile();
    }

    /**
     * Returns the project root folder.
     *
     * @return the project root folder
     */
    public File getProjectRoot() {
        return this.projectRoot.toFile();
    }

    /**
     * Returns the frontend folder.
     *
     * @return the frontend folder
     */
    public File getFrontendFolder() {
        return FrontendUtils.getProjectFrontendDir(applicationConfiguration);
    }

    /**
     * Returns the java resource folder.
     *
     * @return the java resource folder.
     */
    public File getJavaResourceFolder() {
        return applicationConfiguration.getJavaResourceFolder();
    }

    /**
     * Gets current theme name
     *
     * @return optional theme name
     */
    public Optional<String> getThemeName() {
        return ThemeUtils.getThemeName(applicationConfiguration.getContext());
    }

    /**
     * Gets current theme folder
     *
     * @return optional theme folder
     */
    public Optional<File> getThemeFolder() {
        return getThemeName().map(t -> ThemeUtils.getThemeFolder(getFrontendFolder(), t));
    }

    private File getAbsolutePath(String filename) {
        File file = new File(filename);
        if (!file.isAbsolute()) {
            file = new File(projectRoot.toFile(), filename);
        }
        return file;
    }

    /**
     * Makes a string safe to use as a file name
     *
     * @param name
     *            the string to process
     * @return the sanitized string
     */
    public String sanitizeFilename(String name) {
        return name.replaceAll("[^a-zA-Z0-9-_@]", "_");
    }

    /**
     * The folder where Hilla views should be created for the file system router to
     * pick them up.
     *
     * @return the folder where Hilla views should be created
     */
    public File getHillaViewsFolder() {
        return new File(getFrontendFolder(), "views");
    }

    /**
     * Tries to find a common package prefix for all Flow views in the project.
     *
     * <p>
     * If all views are in the same package, that package is returned. If views are
     * in different packages, the common prefix is returned. If no common prefix is
     * found, the main package is returned.
     *
     * @return a suitable package to place a new Flow view in
     */
    public String getFlowViewsPackage() {
        Set<String> viewPackages = new HashSet<>();

        vaadinSession.accessSynchronously(() -> {
            RouteRegistry routeRegistry = vaadinSession.getService().getRouter().getRegistry();
            for (RouteData route : routeRegistry.getRegisteredRoutes()) {
                viewPackages.add(route.getNavigationTarget().getPackageName());
            }
        });

        if (viewPackages.size() == 1) {
            return viewPackages.iterator().next();
        }
        // Find common package prefix
        String commonPrefix = "";
        for (String viewPackage : viewPackages) {
            if (commonPrefix.isEmpty()) {
                commonPrefix = viewPackage;
            } else if (viewPackage.startsWith(commonPrefix)) {
                continue;
            } else {
                while (!commonPrefix.isEmpty() && !viewPackage.startsWith(commonPrefix)) {
                    commonPrefix = commonPrefix.substring(0, commonPrefix.lastIndexOf('.'));
                }
            }
        }
        if (!commonPrefix.isEmpty()) {
            return commonPrefix;
        }

        // Project might not have any Flow views yet
        return getMainPackage() + ".views";
    }

    private String getMainPackage() {
        File packageFolder = Util.getSinglePackage(getJavaSourceFolder());
        return getRelativeName(packageFolder, getJavaSourceFolder()).replace(File.separator, ".");
    }

    public VaadinSession getVaadinSession() {
        return vaadinSession;
    }

}
