package com.vaadin.copilot;

import java.util.List;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasElement;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonValue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Provides server side route information to the client. */
public class RouteHandler implements CopilotCommand {
    private final ProjectManager projectManager;
    private final VaadinServletContext context;
    private final RouteCreator routeCreator;

    /**
     * Creates a new route handler.
     *
     * @param projectManager
     *            the project manager to use
     * @param context
     *            the servlet context to use
     */
    public RouteHandler(ProjectManager projectManager, VaadinServletContext context) {
        this.projectManager = projectManager;
        this.context = context;
        this.routeCreator = new RouteCreator(projectManager);
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-routes")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            boolean securityEnabled = isViewSecurityEnabled();
            returnData.put("securityEnabled", securityEnabled);
            returnData.put("urlPrefix", SpringBridge.getUrlPrefix(context));
            returnData.put("supportsHilla", EndpointRequestUtil.isHillaAvailable());
            VaadinSession vaadinSession = projectManager.getVaadinSession();
            try {
                JsonArray serverRoutes = RouteHandler.getServerRoutes(vaadinSession).stream().map(routeData -> {
                    JsonObject route = Json.createObject();
                    Class<? extends Component> target = routeData.getNavigationTarget();
                    route.put("path", addInitialSlash(routeData.getTemplate()));
                    route.put("navigationTarget", target.getName());
                    if (securityEnabled) {
                        try {
                            AccessRequirement req = AccessRequirementUtil.getAccessRequirement(target, null);

                            route.put("accessRequirement", JsonUtils.beanToJson(req));
                        } catch (Exception e) {
                            getLogger().error("Unable to determine access requirement", e);
                        }
                    }
                    route.put("filename",
                            projectManager.getFileForClass(routeData.getNavigationTarget()).getAbsolutePath());
                    return route;
                }).collect(JsonUtils.asArray());
                returnData.put("routes", serverRoutes);
                devToolsInterface.send("server-routes", returnData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error getting routes", e);
            }
            return true;

        } else if (command.equals("add-route")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));

            String route = data.getString("route");
            if (route.startsWith("/")) {
                route = route.substring(1);
            }

            String framework = data.getString("framework");
            AccessRequirement.Type accessControl = AccessRequirement.Type.valueOf(data.getString("accesscontrol"));
            AccessRequirement accessRequirement;
            if (accessControl == AccessRequirement.Type.ROLES_ALLOWED) {
                accessRequirement = new AccessRequirement(accessControl,
                        JsonUtils.stream(data.getArray("roles")).map(JsonValue::asString).toArray(String[]::new));
            } else {
                accessRequirement = new AccessRequirement(accessControl);
            }
            try {
                if ("hilla".equals(framework)) {
                    routeCreator.createHillaView(route, accessRequirement);
                } else if ("flow".equals(framework)) {
                    Class<? extends HasElement> layoutClass = null;
                    if (data.hasKey("uiId")) {
                        List<Class<? extends HasElement>> viewChain = FlowUtil
                                .getViewChain(projectManager.getVaadinSession(), (int) data.getNumber("uiId"));
                        if (viewChain.size() > 1) {
                            layoutClass = viewChain.get(1);
                        }
                    }
                    routeCreator.createFlowView(route, accessRequirement, layoutClass);
                } else {
                    throw new IllegalArgumentException("Unknown framework: " + framework);
                }
                devToolsInterface.send(command + "-response", returnData);
            } catch (RouteDuplicatedException e) {
                returnData.put("errorDuplicatedRoute", true);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Route already exists", e);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error adding route", e);
            }

            return true;
        }
        return false;
    }

    private boolean isViewSecurityEnabled() {
        return SpringBridge.isSpringAvailable() && SpringBridge.isViewSecurityEnabled(context);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteHandler.class);
    }

    private String addInitialSlash(String template) {
        if (!template.startsWith("/")) {
            return "/" + template;
        }
        return template;
    }

    public static List<RouteData> getServerRoutes(VaadinSession vaadinSession) {
        try {
            vaadinSession.lock();
            return vaadinSession.getService().getRouter().getRegistry().getRegisteredRoutes();
        } catch (Exception e) {
            getLogger().error("Error getting server routes", e);
        } finally {
            vaadinSession.unlock();
        }
        return null;
    }
}
