package com.vaadin.copilot.ide;

import static java.nio.channels.SelectionKey.OP_WRITE;

import java.io.File;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;

import elemental.json.Json;
import elemental.json.JsonObject;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** IDE plugin support utility */
public class CopilotIDEPlugin {

    private static final String UNDO_REDO_PREFIX = "Vaadin Copilot";

    private static final String COPILOT_PLUGIN_DOTFILE = ".copilot-plugin";

    private static final int SELECTOR_TIMEOUT = 1000;

    public enum Commands {
        WRITE("write"),
        WRITE_BASE64("writeBase64"),
        UNDO("undo"),
        REDO("redo"),
        SHOW_IN_IDE("showInIde"),
        REFRESH("refresh");

        final String command;

        Commands(String command) {
            this.command = command;
        }

        Command create(Object data) {
            return new Command(command, data);
        }
    }

    private static class UnsupportedOperationByPluginException extends UnsupportedOperationException {
        public UnsupportedOperationByPluginException(Commands command) {
            super("Used version of Copilot IDE Plugin does not support %s operation, please update plugin to latest version."
                    .formatted(command.command));
        }
    }

    private record Command(String command, Object data) {
    }

    private record RestCommand(String command, String projectBasePath, Object data) {
    }

    private record WriteFileMessage(String file, String undoLabel, String content) {
    }

    private record UndoRedoMessage(List<String> files) {
    }

    private record ShowInIdeMessage(String file, Integer line, Integer column) {
    }

    private record RefreshMessage() {
    }

    private static Path projectRoot;

    private static DevToolsInterface devToolsInterface;

    private static CopilotIDEPlugin INSTANCE;

    private final ObjectMapper objectMapper = new ObjectMapper();

    private final Optional<IdeUtils.IDE> ideThatLaunchedTheApplication;

    private CopilotIDEPlugin() {
        this.ideThatLaunchedTheApplication = IdeUtils.findIde();
    }

    /**
     * Gets instance of CopilotIDEPlugin. Project root must be set before.
     *
     * @return gets or create new instance of CopilotIDEPlugin
     */
    public static CopilotIDEPlugin getInstance() {
        if (INSTANCE == null) {
            if (projectRoot == null) {
                throw new IllegalStateException("Use setProjectRoot before getting instance of CopilotIDEPlugin");
            }
            INSTANCE = new CopilotIDEPlugin();
        }
        return INSTANCE;
    }

    /**
     * Sets project root
     *
     * @param projectRoot
     *            project root path
     */
    public static void setProjectRoot(Path projectRoot) {
        CopilotIDEPlugin.projectRoot = projectRoot;
    }

    /**
     * Sets dev tools interface for user notifications
     *
     * @param devToolsInterface
     */
    public static void setDevToolsInterface(DevToolsInterface devToolsInterface) {
        CopilotIDEPlugin.devToolsInterface = devToolsInterface;
    }

    /**
     * Check if plugin is active based on existing properties file
     *
     * @return true if active, false otherwise
     */
    public boolean isActive() {
        return getDotFile() != null;
    }

    public CopilotIDEPluginProperties getProperties() {
        return new CopilotIDEPluginProperties(getDotFile());
    }

    public IDEPluginInfo getPluginInfo() {
        if (!isActive()) {
            return new IDEPluginInfo(false, IdeUtils.findIde().map(IdeUtils.IDE::getPluginIde).orElse(null));
        }
        CopilotIDEPluginProperties props = new CopilotIDEPluginProperties(getDotFile());

        return new IDEPluginInfo(true, props.getVersion(), props.getSupportedActions(), props.getIde());
    }

    private String getProjectBasePath() throws IOException {
        File dotFile = getDotFile();
        if (dotFile == null) {
            throw new IOException("Dot file not found");
        }
        // assume that project base path is parent of IDE dot dir
        return dotFile.toPath().toRealPath().getParent().getParent().toString();
    }

    private File getDotFile() {
        if (ideThatLaunchedTheApplication.isPresent()) {
            // When running from an IDE, only accept connections from that IDE
            return getDotFile(ideThatLaunchedTheApplication.get());
        }

        // When not running from an IDE, try to find one to use
        for (IdeUtils.IDE ide : IdeUtils.IDE.values()) {
            File dotFile = getDotFile(ide);
            if (dotFile != null) {
                return dotFile;
            }
        }

        return null;
    }

    private File getDotFile(IdeUtils.IDE ide) {
        File dotFile;
        Path dir = projectRoot;
        do {
            dotFile = dir.resolve(ide.getMetaDir()).resolve(COPILOT_PLUGIN_DOTFILE).toFile();
            dir = dir.getParent();
        } while (dir != null && !dotFile.exists());
        if (dotFile.exists() && dotFile.canRead()) {
            return dotFile;
        }
        return null;
    }

    private void removeDotFile() throws IOException {
        Optional.ofNullable(getDotFile()).ifPresent(File::delete);
    }

    /**
     * Calls plugin write file operation
     *
     * @param file
     *            file to be written
     * @param undoLabel
     *            custom undo label
     * @param content
     *            file content
     * @throws IOException
     *             exception if command cannot be serialized
     */
    public JsonObject writeFile(File file, String undoLabel, String content) throws IOException {
        if (!supports(Commands.WRITE)) {
            throw new UnsupportedOperationByPluginException(Commands.WRITE);
        }

        WriteFileMessage data = new WriteFileMessage(file.getAbsolutePath(), undoLabel, content);
        return send(Commands.WRITE.create(data));
    }

    /**
     * Calls plugin writeBase64 file operation
     *
     * @param file
     *            file to be written
     * @param undoLabel
     *            custom undo label
     * @param content
     *            file contents as base 64 encoded string
     * @throws IOException
     *             exception if command cannot be serialized
     */
    public JsonObject writeBase64File(File file, String undoLabel, String content) throws IOException {
        if (!supports(Commands.WRITE_BASE64)) {
            throw new UnsupportedOperationByPluginException(Commands.WRITE_BASE64);
        }

        WriteFileMessage data = new WriteFileMessage(file.getAbsolutePath(), undoLabel, content);
        return send(Commands.WRITE_BASE64.create(data));
    }

    /**
     * Performs Undo for given files
     *
     * @param files
     *            list of files to perform undo
     * @throws IOException
     *             thrown on exception
     */
    public JsonObject undo(List<String> files) throws IOException {
        if (!supports(Commands.UNDO)) {
            throw new UnsupportedOperationByPluginException(Commands.UNDO);
        }

        UndoRedoMessage data = new UndoRedoMessage(files);
        return send(Commands.UNDO.create(data));
    }

    /**
     * Performs Redo for given files
     *
     * @param files
     *            list of files to perform redo
     * @throws IOException
     *             thrown on exception
     */
    public JsonObject redo(List<String> files) throws IOException {
        if (!supports(Commands.REDO)) {
            throw new UnsupportedOperationByPluginException(Commands.REDO);
        }

        UndoRedoMessage data = new UndoRedoMessage(files);
        return send(Commands.REDO.create(data));
    }

    /**
     * Opens editor and places caret on given line and column
     *
     * @param line
     *            line number, use 0 as first line
     * @param column
     *            column number to put caret before, 0 as first column
     * @throws IOException
     *             thrown on exception
     */
    public JsonObject showInIde(String file, Integer line, Integer column) throws IOException {
        if (!supports(Commands.SHOW_IN_IDE)) {
            throw new UnsupportedOperationByPluginException(Commands.SHOW_IN_IDE);
        }

        ShowInIdeMessage data = new ShowInIdeMessage(file, line, column);
        return send(Commands.SHOW_IN_IDE.create(data));
    }

    /**
     * Sends request to synchronize project files with filesystem
     *
     * @throws IOException
     *             thrown on exception
     */
    public JsonObject refresh() throws IOException {
        if (!supports(Commands.REFRESH)) {
            throw new UnsupportedOperationByPluginException(Commands.REFRESH);
        }

        return send(Commands.REFRESH.create(new RefreshMessage()));
    }

    /**
     * Checks if given command is supported by plugin
     *
     * @param command
     *            command to be checked
     * @return true if supported, false otherwise
     */
    public boolean supports(Commands command) {
        return getProperties().getSupportedActions().contains(command.command);
    }

    private JsonObject send(Command command) {
        if (getProperties().getEndpoint() != null) {
            return sendRestSync(command);
        } else {
            sendSocket(command);
            return null;
        }
    }

    // rest client
    private JsonObject sendRestSync(Command command) {
        try {
            RestCommand restCommand = new RestCommand(command.command, getProjectBasePath(), command.data);
            byte[] data = objectMapper.writeValueAsBytes(restCommand);
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(getProperties().getEndpoint()))
                    .header("Content-Type", "application/json").POST(HttpRequest.BodyPublishers.ofByteArray(data))
                    .build();
            HttpResponse<String> response = HttpClient.newHttpClient().send(request,
                    HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != 200) {
                throw new IOException(String.valueOf(response.statusCode()));
            }
            if (response.body() != null && !response.body().isEmpty()) {
                JsonObject responseJson = Json.parse(response.body());
                handleIdeNotifications(responseJson);
                return responseJson;
            }
        } catch (IOException e) {
            // dot file exists but plugin is not running
            try {
                removeDotFile();
            } catch (IOException ex2) {
                getLogger().warn("Cannot remove Copilot plugin properties file", ex2);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }

        return null;
    }

    // plain socket connection
    private void sendSocket(Command command) {
        try (Selector selector = Selector.open(); SocketChannel channel = SocketChannel.open()) {
            channel.configureBlocking(false);

            channel.register(selector, SelectionKey.OP_CONNECT);
            channel.connect(new InetSocketAddress(getProperties().getPort()));

            while (!Thread.interrupted() && selector.isOpen()) {
                selector.select(SELECTOR_TIMEOUT);

                Iterator<SelectionKey> keys = selector.selectedKeys().iterator();

                while (keys.hasNext()) {
                    SelectionKey key = keys.next();
                    keys.remove();

                    if (!key.isValid()) {
                        continue;
                    }

                    if (key.isConnectable()) {
                        if (channel.isConnectionPending()) {
                            channel.finishConnect();
                        }
                        channel.configureBlocking(false);
                        channel.register(selector, OP_WRITE);
                    }

                    if (key.isWritable()) {
                        byte[] data = objectMapper.writeValueAsBytes(command);
                        ByteBuffer buffer = ByteBuffer.wrap(data);
                        while (buffer.hasRemaining()) {
                            channel.write(buffer);
                        }
                        selector.close();
                    }
                }
            }
        } catch (IOException ex) {
            // dot file exists but plugin is not running
            try {
                removeDotFile();
            } catch (IOException ex2) {
                getLogger().warn("Cannot remove Copilot plugin properties file", ex2);
            }
        }
    }

    public static String undoLabel(String operation) {
        return UNDO_REDO_PREFIX + " " + operation;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(CopilotIDEPlugin.class);
    }

    private void handleIdeNotifications(JsonObject response) {
        if (response == null) {
            return;
        }

        if (response.get("blockingPopup") != null) {
            IdeNotification notification = new IdeNotification(
                    "There is a popup in your IDE waiting for action, please check because it might block next Vaadin Copilot operations",
                    IdeNotification.Type.WARNING, "blocking-popup");
            devToolsInterface.send("copilot-ide-notification", notification.asJson());
            response.remove("blockingPopup");
        }
    }
}
