package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashSet;
import java.util.Set;

import com.vaadin.base.devserver.FileWatcher;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.function.SerializableConsumer;
import com.vaadin.flow.internal.DevModeHandlerManager;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Watches target and source files to detect compilation required cases.
 */
public class SourceSyncChecker {

    private static final String JAVA_FILE_EXTENSION = ".java";

    private final Set<String> updatedAtLeastOnce = new HashSet<>();
    private final Set<String> notSyncIssuesStacktrace = new HashSet<>();

    private final ApplicationConfiguration applicationConfiguration;
    private FileWatcher sourceFileWatcher;

    private final SerializableConsumer<File> sourceClassChangeListener = sourceFile -> {
        if (sourceFile.isFile() && sourceFile.getAbsolutePath().endsWith(JAVA_FILE_EXTENSION)) {
            String classPathInProjectWithoutExtension = getSourcePathOfSourceFile(sourceFile);
            updatedAtLeastOnce.add(classPathInProjectWithoutExtension);
        }
    };

    /**
     * Starts the source file watcher with the given application configuration.
     *
     * @param applicationConfiguration
     *            Application configuration to get java source folder
     * @param devModeHandler
     *            flow dev mode handler for registering file watcher shut down
     */
    public SourceSyncChecker(ApplicationConfiguration applicationConfiguration, DevModeHandlerManager devModeHandler) {
        this.applicationConfiguration = applicationConfiguration;
        startSourceFileWatchers(devModeHandler);
    }

    /**
     * Starts a watcher to watch .java file changes in order to collect which files
     * may be out of synced when an exception occurs.
     *
     * @param devModeHandler
     *            flow dev mode handler for registering file watcher shut down
     */
    private void startSourceFileWatchers(DevModeHandlerManager devModeHandler) {
        try {
            File javaSourceFolder = applicationConfiguration.getJavaSourceFolder();
            if (!javaSourceFolder.exists()) {
                getLogger().debug("Java source folder {} does not exist ", javaSourceFolder.getAbsolutePath());
                return;
            }
            sourceFileWatcher = new FileWatcher(sourceClassChangeListener, javaSourceFolder);
            sourceFileWatcher.start();
            getLogger().debug("Started watching {}", applicationConfiguration.getJavaSourceFolder());
            registerWatcherShutdownCommand(devModeHandler);
        } catch (Exception ex) {
            getLogger().error("Could not start file watched for source classes", ex);
        }
    }

    private void registerWatcherShutdownCommand(DevModeHandlerManager devModeHandlerManager) {
        if (sourceFileWatcher == null) {
            return;
        }
        devModeHandlerManager.registerShutdownCommand(() -> {
            try {
                sourceFileWatcher.stop();
            } catch (IOException e) {
                getLogger().error("Could not stop file watcher for source classes", e);
            }
        });
    }

    /**
     * Looks for file has been updated since the startup. Additionally, caches
     * exceptions that happens without any files changes.
     *
     * @param exception
     *            Exception that happens when finding component info fails.
     * @return true if file may be out of sync, false otherwise
     */
    public boolean maybeOutOfSync(ComponentInfoNotFoundException exception) {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = exception.getComponentTypeAndSourceLocation();
        File sourceFile = componentTypeAndSourceLocation.javaFile();
        String sourcePathOfSourceFile = getSourcePathOfSourceFile(sourceFile);
        if (notSyncIssuesStacktrace.contains(ExceptionUtils.getStackTrace(exception))) {
            return false;
        }
        if (!updatedAtLeastOnce.contains(sourcePathOfSourceFile)) {
            notSyncIssuesStacktrace.add(ExceptionUtils.getStackTrace(exception));
            return false;
        }
        return updatedAtLeastOnce.contains(sourcePathOfSourceFile);
    }

    private String getSourcePathOfSourceFile(File file) {
        Path javaSourceFolderPath = applicationConfiguration.getJavaSourceFolder().toPath();
        Path path = file.toPath();
        return javaSourceFolderPath.relativize(path).toString();
    }

    public FileWatcher getSourceFileWatcher() {
        return sourceFileWatcher;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SourceSyncChecker.class);
    }
}
