package com.vaadin.copilot.plugins.i18n;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.Optional;
import java.util.ResourceBundle;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ComponentSourceFinder;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaBatchRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.i18n.DefaultI18NProvider;
import com.vaadin.flow.shared.util.SharedUtil;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;

import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import org.apache.commons.configuration2.PropertiesConfiguration;
import org.apache.commons.configuration2.PropertiesConfigurationLayout;
import org.apache.commons.configuration2.ex.ConfigurationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Command handler for i18n related operations
 */
public class I18nHandler implements CopilotCommand {

    private static final String[] propertiesToTranslate = new String[] { "label", "placeholder", "title", "text",
            "helperText" };

    private static final String GET_TRANSLATIONS_COMMAND = "get-translations";
    private static final String WRITE_TRANSLATIONS_COMMAND = "write-translations";
    private static final String SET_TRANSLATABLE_PROPERTIES_COMMAND = "set-translatable-properties";
    private static final String GET_TRANSLATABLE_PROPERTIES_COMMAND = "get-translatable-properties";
    private static final String TRANSLATIONS_COMMAND = Copilot.PREFIX + "translations"; // NOSONAR
    private static final String TRANSLATIONS_FILE_WRITTEN_COMMAND = Copilot.PREFIX + "translations-file-written";
    private static final String TRANSLATABLE_PROPERTIES_SET_COMMAND = Copilot.PREFIX + "translatable-properties-set";
    private static final String TRANSLATABLE_PROPERTIES_GET_COMMAND = Copilot.PREFIX + "translatable-properties-get";

    private static final String TRANSLATIONS_PROPERTY = "translations";
    private static final String TRANSLATIONS_FILES_PROPERTY = "translationsFiles";
    private static final String COMPONENTS_PROPERTY = "components";
    private static final String COMPONENT_PROPERTY = "component";
    private static final String PROPERTIES_PROPERTY = "properties";
    private static final String KEY_PROPERTY = "key";
    private static final String NAME_PROPERTY = "name";

    private static final String STATUS_ERROR = "error";
    private static final String STATUS_ERROR_CODE = "errorCode";

    public static final int GET_TRANSLATION_ERROR = -1;
    public static final int WRITE_TRANSLATION_ERROR = -2;
    public static final int SET_TRANSLATABLE_PROPERTY_ERROR = -3;
    public static final int GET_TRANSLATABLE_PROPERTY_ERROR = -4;

    private static final String WRITE_UNDO_LABEL = "I18N Properties Write";

    private final ProjectManager projectManager;
    private final ComponentSourceFinder sourceFinder;

    /**
     * Create a new i18n handler
     */
    public I18nHandler(ProjectManager projectManager) {
        this(projectManager, new ComponentSourceFinder(projectManager));
    }

    public I18nHandler(ProjectManager projectManager, ComponentSourceFinder sourceFinder) {
        this.projectManager = projectManager;
        this.sourceFinder = sourceFinder;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {

        if (command.equals(GET_TRANSLATIONS_COMMAND)) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);

            try {
                handleGetTranslations(responseData);
            } catch (Exception e) {
                responseData.put(STATUS_ERROR, true);
                responseData.put(STATUS_ERROR_CODE, GET_TRANSLATION_ERROR);
                getLogger().debug("Unable to get project translations", e);
            }

            devToolsInterface.send(TRANSLATIONS_COMMAND, responseData);

            return true;
        } else if (command.equals(WRITE_TRANSLATIONS_COMMAND)) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);

            try {
                handleWriteTranslationsFile(data.getObject(TRANSLATIONS_PROPERTY));
            } catch (Exception e) {
                responseData.put(STATUS_ERROR, true);
                responseData.put(STATUS_ERROR_CODE, WRITE_TRANSLATION_ERROR);
                getLogger().error("Unable to write project translations", e);
            }

            devToolsInterface.send(TRANSLATIONS_FILE_WRITTEN_COMMAND, responseData);
            return true;
        } else if (command.equals(SET_TRANSLATABLE_PROPERTIES_COMMAND)) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);

            try {
                handleSetComponentProperties(data.getArray(COMPONENTS_PROPERTY));
                handleWriteTranslationsFile(data.getObject(TRANSLATIONS_PROPERTY));
            } catch (Exception e) {
                responseData.put(STATUS_ERROR, true);
                responseData.put(STATUS_ERROR_CODE, SET_TRANSLATABLE_PROPERTY_ERROR);
                getLogger().error("Failed to set translatable properties", e);
            }

            devToolsInterface.send(TRANSLATABLE_PROPERTIES_SET_COMMAND, responseData);

            return true;
        } else if (command.equals(GET_TRANSLATABLE_PROPERTIES_COMMAND)) {
            var reqId = data.getString(KEY_REQ_ID);
            var responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, reqId);

            try {
                handleGetTranslatableProperties(data, responseData);
            } catch (Exception e) {
                responseData.put(STATUS_ERROR, true);
                responseData.put(STATUS_ERROR_CODE, GET_TRANSLATABLE_PROPERTY_ERROR);
                getLogger().debug("Failed to get translatable properties", e);
            }
            devToolsInterface.send(TRANSLATABLE_PROPERTIES_GET_COMMAND, responseData);
            return true;
        }

        return false;
    }

    private void handleGetTranslations(JsonObject responseData) throws IOException {
        var translations = Json.createObject();
        responseData.put(TRANSLATIONS_PROPERTY, translations);
        var translationsFiles = Json.createObject();
        responseData.put(TRANSLATIONS_FILES_PROPERTY, translationsFiles);

        var translationsFileDirPath = getTranslationsFileDirPath();

        // Get the content of all translations files in the directory
        if (Files.exists(translationsFileDirPath)) {
            try (Stream<Path> paths = Files.list(translationsFileDirPath)) {
                var translationsFilesList = paths.filter(Files::isRegularFile)
                        .filter(path -> path.getFileName().toString().endsWith(".properties")
                                && path.getFileName().toString().startsWith(DefaultI18NProvider.BUNDLE_FILENAME))
                        .map(Path::getFileName).map(Path::toString).toArray(String[]::new);

                for (var translationsFile : translationsFilesList) {
                    var translationsFilePath = translationsFileDirPath.resolve(translationsFile);
                    var translationsFileContent = projectManager.readFile(translationsFilePath);
                    translationsFiles.put(translationsFile, translationsFileContent);
                }
            }
        }

        // Get the translations from the default translations file
        var translationsFilePath = getTranslationsFilePath();
        if (Files.exists(translationsFilePath)) {
            // Get the translations from the resource bundle and add
            // them to the translations object
            var resourceBundle = getBundle(Locale.ROOT);
            resourceBundle.keySet().forEach(key -> translations.put(key, resourceBundle.getString(key)));
        }
    }

    private void handleWriteTranslationsFile(JsonObject translations) throws IOException, ConfigurationException {
        // If the translations file doesn't exist, create it
        var translationsFilePath = getTranslationsFilePath();
        if (!Files.exists(translationsFilePath)) {
            // do not use ProjectManager as it is not user action
            Files.createDirectories(translationsFilePath.getParent());
            Files.createFile(translationsFilePath);
        }

        // Load the translations file
        var config = new PropertiesConfiguration();
        var layout = new PropertiesConfigurationLayout();
        layout.setGlobalSeparator("=");
        loadConfig(layout, config);

        // Iterate over the translations and update the content
        for (var key : translations.keys()) {
            config.setProperty(key, translations.getString(key));
            // Touch the key, otherwise the property set with
            // setProperty might not be written
            layout.getComment(key);
        }

        // Write the updated content back to the file
        writeConfig(layout, config);
    }

    private void handleSetComponentProperties(JsonArray components) {
        var componentProperties = new HashMap<ComponentTypeAndSourceLocation, JsonArray>();
        for (int i = 0; i < components.length(); i++) {
            var componentObject = components.getObject(i);
            var component = componentObject.getObject(COMPONENT_PROPERTY);
            var componentInfo = sourceFinder.findTypeAndSourceLocation(component);
            var properties = componentObject.getArray(PROPERTIES_PROPERTY);
            componentProperties.put(componentInfo, properties);
        }

        var batchRewriter = new JavaBatchRewriter(projectManager, new ArrayList<>(componentProperties.keySet()));
        batchRewriter.forEachComponent((source, component, rewriter) -> {
            // Add the import for the translate method
            component.rewriter().addImport("com.vaadin.flow.i18n.I18NProvider.translate", true, false);

            // Replace strings with calls to the translate method
            var properties = componentProperties.get(source);
            for (int p = 0; p < properties.length(); p++) {
                var propertyInfo = properties.getObject(p);
                var property = propertyInfo.getString(NAME_PROPERTY);
                var key = propertyInfo.getString(KEY_PROPERTY);
                var code = new JavaRewriter.Code(String.format("translate(\"%s\")", key));

                var setter = getSetterName(property);
                component.rewriter().replaceFunctionCall(component, setter, code);
            }
        });
        batchRewriter.writeResult();
    }

    private void handleGetTranslatableProperties(JsonObject data, // NOSONAR
            JsonObject responseData) throws IOException, IllegalArgumentException, ComponentInfoNotFoundException {

        var component = data.getObject(COMPONENT_PROPERTY);
        var typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        var javaFile = projectManager.getSourceFile(typeAndSourceLocation.createLocation());
        var rewriter = new JavaRewriter(projectManager.readFile(javaFile));

        var info = rewriter.findComponentInfo(typeAndSourceLocation);

        responseData.put(COMPONENT_PROPERTY, component);
        var listOfProps = Json.createArray();
        for (var property : propertiesToTranslate) {
            try {
                var result = rewriter.getPropertyValue(info, property);

                if (result != null) {
                    var propertyObj = Json.createObject();
                    propertyObj.put("name", property);

                    if (result instanceof String value) {
                        propertyObj.put("value", value);
                    } else if (result instanceof MethodCallExpr methodCall
                            && methodCall.getNameAsString().equals("translate")) {
                        var keyValue = getStringArgument(methodCall.getArgument(0));
                        if (keyValue.isPresent()) {
                            propertyObj.put("key", keyValue.get());
                        } else if (methodCall.getArguments().size() > 1) {
                            keyValue = getStringArgument(methodCall.getArgument(1));
                            keyValue.ifPresent(s -> propertyObj.put("key", s));
                        }
                    }

                    if (propertyObj.keys().length == 2) {
                        listOfProps.set(listOfProps.length(), propertyObj);
                    }
                }
            } catch (IllegalArgumentException ignored) {
                // Ignore
            }
        }
        responseData.put(PROPERTIES_PROPERTY, listOfProps);
    }

    private Optional<String> getStringArgument(Expression argument) {
        if (argument.isStringLiteralExpr()) {
            return Optional.of(argument.asStringLiteralExpr().getValue());
        }
        return Optional.empty();
    }

    private String getSetterName(String property) {
        return "set" + SharedUtil.capitalize(property);
    }

    /**
     * Load the translations file content into the configuration
     */
    private void loadConfig(PropertiesConfigurationLayout layout, PropertiesConfiguration config)
            throws IOException, ConfigurationException {
        String content = projectManager.readFile(getTranslationsFilePath());
        layout.load(config, new StringReader(content));
    }

    /**
     * Write the configuration content to the translations file
     */
    private void writeConfig(PropertiesConfigurationLayout layout, PropertiesConfiguration config)
            throws IOException, ConfigurationException {
        StringWriter writer = new StringWriter();
        layout.save(config, writer);
        projectManager.writeFile(getTranslationsFilePath(), WRITE_UNDO_LABEL, writer.toString());
    }

    /**
     * Get the path to the default translations file
     */
    private Path getTranslationsFilePath() {
        return getTranslationsFileDirPath().resolve(DefaultI18NProvider.BUNDLE_FILENAME + ".properties").normalize();
    }

    /**
     * Get the path to the default translations dir
     */
    private Path getTranslationsFileDirPath() {
        return projectManager.getJavaResourceFolder().toPath().resolve(DefaultI18NProvider.BUNDLE_FOLDER).normalize();
    }

    /**
     * Get the project class loader
     */
    private ClassLoader getProjectClassLoader() throws MalformedURLException {
        URL[] urls = new URL[] { projectManager.getProjectRoot().toURI().toURL() };
        return new URLClassLoader(urls);
    }

    /**
     * Get the project i18n resource bundle for the given locale
     */
    private ResourceBundle getBundle(Locale locale) throws MalformedURLException {
        return ResourceBundle.getBundle(DefaultI18NProvider.BUNDLE_PREFIX, locale, getProjectClassLoader());
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
