package com.vaadin.copilot.plugins.testgeneration;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.ProjectManager;
import com.vaadin.copilot.ai.AICommunicationUtil;
import com.vaadin.copilot.ai.AIConstants;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.router.Route;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;
import com.vaadin.uitest.TestCodeGenerator;
import com.vaadin.uitest.ai.utils.PromptUtils;
import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.UiRoute;

import elemental.json.Json;
import elemental.json.JsonObject;
import elemental.json.impl.JsonUtil;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GenerateTestsHandler implements CopilotCommand {

    // TODO: this should be computed by flow in prepare-frontend goal and make
    // it available to the copilot ProjectManager
    private static final String SRC_TEST_JAVA = "src/test/java";
    private static final String GENERATE_TESTS_COMMAND = "generate-tests";
    private static final String RESPONSE_ERROR_KEY = "error";
    private static final String RESPONSE_TEST_GENERATION_RESULT_SUMMARY_KEY = "testGenerationResultSummary";
    private static final String RESPONSE_TEST_GENERATION_GENERATED_FILE_KEY = "testGenerationResultFile";

    private static final Logger LOGGER = LoggerFactory.getLogger(GenerateTestsHandler.class);

    private final ProjectManager projectManager;

    private final TestGenerationServerClient client = new TestGenerationServerClient();

    public GenerateTestsHandler(ProjectManager projectManager) {
        this.projectManager = projectManager;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (!GENERATE_TESTS_COMMAND.equals(command)) {
            return false;
        }

        ProKey proKey = getProKey();
        String machineId = getMachineId();
        if (proKey == null && machineId == null) {
            AICommunicationUtil.promptTextCannotCall(data, devToolsInterface);
            return true;
        }

        UiRoute route = getRoute(data, devToolsInterface);
        if (route == null) {
            throw new IllegalArgumentException("Route cannot be processed, as reference here is the received data: \n"
                    + JsonUtil.stringify(data, 2));
        }

        JsonObject respData = Json.createObject();
        respData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
        respData.put(RESPONSE_TEST_GENERATION_GENERATED_FILE_KEY, route.getTestfile());
        LOGGER.info("Generating '{}' tests for route: '/{}', file '{}'", route.getFramework(),
                route.getRoute() == null ? "" : route.getRoute(), route.getTestfile());
        try {
            // Call copilot server to run AI request
            String generatedTestSource = client.generateTests(proKey, machineId, route, null);

            // Save the generated test to the file-system
            TestCodeGenerator.writeUiTest(route, generatedTestSource);

            // Call the generator to update pom and package files
            String summary = TestCodeGenerator.addTestDependencies(route, projectManager.getProjectRoot().getPath(),
                    getTestFolder().getPath()) ? "Test generated and dependencies updated."
                            : "Test generation successful.";
            respData.put(RESPONSE_TEST_GENERATION_RESULT_SUMMARY_KEY, summary);

        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (Exception e) {
            LOGGER.error("Failed to generate tests", e);
            respData.put(RESPONSE_ERROR_KEY, "Failed to generate tests. See the server log for more details");
        }

        // return the response with the test generated
        devToolsInterface.send(command + "-response", respData);
        return true;
    }

    private UiRoute getRoute(JsonObject data, DevToolsInterface devToolsInterface) {
        // The list of hilla filenames coming in the "sources" data property
        List<UiRoute> sources = new ArrayList<>(getHillaSources(data));
        // The list of flow sources computed in server based on the "uiid" data
        // property
        try {
            sources.addAll(getJavaSources(data));
        } catch (IOException e) {
            LOGGER.error("Error reading requested project Flow Java files", e);
            devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, Json.createObject());
            return null;
        }
        sources.forEach(s -> s.setSource(PromptUtils.cleanJavaCode(s.getSource())));

        // The url of the current view
        String path = getPath(data);

        // Get the source of the view that matches the route. If not found, get
        // the last source in the list
        UiRoute view = sources.stream().filter(s -> s.getRoute() != null && s.getRoute().equals(path)).findFirst()
                .orElse(sources.get(sources.size() - 1));
        if (view.getRoute() == null) {
            view.setRoute(path);
        }
        view.setRoute(getPath(data));
        view.setHtml(getHtml(data));
        view.setBaseUrl(getBaseUrl(data));
        TestFramework fw = "flow".equals(view.getFramework()) ? TestFramework.PLAYWRIGHT_JAVA
                : TestFramework.PLAYWRIGHT_NODE;
        view.computeTestName(getTestFolder(), fw);

        // Assure that computed test file is relative to the project root
        view.setTestfile(projectManager.getProjectRelativeName(new File(view.getTestfile())));

        view.setProjectRoot(projectManager.getProjectRoot().getAbsolutePath());
        return view;
    }

    private File getTestFolder() {
        return new File(projectManager.getProjectRoot(), SRC_TEST_JAVA);
    }

    private String getPath(JsonObject data) {
        if (!data.hasKey("path")) {
            return null;
        }
        return data.getString("path");
    }

    private String getBaseUrl(JsonObject data) {
        if (!data.hasKey("base")) {
            return null;
        }
        return data.getString("base");
    }

    private String getHtml(JsonObject data) {
        if (!data.hasKey("html")) {
            return null;
        }
        return PromptUtils.cleanHtml(data.getString("html"));
    }

    private List<UiRoute> getHillaSources(JsonObject data) {
        Map<String, String> hillaSourceFiles = AICommunicationUtil.getHillaSourceFiles(data);

        return hillaSourceFiles.entrySet().stream().map(e -> {
            UiRoute route = new UiRoute();
            route.setFile(e.getKey());
            route.getFramework();
            route.setClassName(FilenameUtils.getBaseName(e.getKey()));
            route.setSource(e.getValue());
            return route;
        }).toList();
    }

    private List<UiRoute> getJavaSources(JsonObject data) throws IOException {
        if (!data.hasKey("uiid")) {
            return Collections.emptyList();
        }

        List<UiRoute> sources = new ArrayList<>();

        for (Map.Entry<ComponentTracker.Location, File> entry : FlowUtil
                .findActiveJavaFiles(projectManager, (int) data.getNumber("uiid")).entrySet()) {
            ComponentTracker.Location location = entry.getKey();
            File javaFile = entry.getValue();

            UiRoute route = new UiRoute();
            route.setFile(location.filename());
            route.getFramework();
            route.setClassName(location.className());
            route.setRoute(getAnnotationValue(location.className(), Route.class));
            route.setTagName(getAnnotationValue(location.className(), Tag.class));
            route.setSource(projectManager.readFile(javaFile.getPath()));
            sources.add(route);
        }
        return sources;
    }

    public String getAnnotationValue(String className, Class<? extends Annotation> annotationClass) {
        try {
            // Load the class dynamically
            Class<?> clazz = Class.forName(className);

            // Traverse the class hierarchy
            while (clazz != null) {
                // Check if the class has the specified annotation
                if (clazz.isAnnotationPresent(annotationClass)) {
                    // Get the annotation
                    Annotation annotation = clazz.getAnnotation(annotationClass);
                    // Use reflection to get the value method of the annotation
                    Method valueMethod = annotationClass.getMethod("value");
                    // Invoke the value method to get the annotation value
                    return (String) valueMethod.invoke(annotation);
                }
                // Move to the superclass
                clazz = clazz.getSuperclass();
            }
        } catch (Exception e) {
            LOGGER.error("Error getting annotation value: {} - Class: {} - Annotation: {}", e.getMessage(), className,
                    annotationClass.getName());
        }
        return null;
    }

    private static ProKey getProKey() {
        return LocalProKey.get();
    }

    String getMachineId() {
        return MachineId.get();
    }
}
