package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.auth.AnnotatedViewAccessChecker;
import com.vaadin.flow.server.auth.NavigationAccessControl;
import com.vaadin.hilla.EndpointRegistry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.cglib.proxy.Proxy;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

/**
 * Provides Spring related helpers for copilot. Depends on Spring classes and
 * cannot be directly imported
 */
public class SpringIntegration {

    /**
     * Returns the value of the given property from the Spring environment of the
     * given context. See {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @param property
     *            the property name
     * @return the property value or null if not found
     */
    public static String getPropertyValue(VaadinServletContext context, String property) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            getLogger().error("No web application context available");
            return null;
        }
        Environment env = webAppContext.getEnvironment();
        return env.getProperty(property, (String) null);
    }

    /**
     * Returns the Spring Boot application class of the given context. See
     * {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @return the Spring Boot application class or null if not found
     */
    public static Class<?> getApplicationClass(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            getLogger().error("No web application context available");
            return null;
        }
        Map<String, Object> beans = webAppContext.getBeansWithAnnotation(SpringBootApplication.class);
        Class<?> appClass = beans.values().iterator().next().getClass();
        if (Proxy.isProxyClass(appClass)) {
            appClass = appClass.getSuperclass();
        }
        while (isCglibProxy(appClass)) {
            appClass = appClass.getSuperclass();
        }
        return appClass;
    }

    private static boolean isCglibProxy(Class<?> appClass) {
        return appClass.getName().contains("$$SpringCGLIB$$");
    }

    /**
     * Returns whether Spring Security is enabled in the given context See
     * {@link SpringBridge}
     *
     * @param context
     *            the Vaadin servlet context
     * @return true if Spring Security is enabled, false otherwise
     */
    public static Boolean isViewSecurityEnabled(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            return false;
        }
        String[] naviAccessControl = webAppContext.getBeanNamesForType(NavigationAccessControl.class);
        if (naviAccessControl.length != 1) {
            return false;
        }
        NavigationAccessControl accessControl = (NavigationAccessControl) webAppContext.getBean(naviAccessControl[0]);
        return accessControl.hasAccessChecker(AnnotatedViewAccessChecker.class);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SpringIntegration.class);
    }

    public static List<SpringBridge.ServiceMethodInfo> getEndpoints(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            return Collections.emptyList();
        }
        EndpointRegistry endpointRegistry = webAppContext.getBean(EndpointRegistry.class);
        Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints = endpointRegistry.getEndpoints();

        return getEndpointInfos(vaadinEndpoints);
    }

    public static List<SpringBridge.ServiceMethodInfo> getFlowUIServices(VaadinServletContext context) {
        WebApplicationContext webAppContext = WebApplicationContextUtils.getWebApplicationContext(context.getContext());
        if (webAppContext == null) {
            return Collections.emptyList();
        }
        Collection<Object> springServices = webAppContext.getBeansWithAnnotation(Service.class).values();
        Stream<Class<?>> serviceClassCandidates = springServices.stream()
                .map(inst -> Proxy.isProxyClass(inst.getClass()) || isCglibProxy(inst.getClass())
                        ? inst.getClass().getSuperclass()
                        : inst.getClass());
        serviceClassCandidates = serviceClassCandidates
                .filter(cls -> !cls.getPackage().getName().startsWith("com.vaadin"));

        Stream<SpringBridge.ServiceMethodInfo> serviceMethods = serviceClassCandidates
                .flatMap(cls -> Arrays.stream(cls.getMethods()).filter(m -> !isObjectMethod(m))
                        .map(method -> new SpringBridge.ServiceMethodInfo(cls, method)));
        return serviceMethods.toList();
    }

    private static boolean isObjectMethod(Method m) {
        return m.getDeclaringClass() == Object.class;
    }

    private static List<SpringBridge.ServiceMethodInfo> getEndpointInfos(
            Map<String, EndpointRegistry.VaadinEndpointData> vaadinEndpoints) {
        List<SpringBridge.ServiceMethodInfo> endpointInfos = new ArrayList<>();
        for (EndpointRegistry.VaadinEndpointData endpoint : vaadinEndpoints.values()) {
            if (isInternalEndpoint(endpoint)) {
                continue;
            }
            Class<?> endpointClass = endpoint.getEndpointObject().getClass();
            Map<String, Method> methods = endpoint.getMethods();
            for (Method method : methods.values()) {
                endpointInfos.add(new SpringBridge.ServiceMethodInfo(endpointClass, method));
            }
        }
        return endpointInfos;
    }

    private static boolean isInternalEndpoint(EndpointRegistry.VaadinEndpointData endpointData) {
        String name = endpointData.getEndpointObject().getClass().getName();
        return name.startsWith("com.vaadin");
    }
}
