package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

/**
 * Handles reading and writing files in the project
 */
public class ProjectManager {
    private final ApplicationConfiguration applicationConfiguration;
    private final VaadinSession vaadinSession;

    /**
     * Creates a new project manager
     *
     * @param applicationConfiguration
     *            the application configuration
     * @param vaadinSession
     *            the Vaadin session
     * @throws IOException
     *             if the project folder cannot be resolved
     */
    public ProjectManager(ApplicationConfiguration applicationConfiguration, VaadinSession vaadinSession)
            throws IOException {
        this.applicationConfiguration = applicationConfiguration;
        this.vaadinSession = vaadinSession;
    }

    public String readFile(String filename) throws IOException {
        return ProjectFileManager.getInstance().readFile(filename);
    }

    public String readFile(Path filename) throws IOException {
        return ProjectFileManager.getInstance().readFile(filename);
    }

    public String readFile(File file) throws IOException {
        return ProjectFileManager.getInstance().readFile(file);
    }

    public List<String> readLines(File file) throws IOException {
        return ProjectFileManager.getInstance().readLines(file);
    }

    boolean isFileInsideProject(File file) {
        return ProjectFileManager.getInstance().isFileInsideProject(file);
    }

    public static boolean isFileInside(File toCheck, File container) {
        return ProjectFileManager.isFileInside(toCheck, container);
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param filename
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(String filename, String undoLabel, String content) throws IOException {
        ProjectFileManager.getInstance().writeFile(filename, undoLabel, content);
    }

    /**
     * Writes the given content to the given file inside the project.
     *
     * <p>
     * If the filename is absolute, it is used as is. Otherwise, it is resolved
     * relative to the project root.
     *
     * <p>
     * If the file is outside the project, an exception is thrown
     *
     * @param file
     *            the filename to write to, absolute or relative to the project root
     * @param undoLabel
     *            the undo label for the change
     * @param content
     *            the content to write
     * @throws IOException
     *             if the file cannot be written
     */
    public void writeFile(File file, String undoLabel, String content) throws IOException {
        ProjectFileManager.getInstance().writeFile(file, undoLabel, content);
    }

    public void writeFile(Path file, String undoLabel, String content) throws IOException {
        ProjectFileManager.getInstance().writeFile(file, undoLabel, content);
    }

    public File writeFileBase64(String filename, String undoLabel, String base64Content, boolean renameIfExists)
            throws IOException {
        return ProjectFileManager.getInstance().writeFileBase64(filename, undoLabel, base64Content, renameIfExists);
    }

    public String makeAbsolute(String projectRelativeFilename) throws IOException {
        return ProjectFileManager.getInstance().makeAbsolute(projectRelativeFilename);
    }

    public String makeRelative(String filename) throws IOException {
        return ProjectFileManager.getInstance().makeRelative(filename);
    }

    /**
     * Returns the name of the file, relative to the project root.
     *
     * @param projectFile
     *            the file
     * @return the relative name of the file
     */
    public String getProjectRelativeName(File projectFile) {
        return ProjectFileManager.getRelativeName(projectFile, ProjectFileManager.getInstance().getProjectRoot());
    }

    /**
     * Returns the name of the file, relative to the given folder.
     *
     * @param projectFile
     *            the file
     * @param folder
     *            the folder to make relative to
     * @return the relative name of the file
     */
    static String getRelativeName(File projectFile, File folder) {
        return ProjectFileManager.getRelativeName(projectFile, folder);
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(Class<?> cls) {
        return ProjectFileManager.getInstance().getFileForClass(cls);
    }

    /**
     * Returns the Java file for the given class.
     *
     * @param cls
     *            the class
     * @return the file for the class
     */
    public File getFileForClass(String cls) {
        return ProjectFileManager.getInstance().getFileForClass(cls);
    }

    /**
     * Returns the source folders for the project.
     *
     * @return the source folders
     */
    public List<Path> getSourceFolders() {
        return ProjectFileManager.getInstance().getSourceFolders();
    }

    /**
     * Returns the Java file for the given component location.
     *
     * @param location
     *            the component location
     * @return the file for the class where the component is used
     */
    public File getSourceFile(ComponentTracker.Location location) {
        return getFileForClass(location.className());
    }

    /**
     * Returns the project base folder.
     * <p>
     * The project base folder is the common folder containing all module roots.
     *
     * @return the project root folder
     */
    public File getProjectRoot() {
        return ProjectFileManager.getInstance().getProjectRoot();
    }

    /**
     * Returns the frontend folder.
     *
     * @return the frontend folder
     */
    public File getFrontendFolder() {
        return ProjectFileManager.getInstance().getFrontendFolder();
    }

    /**
     * Returns the java resource folder.
     *
     * @return the java resource folder.
     */
    public File getJavaResourceFolder() {
        return ProjectFileManager.getInstance().getJavaResourceFolder();
    }

    /**
     * Gets current theme name
     *
     * @return optional theme name
     */
    public Optional<String> getThemeName() {
        return ProjectFileManager.getInstance().getThemeName();
    }

    /**
     * Gets current theme folder
     *
     * @return optional theme folder
     */
    public Optional<File> getThemeFolder() {
        return ProjectFileManager.getInstance().getThemeFolder();
    }

    /**
     * Makes a string safe to use as a file name
     *
     * @param name
     *            the string to process
     * @return the sanitized string
     */
    public String sanitizeFilename(String name) {
        return ProjectFileManager.getInstance().sanitizeFilename(name);
    }

    /**
     * Finds the folder where Hilla views should be created for the file system
     * router to pick them up.
     *
     * @return the folder where Hilla views should be created
     */
    public File getHillaViewsFolder() {
        return ProjectFileManager.getInstance().getHillaViewsFolder();
    }

    /**
     * Finds the folder where a new Flow view should be created.
     * <p>
     * If all views are in the same package/folder, that folder is returned. If
     * views are in different packages, the folder for a common prefix is returned.
     * If no common prefix is found, the folder for the main package is returned.
     *
     * @return a suitable folder to place a new Flow view in
     */
    public File getFlowNewViewFolder() {
        Map<String, Class<?>> viewPackages = new HashMap<>();
        vaadinSession.accessSynchronously(() -> {
            RouteRegistry routeRegistry = vaadinSession.getService().getRouter().getRegistry();
            for (RouteData route : routeRegistry.getRegisteredRoutes()) {
                viewPackages.put(route.getNavigationTarget().getPackageName(), route.getNavigationTarget());
            }
        });

        if (!viewPackages.isEmpty()) {
            // There is at least one view, so we can figure out a good package name from the
            // views

            File someViewJavaFile = getFileForClass(viewPackages.values().iterator().next());
            if (viewPackages.size() == 1) {
                return someViewJavaFile.getParentFile();
            }

            // Find common package prefix
            String commonPrefix = "";
            for (String viewPackage : viewPackages.keySet()) {
                if (commonPrefix.isEmpty()) {
                    commonPrefix = viewPackage;
                } else if (viewPackage.startsWith(commonPrefix)) {
                    continue;
                } else {
                    while (!commonPrefix.isEmpty() && !viewPackage.startsWith(commonPrefix)) {
                        commonPrefix = commonPrefix.substring(0, commonPrefix.lastIndexOf('.'));
                    }
                }
            }

            Optional<Path> sourceFolder = findSourceFolder(someViewJavaFile);
            if (!commonPrefix.isEmpty()) {
                for (String folder : commonPrefix.split("\\.")) {
                    sourceFolder = sourceFolder.map(p -> p.resolve(folder));
                }

                if (sourceFolder.isPresent()) {
                    return sourceFolder.get().toFile();
                }
            }
        }
        // Project might not have any Flow views yet
        var sourceFolder = getSourceFolders().get(0);
        if (SpringBridge.isSpringAvailable()) {
            Class<?> springApplicationClass = SpringBridge.getApplicationClass(getVaadinContext());
            if (springApplicationClass != null) {
                Optional<Path> folder = findSourceFolder(springApplicationClass);
                if (folder.isPresent()) {
                    sourceFolder = folder.get();
                }
            }
        }
        File folder = Util.getSinglePackage(sourceFolder.toFile());
        if (folder.getName().equals("views")) {
            return folder;
        }
        return new File(folder, "views");
    }

    /**
     * Finds the source folder where the given class is located.
     *
     * @param cls
     *            the class to find the source folder for
     * @return the source folder or an empty optional if the source folder could not
     *         be found
     * @throws IOException
     *             if something goes wrong
     */
    Optional<Path> findSourceFolder(Class<?> cls) {
        File sourceFile = getFileForClass(cls);
        if (!sourceFile.exists()) {
            return Optional.empty();
        }

        return findSourceFolder(sourceFile);
    }

    /**
     * Finds the given resource by name, in any resource folder.
     *
     * @param resource
     *            the name of the resource to look for
     */
    public Optional<Path> findResource(String resource) {
        return ProjectFileManager.getInstance().findResource(resource);
    }

    /**
     * Finds the source folder where the given file is located.
     *
     * @param sourceFile
     *            the source file to find the source folder for
     */
    public Optional<Path> findSourceFolder(File sourceFile) {
        return ProjectFileManager.getInstance().findSourceFolder(sourceFile);
    }

    public VaadinSession getVaadinSession() {
        return vaadinSession;
    }

    protected ApplicationConfiguration getApplicationConfiguration() {
        return applicationConfiguration;
    }

    private VaadinServletContext getVaadinContext() {
        return (VaadinServletContext) getVaadinSession().getService().getContext();
    }

    /**
     * Gets the java package for the given source file
     *
     * @param sourceFile
     *            the source file
     * @return the java package for the given source file, or null if the source
     *         file is not inside the project
     * @throws IOException
     *             if something goes wrong
     */
    public String getJavaPackage(File sourceFile) throws IOException {
        return ProjectFileManager.getInstance().getJavaPackage(sourceFile);
    }
}
