package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;
import com.vaadin.flow.server.VaadinServletContext;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides endpoint information to the client.
 */
public class UiServiceHandler implements CopilotCommand {
    private final ProjectManager projectManager;
    private final VaadinServletContext context;

    /**
     * Creates a new endpoint handler.
     *
     * @param projectManager
     *            the project manager to use
     * @param context
     *            the servlet context to use
     */
    public UiServiceHandler(ProjectManager projectManager, VaadinServletContext context) {
        this.projectManager = projectManager;
        this.context = context;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-browser-callables")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            List<SpringBridge.ServiceMethodInfo> browserCallables = EndpointRequestUtil.isHillaAvailable()
                    ? SpringBridge.getEndpoints(context)
                    : new ArrayList<>();

            JsonArray browserCallableJson = browserCallables.stream().sorted(UiServiceHandler::sortByClassAndMethodName)
                    .map(serviceMethod -> serviceMethodToJson(serviceMethod, true)).collect(JsonUtils.asArray());
            returnData.put("browserCallables", browserCallableJson);

            JsonArray flowUiServicesJson;
            if (SpringBridge.isSpringAvailable()) {
                List<SpringBridge.ServiceMethodInfo> flowUIServices = SpringBridge.getFlowUIServices(context);
                flowUiServicesJson = flowUIServices.stream().sorted(UiServiceHandler::sortByClassAndMethodName)
                        .map(serviceMethod -> serviceMethodToJson(serviceMethod, false)).collect(JsonUtils.asArray());
            } else {
                flowUiServicesJson = Json.createArray();
            }

            returnData.put("flowServices", flowUiServicesJson);

            devToolsInterface.send("resp" + command, returnData);
            return true;
        }
        return false;
    }

    private JsonObject serviceMethodToJson(SpringBridge.ServiceMethodInfo serviceMethodInfo,
            boolean availableInTypescript) {
        JsonObject json = Json.createObject();
        // canonical name can be null e.g. for anonymous classes

        json.put("className", JavaReflectionUtil.getClassName(serviceMethodInfo.serviceClass()));
        json.put("filename", projectManager.getFileForClass(serviceMethodInfo.serviceClass()).getAbsolutePath());
        json.put("lineNumber", 1);
        json.put("methodName", serviceMethodInfo.serviceMethod().getName());
        List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes = JavaReflectionUtil
                .getParameterTypes(serviceMethodInfo.serviceMethod(), serviceMethodInfo.serviceClass());

        json.put("parameters", parameterTypes.stream().map(param -> {
            JsonObject paramInfo = Json.createObject();
            paramInfo.put("name", param.parameterName());
            paramInfo.put("type", typeToJson(param.typeInfo()));
            return paramInfo;
        }).collect(JsonUtils.asArray()));
        JavaReflectionUtil.TypeInfo returnTypeInfo = JavaReflectionUtil.getReturnType(serviceMethodInfo.serviceMethod(),
                serviceMethodInfo.serviceClass());

        json.put("returnType", typeToJson(returnTypeInfo));
        json.put("availableInTypescript", availableInTypescript);
        try {
            AccessRequirement req = AccessRequirementUtil.getAccessRequirement(serviceMethodInfo.serviceMethod(),
                    serviceMethodInfo.serviceClass());
            json.put("accessRequirement", JsonUtils.beanToJson(req));
        } catch (Exception e) {
            getLogger().error("Unable to determine access requirement", e);
        }
        return json;
    }

    private JsonObject typeToJson(JavaReflectionUtil.TypeInfo typeInfo) {
        JsonObject json = Json.createObject();
        json.put("typeName", typeInfo.typeName());
        json.put("typeParameters",
                typeInfo.typeParameters().stream().map(this::typeToJson).collect(JsonUtils.asArray()));
        return json;
    }

    public static int sortByClassAndMethodName(SpringBridge.ServiceMethodInfo e1, SpringBridge.ServiceMethodInfo e2) {
        return sortByClassAndMethodName(e1.serviceClass(), e1.serviceMethod(), e2.serviceClass(), e2.serviceMethod());
    }

    private static int sortByClassAndMethodName(Class<?> class1, Method method1, Class<?> class2, Method method2) {
        if (!class1.equals(class2)) {
            return JavaReflectionUtil.getClassName(class1).compareTo(JavaReflectionUtil.getClassName(class2));
        } else {
            return method1.getName().compareTo(method2.getName());
        }

    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(UiServiceHandler.class);
    }
}
