package com.vaadin.copilot;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ErrorHandler extends CopilotServerClient implements CopilotCommand {

    private static final Logger logger = LoggerFactory.getLogger(ErrorHandler.class);

    public static final String ERROR_KEY = "error";
    public static final String ERROR_EXCEPTION_MESSAGE_KEY = "errorMessage";
    public static final String ERROR_STACKTRACE_KEY = "errorStacktrace";

    public static JsonArray toJson(Exception e) {
        StringWriter sw = new StringWriter();
        try (PrintWriter pw = new PrintWriter(sw)) {
            e.printStackTrace(pw);
        }
        // We don't need the stack trace after
        // "com.vaadin.base.devserver.DebugWindowConnection.onMessage" as that
        // is outside Copilot
        // But we want all "causes" parts of the stack trace
        String[] stack = sw.toString().split("\n");

        AtomicBoolean include = new AtomicBoolean(true);
        Stream<String> filtered = Arrays.stream(stack).filter(s -> {
            if (s.contains("com.vaadin.base.devserver.DebugWindowConnection.onMessage")) {
                include.set(false);
            }
            if (!s.startsWith("\t")) {
                include.set(true);
            }
            return include.get();
        });
        return filtered.map(Json::create).collect(JsonUtils.asArray());
    }

    public static void setError(JsonObject respData, String error) {
        respData.put(ErrorHandler.ERROR_KEY, error);
    }

    public static void setError(JsonObject respData, String error, Exception e) {
        setError(respData, error);
        if (e != null) {
            String exceptionMessage = e.getMessage();
            if (exceptionMessage == null) {
                exceptionMessage = "";
            }
            respData.put(ERROR_EXCEPTION_MESSAGE_KEY, exceptionMessage);
            respData.put(ERROR_STACKTRACE_KEY, ErrorHandler.toJson(e));
            if (Copilot.isDevelopmentMode()) {
                logger.error("Stack trace because in Copilot development mode", e);
            }
        }
    }

    public static void sendErrorResponse(DevToolsInterface devToolsInterface, String command, JsonObject responseData,
            String error, Exception e) {
        ErrorHandler.setError(responseData, error, e);
        devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
    }

    private final boolean sendErrors;

    public ErrorHandler(boolean sendErrors) {
        this.sendErrors = sendErrors;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if ("error".equals(command)) {

            if (!sendErrors) {
                return true;
            }

            String proKey = LocalProKey.get().getProKey();
            String machineId = MachineId.get();
            URI uri = getQueryURI("errors");
            ErrorsRequest trackingRequest = new ErrorsRequest(machineId, proKey, data.getString("message"),
                    Map.of("details", data.getString("details"), "versions", data.getString("versions")));
            String json = writeAsJsonString(trackingRequest);
            HttpRequest request = buildRequest(uri, json);
            getHttpClient().sendAsync(request, HttpResponse.BodyHandlers.ofString());
            return true;
        }
        return false;
    }
}
