package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaBatchRewriter;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaDataProviderHandler;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterCopyPasteHandler;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles commands to rewrite Java source code.
 */
public class JavaRewriteHandler implements CopilotCommand {

    public static final String UNDO_LABEL = CopilotIDEPlugin.undoLabel("Java view update");
    private final ProjectManager projectManager;
    private final SourceSyncChecker sourceSyncChecker;
    private final ComponentSourceFinder sourceFinder;

    private interface Handler {
        void handle(JsonObject data, JsonObject respData) throws IOException;
    }

    private static class RewriteHandler {
        private final String what;
        private final Handler handler;

        public RewriteHandler(String what, Handler handler) {
            this.what = what;
            this.handler = handler;
        }

        public void handle(JsonObject data, JsonObject respData) throws IOException {
            this.handler.handle(data, respData);
        }

        public String getWhat() {
            return what;
        }
    }

    private final Map<String, RewriteHandler> handlers = new HashMap<>();

    /**
     * Creates the one and only handler.
     *
     * @param projectManager
     *            the project manager
     * @param sourceSyncChecker
     *            the source sync checker for detecting out of sync scenarios
     */
    public JavaRewriteHandler(ProjectManager projectManager, SourceSyncChecker sourceSyncChecker) {
        this.projectManager = projectManager;
        this.sourceSyncChecker = sourceSyncChecker;
        this.sourceFinder = new ComponentSourceFinder(projectManager);

        handlers.put("set-component-property",
                new RewriteHandler("set component property", this::handleSetComponentProperty));
        handlers.put("add-call", new RewriteHandler("add call", this::handleAddCall));
        handlers.put("add-template", new RewriteHandler("add call", this::handleAddTemplate));
        handlers.put("delete-components", new RewriteHandler("delete components", this::handleDeleteComponents));
        handlers.put("duplicate-components",
                new RewriteHandler("duplicate components", this::handleDuplicateComponents));
        handlers.put("drag-and-drop", new RewriteHandler("drop component", this::handleDragAndDrop));
        handlers.put("set-alignment", new RewriteHandler("set alignment", this::handleAlignment));
        handlers.put("set-gap", new RewriteHandler("set gap", this::handleGap));
        handlers.put("wrap-with", new RewriteHandler("wrap with", this::handleWrapWith));
        handlers.put("set-styles", new RewriteHandler("set styles", this::handleSetStyles));
        handlers.put("set-padding", new RewriteHandler("set padding", this::handlePadding));
        handlers.put("copy", new RewriteHandler("copy", this::handleCopy));
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        RewriteHandler handler = handlers.get(command);
        if (handler == null) {
            return false;
        }
        String reqId = data.getString(KEY_REQ_ID);
        JsonObject respData = Json.createObject();
        respData.put(KEY_REQ_ID, reqId);

        try {
            handler.handle(data, respData);
            devToolsInterface.send(command + "-response", respData);
        } catch (ComponentInfoNotFoundException e) {
            if (sourceSyncChecker.maybeOutOfSync(e)) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData,
                        e.getComponentTypeAndSourceLocation().javaFile().getName()
                                + " may be out of sync. Please recompile and deploy the file",
                        e);
            } else {
                getLogger().debug("Failed to {} for input {}", handler.getWhat(), data.toJson(), e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to " + handler.getWhat(),
                        e);
            }
        } catch (Exception e) {
            getLogger().debug("Failed to {} for input {}", handler.getWhat(), data.toJson(), e);
            ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to " + handler.getWhat(), e);
        }

        return true;
    }

    private void handleAddCall(JsonObject data, JsonObject respData) throws IOException {
        String func = data.getString("func");
        String parameter = data.getString("parameter");
        Integer lineToShowInIde = data.hasKey("lineToShowInIde") ? (int) data.getNumber("lineToShowInIde") : null;
        var component = data.getObject("component");

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        File javaFile = projectManager.getSourceFile(typeAndSourceLocation.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo info = rewriter.findComponentInfo(typeAndSourceLocation);
        if (!JavaRewriterUtil.hasSingleParameterMethod(info.type(), func)) {
            throw new IllegalArgumentException("Component does not support the given method");
        }
        rewriter.addCall(info, func, new JavaRewriter.Code(parameter));
        String result = rewriter.getResult();
        projectManager.writeFile(javaFile, UNDO_LABEL, result);

        if (lineToShowInIde != null) {
            int lineNumber = rewriter.getFirstModifiedRow() + lineToShowInIde;
            IdeUtils.openFile(javaFile, lineNumber);
            CompletableFuture.runAsync(() -> {
                try {
                    // Workaround for
                    // https://youtrack.jetbrains.com/issue/IDEA-342750
                    Thread.sleep(1000);
                    IdeUtils.openFile(javaFile, lineNumber);
                } catch (InterruptedException e) {
                    getLogger().error("Failed to show file in IDE", e);
                    Thread.currentThread().interrupt();
                }
            });
        }
    }

    private void handleDeleteComponents(JsonObject data, JsonObject respData) {
        JsonArray componentsJson = data.getArray("components");
        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            List<ComponentTypeAndSourceLocation> found = findTypeAndSourceLocationIncludingChildren(
                    componentsJson.getObject(i));
            components.addAll(found);
        }

        JavaBatchRewriter batchRewriter = new JavaBatchRewriter(projectManager, components);

        batchRewriter.deleteAll();
        batchRewriter.writeResult();
    }

    private List<ComponentTypeAndSourceLocation> findTypeAndSourceLocationIncludingChildren(JsonObject object) {
        ArrayList<ComponentTypeAndSourceLocation> all = new ArrayList<>();
        ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(object, true);

        addRecursively(all, root);
        return all;
    }

    private void addRecursively(ArrayList<ComponentTypeAndSourceLocation> all, ComponentTypeAndSourceLocation root) {
        all.add(root);
        for (ComponentTypeAndSourceLocation child : root.children()) {
            if (child.javaFile().exists()) {
                addRecursively(all, child);
            } else {
                getLogger().debug(
                        "Excluding file {} because it does not exist. Assuming this is a component created internally by the parent component and not from the project source",
                        child.javaFile());
            }
        }
    }

    private void handleCopy(JsonObject data, JsonObject response) throws IOException {
        int componentId = (int) data.getNumber("componentId");

        int uiId = (int) data.getNumber("uiId");
        ComponentTypeAndSourceLocation copiedComponent = sourceFinder.findTypeAndSourceLocation(uiId, componentId,
                true);

        JavaRewriterCopyPasteHandler handler = new JavaRewriterCopyPasteHandler(projectManager);
        JavaComponent copiedJavaComponent = handler.getCopiedJavaComponent(copiedComponent);
        String s = new ObjectMapper().writeValueAsString(copiedJavaComponent);
        response.put("component", s);
    }

    private void handleDuplicateComponents(JsonObject data, JsonObject respData) {
        JsonArray componentsJson = data.getArray("components");
        ArrayList<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        List<ComponentTypeAndSourceLocation> selectedComponents = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(componentsJson.getObject(i),
                    true);
            selectedComponents.add(root);
            addRecursively(components, root);
        }

        JavaBatchRewriter batchRewriter = new JavaBatchRewriter(projectManager, components);
        selectedComponents.forEach(batchRewriter::duplicate);
        batchRewriter.writeResult();
    }

    private void handleWrapWith(JsonObject data, JsonObject respData) throws IOException {
        JsonArray componentsJson = data.getArray("components");
        JavaComponent wrapperComponent = JavaComponent.componentFromJson(data.getObject("wrapperComponent"));
        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            components.add(sourceFinder.findTypeAndSourceLocation(componentsJson.getObject(i)));
        }
        File javaFile = projectManager.getSourceFile(components.get(0).createLocation());

        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));

        List<ComponentInfo> componentInfos = components.stream().map(rewriter::findComponentInfo).toList();

        rewriter.mergeAndReplace(componentInfos, wrapperComponent);
        String result = rewriter.getResult();
        projectManager.writeFile(javaFile, UNDO_LABEL, result);
    }

    private void handleSetComponentProperty(JsonObject data, JsonObject respData) throws IOException {
        String property = data.getString("property");
        String value = data.getString("value");
        var component = data.getObject("component");

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        if (JavaDataProviderHandler.isDataProviderItemChange(typeAndSourceLocation)) {
            JavaDataProviderHandler dataProviderHandler = new JavaDataProviderHandler(projectManager,
                    typeAndSourceLocation);
            JavaDataProviderHandler.JavaDataProviderHandlerResult javaDataProviderHandlerResult = dataProviderHandler
                    .handleSetComponentProperty(property, value);
            projectManager.writeFile(javaDataProviderHandlerResult.file(), UNDO_LABEL,
                    javaDataProviderHandlerResult.result());
            return;
        }
        File javaFile = projectManager.getSourceFile(typeAndSourceLocation.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo info = rewriter.findComponentInfo(typeAndSourceLocation);
        String setter = JavaRewriterUtil.getSetterName(property, info.type(), true);
        rewriter.replaceFunctionCall(info, setter, value);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleAddTemplate(JsonObject data, JsonObject respData) throws IOException {
        List<JavaComponent> template = JavaComponent.componentsFromJson(data.getArray("template"));
        JavaRewriter.Where where = JavaRewriter.Where.valueOf(data.getString("where").toUpperCase(Locale.ENGLISH));

        ComponentTypeAndSourceLocation refSource = sourceFinder.findTypeAndSourceLocation(data.getObject("refNode"));

        File javaFile = projectManager.getSourceFile(refSource.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo ref = rewriter.findComponentInfo(refSource);
        if (where == JavaRewriter.Where.APPEND) {
            rewriter.addComponentUsingTemplate(ref, where, template);
        } else {
            if (!refSource.parent().javaFile().equals(refSource.javaFile())) {
                throw new IllegalArgumentException(
                        "Cannot insert before a component in one file (" + refSource.javaFile()
                                + ") when the parent is in another file (" + refSource.parent().javaFile() + ")");
            }
            rewriter.addComponentUsingTemplate(ref, where, template);
        }
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleDragAndDrop(JsonObject data, JsonObject respData) throws IOException {
        JavaRewriter.Where where = JavaRewriter.Where.valueOf(data.getString("where").toUpperCase(Locale.ENGLISH));

        ComponentTypeAndSourceLocation dragged = sourceFinder.findTypeAndSourceLocation(data.getObject("dragged"));
        ComponentTypeAndSourceLocation container = sourceFinder.findTypeAndSourceLocation(data.getObject("container"));

        ComponentTypeAndSourceLocation insertBefore = where == JavaRewriter.Where.BEFORE
                ? sourceFinder.findTypeAndSourceLocation(data.getObject("insertBefore"))
                : null;

        File javaFile = projectManager.getSourceFile(container.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));

        if (!dragged.javaFile().equals(container.javaFile())) {
            throw new IllegalArgumentException("Cannot move a component in one file (" + dragged.javaFile()
                    + ") to another file (" + container.javaFile() + ")");
        }

        ComponentInfo draggedRef = rewriter.findComponentInfo(dragged);
        ComponentInfo containerRef = rewriter.findComponentInfo(container);
        ComponentInfo insertBeforeRef = insertBefore == null ? null : rewriter.findComponentInfo(insertBefore);

        rewriter.moveComponent(draggedRef, containerRef, insertBeforeRef, where);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleAlignment(JsonObject data, JsonObject respData) throws IOException {
        JavaRewriter.AlignmentMode mode = JavaRewriter.AlignmentMode
                .valueOf(data.getString("alignmentMode").replace('-', '_').toUpperCase(Locale.ENGLISH));
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject("componentId"));
        JsonArray lumoClassesJson = data.getArray("lumoClasses");
        List<String> lumoClasses = new ArrayList<>();
        for (int i = 0; i < lumoClassesJson.length(); i++) {
            lumoClasses.add(lumoClassesJson.getString(i));
        }
        boolean selected = data.getBoolean("selected");
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        rewriter.setAlignment(componentInfo, mode, selected, lumoClasses);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleSetStyles(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject("componentId"));
        JsonArray added = data.getArray("added");
        JsonArray removed = data.getArray("removed");
        Set<String> toRemove = new HashSet<>();
        for (int i = 0; i < removed.length(); i++) {
            toRemove.add(removed.getObject(i).getString("key"));
        }

        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);

        for (int i = 0; i < added.length(); i++) {
            JsonObject rule = added.getObject(i);
            String key = rule.getString("key");
            String value = rule.getString("value");
            rewriter.setStyle(componentInfo, key, value);
            toRemove.remove(key);
        }

        for (String key : toRemove) {
            rewriter.setStyle(componentInfo, key, null);
        }
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handleGap(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject("componentId"));
        String newValue = data.getString("newValue");
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        rewriter.setGap(componentInfo, newValue);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private void handlePadding(JsonObject data, JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject("componentId"));
        String newClassName = data.hasKey("newClassName") ? data.getString("newClassName") : null;
        File javaFile = projectManager.getSourceFile(componentTypeAndSourceLocation.createLocation());
        JavaRewriter rewriter = new JavaRewriter(projectManager.readFile(javaFile));
        ComponentInfo componentInfo = rewriter.findComponentInfo(componentTypeAndSourceLocation);
        rewriter.setPadding(componentInfo, newClassName);
        projectManager.writeFile(javaFile, UNDO_LABEL, rewriter.getResult());
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
