package com.vaadin.copilot.javarewriter.custom;

import java.util.ArrayList;
import java.util.List;

import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.FlowComponentQuirks;
import com.vaadin.copilot.javarewriter.InsertionPoint;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.ClassExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.nodeTypes.NodeWithArguments;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

/** Crud component handler. */
public class CrudComponentHandle extends CustomComponentHandle {

    @Override
    public List<VariableDeclarator> createComponentStatements(JavaRewriter javaRewriter, JavaComponent component,
            InsertionPoint insertionPoint, JavaComponent parent, String layoutVariableName,
            ComponentInfo referenceComponent) {

        String componentClassName = FlowComponentQuirks.getClassForComponent(component);
        ClassOrInterfaceType fullType = StaticJavaParser.parseClassOrInterfaceType(componentClassName); // Import

        JavaRewriterUtil.addImport(javaRewriter.getCompilationUnit(), fullType.getNameWithScope());
        javaRewriter.addImport("com.vaadin.flow.component.crud.BinderCrudEditor", false, false);
        javaRewriter.addImport("com.vaadin.flow.data.binder.Binder", false, false);

        JavaComponent grid = component.withTag("Grid");
        List<VariableDeclarator> gridDeclaration = javaRewriter.createComponentStatements(insertionPoint, parent, grid,
                false, layoutVariableName, referenceComponent);
        if (gridDeclaration.isEmpty()) {
            throw new IllegalArgumentException("Could not generate Grid for the Crud component");
        }
        VariableDeclarator gridVariableDeclarator = gridDeclaration.get(0);
        String gridVariableName = gridVariableDeclarator.getNameAsString();
        String crudVariableName = JavaRewriterUtil.getNewComponentName(component, fullType, insertionPoint);

        ClassExpr beanClassExpression = getBeanClassExpressionFromGrid(gridVariableDeclarator);

        fullType.setTypeArguments(beanClassExpression.getType());
        VariableDeclarator decl = new VariableDeclarator(fullType.removeScope(), crudVariableName);

        ObjectCreationExpr expr = new ObjectCreationExpr();
        expr.setType("Crud");

        ClassOrInterfaceType toDeleteType = new ClassOrInterfaceType();
        toDeleteType.setName("DELETE_THIS");
        expr.getType().setTypeArguments(toDeleteType);
        decl.setInitializer(expr);

        expr.getArguments().add(beanClassExpression);
        expr.getArguments().add(new NameExpr(gridVariableName));
        expr.getArguments().add(new NameExpr("new BinderCrudEditor<>(new Binder<>())"));

        VariableDeclarationExpr declarationExpr = new VariableDeclarationExpr(decl);

        insertionPoint.add(new ExpressionStmt(declarationExpr));
        javaRewriter.attachComponent(insertionPoint, component, parent, layoutVariableName, referenceComponent,
                new NameExpr(crudVariableName), crudVariableName);

        return new ArrayList<>();
    }

    private ClassExpr getBeanClassExpressionFromGrid(VariableDeclarator declarator) {
        if (declarator.getInitializer().isPresent()) {
            Expression expression = declarator.getInitializer().get();
            if (expression instanceof NodeWithArguments<?> nodeWithArguments) {
                for (Expression argument : nodeWithArguments.getArguments()) {
                    if (argument instanceof ClassExpr) {
                        return (ClassExpr) argument;
                    }
                }
            }
        }
        throw new IllegalArgumentException("Unable to locate bean type of the grid");
    }
}
