package com.vaadin.copilot;

import java.util.ArrayList;
import java.util.List;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;
import com.vaadin.flow.server.VaadinServletContext;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides endpoint information to the client.
 */
public class UiServiceHandler implements CopilotCommand {
    private final ProjectManager projectManager;
    private final VaadinServletContext context;

    /**
     * Creates a new endpoint handler.
     *
     * @param projectManager
     *            the project manager to use
     * @param context
     *            the servlet context to use
     */
    public UiServiceHandler(ProjectManager projectManager, VaadinServletContext context) {
        this.projectManager = projectManager;
        this.context = context;
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-browser-callables")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            List<SpringBridge.EndpointInfo> browserCallables = EndpointRequestUtil.isHillaAvailable()
                    ? SpringBridge.getEndpoints(context)
                    : new ArrayList<>();

            JsonArray browserCallableJson = browserCallables.stream().sorted(UiServiceHandler::sortByClassName)
                    .map(browserCallable -> {
                        JsonObject json = Json.createObject();
                        json.put("className", browserCallable.endpointClass().getName());
                        json.put("filename",
                                projectManager.getFileForClass(browserCallable.endpointClass()).getAbsolutePath());
                        json.put("lineNumber", 1);
                        json.put("methodName", browserCallable.method().getName());
                        List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes = JavaReflectionUtil
                                .getParameterTypes(browserCallable.method(), browserCallable.endpointClass());

                        json.put("parameters", parameterTypes.stream().map(param -> {
                            JsonObject paramInfo = Json.createObject();
                            paramInfo.put("name", param.parameterName());
                            paramInfo.put("type", typeToJson(param.typeInfo()));
                            return paramInfo;
                        }).collect(JsonUtils.asArray()));
                        JavaReflectionUtil.TypeInfo returnTypeInfo = JavaReflectionUtil
                                .getReturnType(browserCallable.method(), browserCallable.endpointClass());

                        json.put("returnType", typeToJson(returnTypeInfo));
                        try {
                            AccessRequirement req = AccessRequirementUtil.getAccessRequirement(browserCallable.method(),
                                    browserCallable.endpointClass());
                            json.put("accessRequirement", JsonUtils.beanToJson(req));
                        } catch (Exception e) {
                            getLogger().error("Unable to determine access requirement", e);
                        }
                        return json;
                    }).collect(JsonUtils.asArray());

            returnData.put("browserCallables", browserCallableJson);

            devToolsInterface.send("resp" + command, returnData);
            return true;
        }
        return false;
    }

    private JsonObject typeToJson(JavaReflectionUtil.TypeInfo typeInfo) {
        JsonObject json = Json.createObject();
        json.put("typeName", typeInfo.typeName());
        json.put("typeParameters",
                typeInfo.typeParameters().stream().map(this::typeToJson).collect(JsonUtils.asArray()));
        return json;
    }

    public static int sortByClassName(SpringBridge.EndpointInfo e1, SpringBridge.EndpointInfo e2) {
        if (!e1.endpointClass().equals(e2.endpointClass())) {
            return e1.endpointClass().getName().compareTo(e2.endpointClass().getName());
        } else {
            return e1.method().getName().compareTo(e2.method().getName());
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(UiServiceHandler.class);
    }
}
