package com.vaadin.copilot.javarewriter;

import static com.vaadin.copilot.JavaRewriteHandler.UNDO_LABEL;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotException;
import com.vaadin.copilot.ProjectManager;

import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.ObjectCreationExpr;

public class JavaBatchRewriter {
    private final ProjectManager projectManager;
    private final Map<File, JavaRewriter> rewriters = new HashMap<>();
    private final Map<ComponentTypeAndSourceLocation, ComponentInfo> components;

    public JavaBatchRewriter(ProjectManager projectManager, List<ComponentTypeAndSourceLocation> componentLocations) {
        this.projectManager = projectManager;

        // Parse all needed Java files
        Stream<ComponentTypeAndSourceLocation> componentsInSource = componentLocations.stream()
                .filter(componentTypeAndSourceLocation -> componentTypeAndSourceLocation.javaFile().isPresent())
                .distinct();

        this.components = componentsInSource.collect(Collectors.toMap(component -> component, component -> {
            File file = component.javaFile().orElseThrow();
            JavaRewriter rewriter = rewriters.computeIfAbsent(file, (newFile) -> {
                try {
                    return new JavaRewriter(projectManager.readFile(newFile));
                } catch (IOException e) {
                    throw new CopilotException(e);
                }
            });
            return rewriter.findComponentInfo(component);
        }));
    }

    public void deleteAll() {
        forEachComponent((source, component, rewriter) -> {
            if (component.objectCreationExpr() == null) {
                // It is not clear when this is needed, or if it is needed. It is here only
                // because it was in the original code.
                return;
            }
            component.rewriter().delete(component);
        });
    }

    public void duplicate(ComponentTypeAndSourceLocation componentSource) {
        ComponentInfo component = components.get(componentSource);
        if (component == null) {
            throw new IllegalArgumentException("The component to be duplicated must be included in the "
                    + getClass().getSimpleName() + " constructor call");
        }
        if (component.routeConstructor() != null) {
            throw new IllegalArgumentException("Routes cannot be duplicated.");
        }

        duplicateComponent(componentSource, component, true);
    }

    private JavaRewriter.DuplicateInfo duplicateComponent(ComponentTypeAndSourceLocation componentSource,
            ComponentInfo component, boolean attachIt) {
        // Duplicate this component first in a normal way
        JavaRewriter.DuplicateInfo result = component.rewriter().duplicate(component, attachIt);

        for (ComponentTypeAndSourceLocation child : componentSource.children()) {
            ComponentInfo childComponent = components.get(child);

            if (childComponent == null || childComponent.isAnonymousComponent()) {
                continue;
            }

            JavaRewriter.DuplicateInfo childResult = duplicateComponent(child, childComponent, false);

            // Fix attach calls for the children
            for (Map.Entry<String, String> entry : childResult.nameMapping().entrySet()) {
                String oldName = entry.getKey();
                String newName = entry.getValue();

                result.childAddCalls()
                        .forEach(addCall -> component.rewriter().replaceCallParameter(addCall, oldName, newName));
                // Fix the name if the child is attached in the constructor
                if (childComponent.attachCall().getObjectCreationExpression() != null) {
                    // The parent is a variable
                    if (result.variableDeclaration() != null
                            && result.variableDeclaration().getVariables().isNonEmpty()) {
                        Expression vd = result.variableDeclaration().getVariable(0).getInitializer().orElse(null);
                        if (vd != null && vd.isObjectCreationExpr()) {
                            component.rewriter().replaceCallParameter(vd.asObjectCreationExpr(), oldName, newName);
                        }
                    } // The parent is a field
                    else if (result.assignExpr() != null && result.assignExpr().getValue() != null
                            && result.assignExpr().getValue().isObjectCreationExpr()) {
                        ObjectCreationExpr parentObjectCreationExpr = result.assignExpr().getValue()
                                .asObjectCreationExpr();
                        component.rewriter().replaceCallParameter(parentObjectCreationExpr, oldName, newName);
                    }
                }
            }
        }
        return result;
    }

    public interface Callback {
        void accept(ComponentTypeAndSourceLocation source, ComponentInfo component, JavaRewriter rewriter);
    }

    public void forEachComponent(Callback cb) {
        components.forEach((source, component) -> {
            JavaRewriter rewriter = rewriters.get(source.javaFile().orElseThrow(Copilot.sourceNotFound));
            cb.accept(source, component, rewriter);
        });
    }

    public Map<File, String> getResults() {
        return rewriters.keySet().stream()
                .collect(Collectors.toMap(file -> file, file -> rewriters.get(file).getResult()));
    }

    public void writeResult() {
        getResults().forEach((file, result) -> {
            try {
                projectManager.writeFile(file, UNDO_LABEL, result);
            } catch (IOException e) {
                throw new CopilotException(e);
            }
        });
    }
}
